import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { DatePicker } from 'amos-antd';
import moment from 'moment';
import { Form, Button, Select } from 'amos-framework';
import { queryDeptByOrgCodeAction1 } from '../../../services/checkService';
import { exportExcel } from 'UTILS/tools';
import { patrolUrls } from 'CONSTS/urlConsts';


const FormItem = Form.Item;
const Option = Select.Option;

/**
 * 生成报表条件
 *
 * @class ExportSearch
 * @extends {Component}
 */
class ExportSearch extends Component {
  constructor(props) {
    super(props);
    this.state = {
      form: {
        deptId: {
          name: 'deptId',
          value: ''
        },
        beginDate: {
          name: 'beginDate',
          value: moment()
            .startOf('day')
            .format('YYYY-MM-DD HH:mm:ss')
        },
        endDate: {
          name: 'endDate',
          value: moment()
            .endOf('day')
            .format('YYYY-MM-DD HH:mm:ss')
        },
        statement: { name: 'statement', value: 'technology' }
      },
      rules: {
        deptId: [
          { required: true, message: '巡检部门不能为空！' }
        ],
        statement: [
          { required: true, message: '报表模板不能为空！' }
        ],
        beginDate: [
          { required: true, message: '开始时间不能为空！' }
        ],
        endDate: [
          { required: true, message: '结束时间不能为空！' }
        ]
      },
      deptData: [],
      statementList: [
        { name: '技术', value: 'technology' },
        { name: '业务', value: 'business' }
      ]
    };
  }

  componentWillMount = () => {
    this.queryDeptByOrgCode(); // 查询部门
  };

  queryDeptByOrgCode = () => {
    queryDeptByOrgCodeAction1().then(data => {
      this.setState({ deptData: data });
    });
  };

  onExportExcel = () => {
    let { form } = this.state;
    let quest = [];
    if (form) {
      for (let key in form) {
        let obj = form[key];
        quest.push(obj);
      }
    }
    const millisecond = moment().format('YYYYMMDDHHmmss');
    exportExcel({ url: patrolUrls.daySituationExportUrl, method: 'post', body: quest }, '每日巡检情况报表' + millisecond);
  }

  onChange = (key, e) => {
    let { form } = this.state;
    form[key].value = e.target.value;
    this.setState(form);
  };

  onDateChange = (key, date) => {
    const { form } = this.state;
    form[key].value = moment(date).format('YYYY-MM-DD HH:mm:ss');
    this.setState({ form });
  };

  onSelectChange = (key, value) => {
    let { form } = this.state;
    form[key].value = value;
    this.setState(form);
  };

  onDeptChange = (key, value) => {
    const { form } = this.state;
    form[key].value = value;
    this.setState({ form });
  };

  render() {
    const { form, rules, deptData, statementList } = this.state;
    const formItemLayout = {
      labelCol: {
        xs: { span: 24 },
        sm: { span: 4 },
        className: 'colspanlab'
      },
      wrapperCol: {
        xs: { span: 24 },
        sm: { span: 14 },
        className: 'colspan'
      }
    };
    return (
      <Form style={{ padding: '30px 0 0 0' }} className="basic-demo" model={form} rules={rules}>
        <FormItem label="巡检部门" field="deptId" {...formItemLayout} >
          <Select
            className="search_Input"
            data={deptData ? deptData : []}
            style={{ height: 30 }}
            renderOption={item => item && <Option value={item.sequenceNbr}>{item.departmentName}</Option>}
            defaultOption={<Option>全部</Option>}
            value={form.deptId.value}
            onChange={(e) => this.onDeptChange('deptId', e)}
          />
        </FormItem>
        <FormItem label={<span>报表模板</span>} field="statement" {...formItemLayout}>
          <Select
            data={statementList}
            renderOption={item => <Option value={item.value}>{item.name}</Option>}
            value={form.statement.value}
            className="search_Input"
            onChange={e => this.onSelectChange('statement', e)}
          />
        </FormItem>
        <FormItem label={<span>开始日期</span>} field="beginDate" {...formItemLayout}>
          <DatePicker
            style={{ width: 178 }}
            className="search_Input"
            value={moment(form.beginDate.value)}
            allowClear={false}
            format="YYYY-MM-DD HH:mm:ss"
            onChange={e => this.onDateChange('beginDate', e)}
            //允许选择时间
            showTime={{ defaultValue: moment('00:00:00', 'HH:mm:ss') }}
          />
        </FormItem>
        <FormItem label={<span>结束日期</span>} field="endDate" {...formItemLayout}>
          <DatePicker
            style={{ width: 178 }}
            className="search_Input"
            value={moment(form.endDate.value)}
            allowClear={false}
            format="YYYY-MM-DD HH:mm:ss"
            onChange={e => this.onDateChange('endDate', e)}
            showTime={{ defaultValue: moment('00:00:00', 'HH:mm:ss') }}
          />
        </FormItem>
        <FormItem className="search-btn">
          <Button onClick={this.onExportExcel}>生成报表</Button>
          <Button onClick={this.onResetFields}>重置</Button>
        </FormItem>
      </Form>
    );
  }
}

ExportSearch.propTypes = {

};

export default ExportSearch;
