package com.yeejoin.amos.spc.business.controller;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.yeejoin.amos.op.core.common.query.DaoCriteria;
import com.yeejoin.amos.op.core.common.response.CommonResponse;
import com.yeejoin.amos.op.core.util.CommonResponseUtil;
import com.yeejoin.amos.spc.business.param.ReginParams;
import com.yeejoin.amos.spc.business.service.intfc.ISpcCameraService;
import com.yeejoin.amos.spc.business.service.intfc.ISpcEquipmentService;
import com.yeejoin.amos.spc.common.enums.EquipmentTypeEnum;
import com.yeejoin.amos.spc.core.common.request.CommonPageable;
import com.yeejoin.amos.spc.core.common.request.CommonRequest;
import com.yeejoin.amos.spc.core.enums.QueryOperatorEnum;
import com.yeejoin.amos.spc.dao.entity.SpcEquipment;
import com.yeejoin.amos.spc.exception.YeeException;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@RestController
@RequestMapping(value = "/api/spcEuipmentFacilities")
@Api(tags = "设备设施api")
public class SpcEquipmentController extends BaseController {
    private final Logger log = LoggerFactory.getLogger(SpcEquipmentController.class);

    @Autowired
    ISpcCameraService iCameraService;

    @Autowired
    ISpcEquipmentService iEquipmentService;

    @ApiOperation(value = "新增设备设施", notes = "新增设备设施")
    @PostMapping(value = "/euipmentFacilities")
    public CommonResponse createEuipFacility(
            @ApiParam(value = "设备设施", required = true) @RequestBody SpcEquipment euipFacility) {
        ReginParams reginParams = getSelectedOrgInfo();
        euipFacility.setCreatorId(getUserId());
        euipFacility.setOrgCode(getLoginOrgCode(reginParams));
        try {
            iEquipmentService.saveEquipment(euipFacility);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            String message = e.getMessage();
            return CommonResponseUtil.failure("设备设施新增失败", message);
        }
    }

    @ApiOperation(value = "删除设备设施", notes = "删除设备设施")
    @DeleteMapping(value = "/euipmentFacilities/{ids}")
    public CommonResponse deleteEquipment(@ApiParam(value = "设备设施", required = true) @PathVariable String ids) {
		String[] arry = ids.split(",");
		List<Long> idList = new ArrayList<Long>();
		for(String id : arry){
			idList.add(Long.parseLong(id));
		}
        try {
            iEquipmentService.deleteEquipmentById(idList);
            return CommonResponseUtil.success();
        } catch (YeeException e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure(e.getMessage());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("设备设施删除失败");
        }
    }

    @ApiOperation(value = "分页查询设备设施信息", notes = "分页查询设备设施信息")
    @PostMapping(value = "/list", produces = "application/json;charset=UTF-8")
    public CommonResponse queryEquipmentListByPage(
            @ApiParam(value = "查询条件") @RequestBody(required = false) List<CommonRequest> queryRequests,
            @ApiParam(value = "分页参数", required = true, defaultValue = "pageNumber=0&pageSize=10") CommonPageable commonPageable) {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            List<DaoCriteria> criterias = buildDaoCriterias(queryRequests);
            DaoCriteria compDaoCriteria = new DaoCriteria();
            compDaoCriteria.setPropertyName("orgCode");
            compDaoCriteria.setOperator(QueryOperatorEnum.EQUAL.getName());
            compDaoCriteria.setValue(getLoginOrgCode(reginParams));
            criterias.add(compDaoCriteria);
            Page<SpcEquipment> equipmentList = iEquipmentService.queryEquipmentByPage(criterias, commonPageable);
            return CommonResponseUtil.success(equipmentList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询设备设施信息失败");
        }
    }

    @ApiOperation(value = "查询设备设施详情", notes = "查询设备设施详情")
    @GetMapping(value = "/{id}", produces = "application/json;charset=UTF-8")
    public CommonResponse queryEquipmentDetailById(@ApiParam(value = "设备设施id", required = true) @PathVariable Long id) {
        try {
            SpcEquipment result = iEquipmentService.queryEquipmentDetailById(id);
            return CommonResponseUtil.success(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询设备设施详情失败");
        }
    }

    @ApiOperation(value = "设施设备绑定摄像机", notes = "设施设备绑定摄像机")
    @PostMapping(value = "/equipment/{equipId}/bingCameras/{cameraIds}", produces = "application/json;charset=UTF-8")
    public CommonResponse equipmentCameraBind(@ApiParam(value = "设备设施id", required = true) @PathVariable Long equipId,
                                              @ApiParam(value = "摄像机id", required = true) @PathVariable String cameraIds) {
        try {
            String[] ids = cameraIds.split(",");
            iEquipmentService.bindCamera(equipId, ids);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            return CommonResponseUtil.failure("设备设施绑定摄像机失败");
        }
    }

    @ApiOperation(value = "设施设备解绑摄像机", notes = "设施设备解绑摄像机")
    @PostMapping(value = "/equipment/{equipId}/unbindCameras/{cameraIds}", produces = "application/json;charset=UTF-8")
    public CommonResponse equipmentCameraUnBind(@ApiParam(value = "设备设施id", required = true) @PathVariable Long equipId,
                                                @ApiParam(value = "摄像机id", required = true) @PathVariable String cameraIds) {
        try {
            String[] ids = cameraIds.split(",");
            iEquipmentService.unBindCamera(equipId, ids);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            return CommonResponseUtil.failure("设施设备解绑摄像机");
        }
    }

    @ApiOperation(value = "条件查询设备下拉选项", notes = "条件查询设备下拉选项")
    @PostMapping(value = "/equipment/option/listByCondition")
    public CommonResponse listOptionByCondition(@ApiParam(value = "所在分区", required = true) @RequestParam Long regionId,
                                                @ApiParam(value = "部门车间", required = true) @RequestParam Long departmentId,
                                                @ApiParam(value = "所属工段") @RequestParam(required = false) String workshopSection,
                                                @ApiParam(value = "设备名称") @RequestParam(required = false) String name) {
        try {
            return CommonResponseUtil.success(iEquipmentService.listOptionByCondition(regionId, departmentId, workshopSection, name));
        } catch (Exception e) {
            log.error("条件查询设备下拉选项异常", e);
            return CommonResponseUtil.failure("系统繁忙，请稍后再试");
        }
    }
    
    /**
     * 根据设备id获取绑定摄像头
     * @param equipId
     * @return
     */
    @ApiOperation(value = "获取设备绑定的摄像头", notes = "获取设备绑定的摄像头")
    @PostMapping(value = "/bingCameras/{equipId}")

    public CommonResponse getBingCamerasByEquipId(@ApiParam(value = "设备设施id", required = true) @PathVariable Long equipId) {
        try {
            return CommonResponseUtil.success(iEquipmentService.getBingCamerasByEquipId(equipId));
        } catch (Exception e) {
            return CommonResponseUtil.failure("条件查询设备下拉选项");
        }
    }

    /**
     * 获取类型
     *
     * @return
     */
    @ApiOperation(httpMethod = "GET", value = "获取类型", notes = "获取类型")
    @GetMapping(value = "/types", produces = "application/json;charset=UTF-8")
    public CommonResponse getType() {
        return CommonResponseUtil.success(EquipmentTypeEnum.getEnumList());
    }

}
