package com.yeejoin.amos.spc.business.controller;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.*;
import com.yeejoin.amos.op.core.common.query.DaoCriteria;
import com.yeejoin.amos.op.core.common.response.CommonResponse;
import com.yeejoin.amos.op.core.util.CommonResponseUtil;
import com.yeejoin.amos.spc.bo.RiskSourceBo;
import com.yeejoin.amos.spc.business.param.ReginParams;
import com.yeejoin.amos.spc.business.service.intfc.ISpcRiskSourceService;
import com.yeejoin.amos.spc.common.enums.RiskSourceLevelEnum;
import com.yeejoin.amos.spc.common.enums.RiskSourceTypeEnum;
import com.yeejoin.amos.spc.core.common.request.CommonPageable;
import com.yeejoin.amos.spc.core.common.request.CommonRequest;
import com.yeejoin.amos.spc.core.enums.QueryOperatorEnum;
import com.yeejoin.amos.spc.dao.entity.SpcRiskSource;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@RestController
@RequestMapping(value = "/api/spcRiskSource")
@Api(tags = "风险源管理")
public class SpcRiskSourceController extends BaseController {

    private final Logger log = LoggerFactory.getLogger(SpcRiskSourceController.class);


    @Autowired
    private ISpcRiskSourceService riskSourceService;
    

    /**
     * 获取风险源类型
     */
    @ApiOperation(httpMethod = "GET", value = "获取风险源类型", notes = "获取风险源类型")
    @GetMapping(value = "/riskType", produces = "application/json;charset=UTF-8")
    public CommonResponse getRiskType() {
        return CommonResponseUtil.success(RiskSourceTypeEnum.getEnumList());
    }

    /**
     * 获取风险源类型
     */
    @ApiOperation(httpMethod = "GET", value = "获取风险源等级", notes = "获取风险源等级")
    @GetMapping(value = "/riskLevel", produces = "application/json;charset=UTF-8")
    public CommonResponse getRiskLevel() {
        return CommonResponseUtil.success(RiskSourceLevelEnum.getEnumList());
    }

    @ApiOperation(httpMethod = "GET", value = "获取风险点", notes = "获取风险点")
    @GetMapping(value = "/list/{type}", produces = "application/json;charset=UTF-8")
    public CommonResponse getRiskSourceByType(@ApiParam(value = "风险点类型：01-厂区；02-区域；03-位置；04-危险源") @PathVariable(value = "type") String type){
    	return CommonResponseUtil.success(riskSourceService.getRiskSourceByType(type));
    }

    /**
     * 新增风险源
     */
    @ApiOperation(httpMethod = "POST", value = "新增风险源", notes = "新增风险源")
    @PostMapping(value = "/save", produces = "application/json;charset=UTF-8")
    public CommonResponse save(@RequestBody SpcRiskSource riskSource) {
        ReginParams reginParams = getSelectedOrgInfo();
        riskSource.setCreatorId(getUserId());
        riskSource.setOrgCode(getLoginOrgCode(reginParams));
        riskSource = riskSourceService.save(riskSource, getToken());
        return CommonResponseUtil.success(riskSource);
    }


    /**
     * 获取风险点树型结构
     */
    @ApiOperation(httpMethod = "GET", value = "获取风险点树型结构", notes = "获取风险点树型结构")
    @GetMapping(value = "/riskSourceTress", produces = "application/json;charset=UTF-8")
    public CommonResponse getRiskSourceTress() {
        ReginParams reginParams = getSelectedOrgInfo();
        List<RiskSourceBo> riskSourceBos = riskSourceService.listAll(reginParams.getCompany().getOrgCode());
        return CommonResponseUtil.success(getRiskSourceBoTree(riskSourceBos));
    }

    private static List<SpcRiskSource> getRiskSourcesTree(List<SpcRiskSource> list) {
        List<SpcRiskSource> treeList = new ArrayList<>();
        for (SpcRiskSource tree : list) {
            if (tree.getParentId() != null && tree.getParentId() == 0) {
                treeList.add(tree);
            }
            for (SpcRiskSource treeNode : list) {
                if (treeNode.getParentId().longValue() == tree.getId()) {
                    if (tree.getChildren() == null) {
                        tree.setChildren(new ArrayList<>());
                    }
                    tree.getChildren().add(treeNode);
                }
            }
        }
        return treeList;
    }

    private static List<RiskSourceBo> getRiskSourceBoTree(List<RiskSourceBo> list) {
        List<RiskSourceBo> treeList = new ArrayList<>();
        for (RiskSourceBo tree : list) {
            if (tree.getParentId() != null && tree.getParentId() == 0) {
                treeList.add(tree);
            }
            for (RiskSourceBo treeNode : list) {
                if (treeNode.getParentId().longValue() == tree.getId()) {
                    if (tree.getChildren() == null) {
                        tree.setChildren(new ArrayList<>());
                    }
                    tree.getChildren().add(treeNode);
                }
            }
        }
        return treeList;
    }

    /**
     * 删除风险点
     */
    @ApiOperation(httpMethod = "DELETE", value = "删除风险点", notes = "删除风险点")
    @DeleteMapping(value = "/delete/{ids}", produces = "application/json;charset=UTF-8")
    public CommonResponse delete(@PathVariable String ids) {
        String[] idArray = ids.split(",");
        riskSourceService.delete(idArray);
        return CommonResponseUtil.success();
    }


    /**
     * 根据id获取风险点详情
     */
    @ApiOperation(httpMethod = "GET", value = "根据id获取风险点详情", notes = "根据id获取风险点详情")
    @GetMapping(value = "/details/{id}", produces = "application/json;charset=UTF-8")
    public CommonResponse getDetails(@PathVariable Long id) {
//        return CommonResponseUtil.success(riskSourceService.getRiskSourceDetailsById(id));
        return CommonResponseUtil.success(riskSourceService.getRiskSourceBoDetailById(id));
    }

    
    
    /**
     * 查询风险源信息支持分页,条件查询
     *
     * @param queryRequests
     * @param commonPageable
     * @return
     */
    @ApiOperation(value = "查询风险源信息支持分页,条件查询", notes = "查询风险源信息支持分页,条件查询")
    @PostMapping(value = "/list", produces = "application/json;charset=UTF-8")
    public CommonResponse queryEquipmentListByPage(
            @ApiParam(value = "查询条件") @RequestBody(required = false) List<CommonRequest> queryRequests,
            @ApiParam(value = "分页参数", required = true, defaultValue = "pageNumber=0&pageSize=10") CommonPageable commonPageable) {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            List<DaoCriteria> criterias = buildDaoCriterias(queryRequests);
            DaoCriteria compDaoCriteria = new DaoCriteria();
            compDaoCriteria.setPropertyName("orgCode");
            compDaoCriteria.setOperator(QueryOperatorEnum.EQUAL.getName());
            compDaoCriteria.setValue(getLoginOrgCode(reginParams));
            criterias.add(compDaoCriteria);
            Page<SpcRiskSource> spcRiskSourceList = riskSourceService.querySpcRiskSourceByPage(criterias, commonPageable);
            return CommonResponseUtil.success(spcRiskSourceList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询风险源信息");
        }
    }

    @ApiOperation(httpMethod = "GET", value = "根据楼层获取风险点", notes = "根据楼层获取风险点")
    @GetMapping(value = "/queryByFloor", produces = "application/json;charset=UTF-8")
    public CommonResponse queryByFloor(@RequestParam("floor") String floor, @RequestParam("orgCode") String orgCode, @RequestParam(name = "name", required = false) String name) {
        return CommonResponseUtil.success(riskSourceService.getRiskSourceByFloor(floor, orgCode));
    }

    @ApiOperation(httpMethod = "POST", value = "更新风险点三维坐标", notes = "更新风险点三维坐标")
    @PostMapping(value = "/updatePosition3dById", produces = "application/json;charset=UTF-8")
    public CommonResponse updatePosition3dById(@RequestBody SpcRiskSource riskSource) {
        riskSourceService.updatePosition3dById(riskSource);
        return CommonResponseUtil.success();
    }



//    /**
//<<<<<<< .mine
//     * 风险点检查结果推送
//     * @param riskSourceChecks
//     * @return
//     */
//    @PostMapping(value = "/checkResult",produces = "application/json;charset=UTF-8")
//	@ApiOperation(value = "新增巡检记录信息", notes = "新增巡检记录信息")
//	public CommonResponse checkResult(
//			@ApiParam(value = "巡检记录信息", required = true) @RequestBody List<PointCheckInfoBusinessRespone> riskSourceChecks)
//	{
//		if (riskSourceChecks == null)
//		{
//			return CommonResponseUtil.failure("参数为null");
//		}
//		for (PointCheckInfoBusinessRespone patrolCheckDTO : riskSourceChecks)
//		{
//
//
//		}
//		return CommonResponseUtil.success();
//	}
//    /**
//||||||| .r3262
//     * 风险点检查结果推送
//     * @param riskSourceChecks
//     * @return
//     */
//    @PostMapping(value = "/checkResult",produces = "application/json;charset=UTF-8")
//	@ApiOperation(value = "新增巡检记录信息", notes = "新增巡检记录信息")
//	public CommonResponse checkResult(
//			@ApiParam(value = "巡检记录信息", required = true) @RequestBody List<PointCheckInfoBusinessRespone> riskSourceChecks)
//	{
//		if (riskSourceChecks == null)
//		{
//			return CommonResponseUtil.failure("参数为null");
//		}
//		for (PointCheckInfoBusinessRespone patrolCheckDTO : riskSourceChecks)
//		{
//
//
//		}
//		return CommonResponseUtil.success();
//	}
//    /**
//     * 风险清单-风险区域列表
//     */
//    @Authorization(ingore = true)
//    @ApiOperation(value = "风险清单-风险区域列表", notes = "风险清单-风险区域列表")
//    @GetMapping(value = "/region/list", produces = "application/json;charset=UTF-8")
//    public CommonResponse listRegion(@ApiParam(value = "查询条件") @RequestBody(required = false) List<CommonRequest> queryRequests,
//                                     @ApiParam(value = "分页参数", required = true) CommonPageable commonPageable) {
//        CommonPageInfoParam param = CommonPageParamUtil.fillCommonPageInfoParam(queryRequests, commonPageable);
////        Page<HashMap<String, Object>> page = riskSourceService.listRegion(param);
//        return CommonResponseUtil.success();
//    }

}
