package com.yeejoin.amos.spc.business.service.impl;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.yeejoin.amos.feign.privilege.model.AgencyUserModel;
import com.yeejoin.amos.feign.privilege.model.DepartmentModel;
import com.yeejoin.amos.op.core.common.response.CommonResponse;
import com.yeejoin.amos.op.core.util.CommonResponseUtil;
import com.yeejoin.amos.spc.business.dao.mapper.StatisticsMapper;
import com.yeejoin.amos.spc.business.remote.RemoteSecurityService;
import com.yeejoin.amos.spc.business.service.intfc.ISpcStatisticsService;
import com.yeejoin.amos.spc.core.util.DateUtils;

@Service
public class SpcStatisticsServiceImpl implements ISpcStatisticsService {

    private static final String DATETYPE_YYYYMMDD = "yyyyMMdd"; //天

    private static final String DATETYPE_YYYYWW = "yyyyWW";//周

    private static final String DATETYPE_YYYYS = "yyyyS";//季节

    private static final String DATETYPE_YYYYMM = "yyyyMM";//月

    private static final String DATETYPE_YYYY = "yyyy";//年

    private static final SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    @Autowired
    private StatisticsMapper statisticsMapper;

    @Autowired
    private RemoteSecurityService remoteSecurityService;

    @Override
    public CommonResponse riskFactorRiseRate() throws Exception {
        JSONObject json = new JSONObject();
        Date now = new Date();
        json.put("month", getRiskFactorRiseRateByTime(getDateInfo(now, DATETYPE_YYYYMM)));
        json.put("season", getRiskFactorRiseRateByTime(getDateInfo(now, DATETYPE_YYYYS)));
        json.put("year", getRiskFactorRiseRateByTime(getDateInfo(now, DATETYPE_YYYY)));
        return CommonResponseUtil.success(json);
    }

    @Override
    public CommonResponse riskSourceGroupByDepartment() {
        String firstLevelName = "1级";
        String secondLevelName = "2级";
        String thirdLevelName = "3级";
        String fourthLevelName = "4级";
        List<String> legendData = Lists.newArrayList(firstLevelName, secondLevelName, thirdLevelName, fourthLevelName);
        List<Map<String, Object>> list = statisticsMapper.riskSourceBroupByDepartment();
        JSONObject jsonObject = buildBarChartJson(list, legendData, "belongDepartmentId");
        return CommonResponseUtil.success(jsonObject);
    }

    private JSONObject buildBarChartJson(List<Map<String, Object>> list, List<String> legendData, String groupByKey) {
        JSONObject json = new JSONObject();
        Map<Long, DepartmentModel> DepartmentModelMap = getDepartmentModelMap(list);
        Map<String, JSONObject> seryMap = Maps.newLinkedHashMap();
        List<String> groupByNames = Lists.newArrayList();
        list.forEach(map -> {
            String deptId = map.get(groupByKey).toString();
            groupByNames.add(DepartmentModelMap.get(deptId) == null ? "" : DepartmentModelMap.get(deptId).getDepartmentName());
            legendData.forEach(name -> {
                if (seryMap.containsKey(name)) {
                    seryMap.put(name, fillSery(map, seryMap.get(name), name));
                } else {
                    seryMap.put(name, buildSery(map, name));
                }
            });
        });
        JSONArray series = new JSONArray();
        for (Map.Entry<String, JSONObject> entry : seryMap.entrySet()) {
            series.add(entry.getValue());
        }
        json.put("legendData", legendData);
        json.put("xAxisData", groupByNames);
        json.put("series", series);
        return json;
    }

    @Override
    public CommonResponse latentDangerGroupByDepartment() {
        String normalDanger = "一般隐患";
        String seriousDanger = "重大隐患";
        List<String> legendData = Lists.newArrayList(normalDanger, seriousDanger);
        List<Map<String, Object>> list = statisticsMapper.latentDangerBroupByDepartment();
        JSONObject jsonObject = buildBarChartJson(list, legendData, "belongDepartmentId");
        return CommonResponseUtil.success(jsonObject);
    }

    private Map<Long, DepartmentModel> getDepartmentModelMap(List<Map<String, Object>> list) {
        Set<String> deptSetIds = Sets.newHashSet();
        for (Map<String, Object> map : list) {
            deptSetIds.add(map.get("belongDepartmentId").toString());
        }
        List<DepartmentModel> DepartmentModels = remoteSecurityService.listDepartmentByDeptIds(Joiner.on(",").join(deptSetIds));
        return Maps.uniqueIndex(DepartmentModels, DepartmentModel::getSequenceNbr);
    }

    @Override
    public CommonResponse latentDangerRateGroupByDepartment() {
        List<Map<String, Object>> list = statisticsMapper.latentDangerRateGroupByDepartment();
        Set<String> deptSetIds = Sets.newHashSet();
        list.forEach(e -> {
            deptSetIds.add(e.get("belongDepartmentId").toString());
        });
        List<DepartmentModel> DepartmentModels = remoteSecurityService.listDepartmentByDeptIds(Joiner.on(",").join(deptSetIds));
        Map<Long, DepartmentModel> DepartmentModelMap = Maps.uniqueIndex(DepartmentModels, DepartmentModel::getSequenceNbr);
        for (Map<String, Object> map : list) {
            DepartmentModel DepartmentModel = DepartmentModelMap.get(map.get("belongDepartmentId").toString());
            map.put("departmentName", DepartmentModel == null ? "" : DepartmentModel.getDepartmentName());
        }
        return CommonResponseUtil.success(list);
    }

    @Override
    public CommonResponse patrolPlanList(Integer pageIndex, Integer pageSize, String deptId) {
        JSONObject jsonObject = new JSONObject();
        Long count = statisticsMapper.countPatrolPlan(deptId);
        JSONArray jsonArray = new JSONArray();
        if (count.equals(0L)) {
            jsonObject.put("count", 0L);
            jsonObject.put("list", jsonArray);
            return CommonResponseUtil.success(jsonObject);
        }
        List<Map<String, Object>> list = statisticsMapper.listPatrolPlan(pageIndex * pageSize, pageSize, deptId);
        Set<String> userIds = Sets.newHashSet();
        Set<String> deptIds = Sets.newHashSet();
        List<Long> planTaskIds = Lists.newArrayList();
        list.forEach(m -> {
            if (m.get("deptId") != null) {
                deptIds.add(m.get("deptId").toString());
            }
            if (m.get("userId") != null) {
                userIds.add(m.get("userId").toString());
            }
            planTaskIds.add(Long.valueOf(m.get("planTaskId").toString()));
        });
        List<Map<String, Object>> dataList = statisticsMapper.listByTaskNo(planTaskIds);
        Map<Long, List<Map<String, Object>>> dataListMap = handleDataList(dataList);
        Map<Long, DepartmentModel> DepartmentModelMap = null;
        Map<String, AgencyUserModel> AgencyUserModelMap = null;
        if (deptIds.size() > 0) {
            List<DepartmentModel> DepartmentModels = remoteSecurityService.listDepartmentByDeptIds(Joiner.on(",").join(deptIds));
            DepartmentModelMap = Maps.uniqueIndex(DepartmentModels, DepartmentModel::getSequenceNbr);
        }
        if (userIds.size() > 0) {
            List<AgencyUserModel> AgencyUserModels = remoteSecurityService.listUserByUserIds(Joiner.on(",").join(userIds));
            AgencyUserModelMap = Maps.uniqueIndex(AgencyUserModels, AgencyUserModel::getUserId);
        }
        for (Map<String, Object> map : list) {
            JSONObject json = new JSONObject();
            json.put("id", map.get("planTaskId"));
            json.put("planName", map.get("planName"));
            if (AgencyUserModelMap != null) {
                String userId = map.get("userId").toString();
                AgencyUserModel AgencyUserModel = AgencyUserModelMap.get(userId);
                if (AgencyUserModel != null) {
                    json.put("userName", AgencyUserModel.getRealName());
                }
            }
            if (DepartmentModelMap != null) {
                String departmentId = map.get("deptId") == null ? "" : map.get("deptId").toString();
                DepartmentModel DepartmentModel = DepartmentModelMap.get(departmentId);
                if (DepartmentModel != null) {
                    json.put("deptName", DepartmentModel.getDepartmentName());
                }
            }
            json.put("statusCount", buildArr(map));
            json.put("pointData", dataListMap.get(Long.valueOf(map.get("planTaskId").toString())));
            jsonArray.add(json);
        }
        jsonObject.put("count", count);
        jsonObject.put("list", jsonArray);
        return CommonResponseUtil.success(jsonObject);
    }

    private JSONArray buildArr(Map<String, Object> map) {

        JSONArray jsonArray = new JSONArray();
        JSONObject noStartJson = new JSONObject();
        noStartJson.put("id", 1);
        noStartJson.put("countTypeName", "未执行");
        noStartJson.put("status", "0");
        noStartJson.put("countTypeNumber", map.get("noStartNum"));
        JSONObject passJson = new JSONObject();
        passJson.put("id", 2);
        passJson.put("countTypeName", "合格");
        passJson.put("status", "1");
        passJson.put("countTypeNumber", map.get("passNum"));
        JSONObject noPassJson = new JSONObject();
        noPassJson.put("id", 3);
        noPassJson.put("countTypeName", "不合格");
        noPassJson.put("status", "2");
        noPassJson.put("countTypeNumber", map.get("noPassNum"));
        JSONObject overTimeJson = new JSONObject();
        overTimeJson.put("id", 4);
        overTimeJson.put("countTypeName", "漏检");
        overTimeJson.put("status", "3");
        overTimeJson.put("countTypeNumber", map.get("overTimeNum"));
        jsonArray.add(noStartJson);
        jsonArray.add(passJson);
        jsonArray.add(noPassJson);
        jsonArray.add(overTimeJson);
        return jsonArray;
    }

    private Map<Long, List<Map<String, Object>>> handleDataList(List<Map<String, Object>> dataList) {
        Map<Long, List<Map<String, Object>>> map = Maps.newHashMap();
        for (Map<String, Object> dataMap : dataList) {
            Long planTaskId = Long.valueOf(dataMap.get("planTaskId").toString());
            if (map.containsKey(planTaskId)) {
                Map<String, Object> newMap = Maps.newHashMap();
                newMap.put("id", dataMap.get("pointId"));
                newMap.put("name", dataMap.get("pointName"));
                newMap.put("status", dataMap.get("status"));
                map.get(planTaskId).add(newMap);
            } else {
                List<Map<String, Object>> list = Lists.newArrayList();
                Map<String, Object> newMap = Maps.newHashMap();
                newMap.put("id", dataMap.get("pointId"));
                newMap.put("name", dataMap.get("pointName"));
                newMap.put("status", dataMap.get("status"));
                list.add(newMap);
                map.put(planTaskId, list);
            }
        }
        return map;
    }

    private JSONObject buildSery(Map<String, Object> map, String name) {
        JSONObject sery = new JSONObject();
        sery.put("name", name);
        sery.put("type", "bar");
//        sery.put("stack", "总量");
        JSONObject label = new JSONObject();
        JSONObject normal = new JSONObject();
        normal.put("show", "true");
        normal.put("position", "top");
        label.put("normal", normal);
        sery.put("label", label);
        JSONArray data = new JSONArray();
        data.add(map.get(name).toString());
        sery.put("data", data);
        return sery;
    }

    private JSONObject fillSery(Map<String, Object> map, JSONObject sery, String name) {
        sery.getJSONArray("data").add(map.get(name).toString());
        return sery;
    }

    private Map<String, Object> getRiskFactorRiseRateByTime(Date[] dates) {
        String startTime = format.format(dates[0]);
        String endTime = format.format(dates[1]);
        return statisticsMapper.queryForRiseUp(startTime, endTime);
    }

    private Date[] getDateInfo(Date date, String datetype) throws Exception {
        Date[] returnDate = new Date[2];
        Date start = null;
        Date end = null;
        if (DATETYPE_YYYYMMDD.equals(datetype)) {
            start = DateUtils.getCurrentDayStartTime(date);
            end = DateUtils.getCurrentDayEndTime(date);
        }

        if (DATETYPE_YYYYWW.equals(datetype)) {
            start = DateUtils.getCurrentWeekDayStartTime(date);
            end = DateUtils.getCurrentWeekDayEndTime(date);
        }

        if (DATETYPE_YYYYS.equals(datetype)) {
            start = DateUtils.getCurrentQuarterStartTime(date);
            end = DateUtils.getCurrentQuarterEndTime(date);
        }

        if (DATETYPE_YYYYMM.equals(datetype)) {
            start = DateUtils.getCurrentMonthStartTime(date);
            end = DateUtils.getCurrentMonthEndTime(date);
        }

        if (DATETYPE_YYYY.equals(datetype)) {
            start = DateUtils.getCurrentYearStartTime(date);
            end = DateUtils.getCurrentYearEndTime(date);
        }
        returnDate[0] = start;
        returnDate[1] = end;
        return returnDate;
    }
}
