package com.yeejoin.amos.spc.business.service.impl;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.assertj.core.util.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import com.yeejoin.amos.feign.privilege.model.DepartmentModel;
import com.yeejoin.amos.op.core.common.query.DaoCriteria;
import com.yeejoin.amos.spc.business.dao.mapper.EquipmentMapper;
import com.yeejoin.amos.spc.business.dao.repository.ISpcCameraDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcEquipmentCameraBindDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcEquipmentDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcRiskFactorDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcRiskSourceDao;
import com.yeejoin.amos.spc.business.remote.RemoteSecurityService;
import com.yeejoin.amos.spc.business.service.intfc.ISpcEquipmentService;
import com.yeejoin.amos.spc.common.enums.CamerasEnum;
import com.yeejoin.amos.spc.common.enums.EquipmentTypeEnum;
import com.yeejoin.amos.spc.core.common.request.CommonPageable;
import com.yeejoin.amos.spc.core.util.StringUtil;
import com.yeejoin.amos.spc.core.util.query.BaseQuerySpecification;
import com.yeejoin.amos.spc.dao.entity.SpcCameras;
import com.yeejoin.amos.spc.dao.entity.SpcEquipment;
import com.yeejoin.amos.spc.dao.entity.SpcEquipmentCameraBind;
import com.yeejoin.amos.spc.dao.entity.SpcRiskSource;
import com.yeejoin.amos.spc.exception.YeeException;

@Service("iEquipmentService")
public class SpcEquipmentServiceImpl implements ISpcEquipmentService {

    @Autowired
    private ISpcEquipmentDao iSpcEquipmentDao;

    @Autowired
    private EquipmentMapper equipmentMapper;

    @Autowired
    private ISpcEquipmentCameraBindDao iSpcEquipmentCameraBindDao;

    @Autowired
    ISpcCameraDao iSpcCameraDao;

    @Autowired
    ISpcRiskFactorDao iSpcRiskFactorDao;

    @Autowired
    ISpcRiskSourceDao iSpcRiskSourceDao;

    @Autowired
    private RemoteSecurityService remoteSecurityService;

    @Override
    public SpcEquipment saveEquipment(SpcEquipment equipment) {
        List<SpcEquipment> eqList = iSpcEquipmentDao.findByCodeAndOrgCode(equipment.getCode(), equipment.getOrgCode());
        boolean isExistCode = false;
        if (!eqList.isEmpty() && eqList.size() > 1) {
            isExistCode = true;
        } else if (!eqList.isEmpty() && eqList.size() == 1) {
            if (StringUtil.isNotEmpty(equipment.getId())) {
                if (eqList.get(0).getId() != equipment.getId()) {
                    isExistCode = true;
                }
            } else {
                isExistCode = true;
            }
        }
        if (isExistCode) {
            throw new YeeException("设备位号【" + equipment.getCode() + "】已存在，请重新填写！");
        }
        if (StringUtils.isEmpty(equipment.getRegionId())) {
            throw new YeeException("请选择风险区域");
        }
        SpcRiskSource riskSource = iSpcRiskSourceDao.findById(Long.valueOf(equipment.getRegionId())).get();
        if (riskSource == null) {
            throw new YeeException("请选择风险区域");
        }
        iSpcEquipmentDao.saveAndFlush(equipment);
        equipment.setEquipmentNumber(buildEquipmentNumber(riskSource.getCode(),equipment.getId()));
        iSpcEquipmentDao.save(equipment);
        return equipment;
    }

    private String buildEquipmentNumber(String sourceCode,Long id) {
        String code = String.format("%5d",id).replace(" ","0");
        return sourceCode+"EQ"+code;
//        return code + "_" + RandomUtil.buildRandom(6) + "_" + System.currentTimeMillis();
    }

    @Override
    public List<Long> deleteEquipmentById(List<Long> ids) {
        // TODO 未进行校验：绑定风险因素的设备不能删除
        int count = iSpcRiskFactorDao.countByControlObjectIdsAndType("1", ids);
        if (count > 0) {
            throw new YeeException("该设备设施正在使用中，删除失败！");
        }
        iSpcEquipmentDao.deleteBatchByIds(ids);
        return ids;
    }

    @Override
    public Page<SpcEquipment> queryEquipmentByPage(List<DaoCriteria> criterias, CommonPageable commonPageable) {
        BaseQuerySpecification<SpcEquipment> specification = new BaseQuerySpecification<>(criterias);
        Page<SpcEquipment> page = iSpcEquipmentDao.findAll(specification, commonPageable);
        List<SpcEquipment> list = page.getContent();
        if (!CollectionUtils.isEmpty(list)) {
            Set<String> deptIdSets = Sets.newHashSet();
            list.forEach(e -> {
                deptIdSets.add(e.getDepartmentId());
            });
            deptIdSets.remove(null);
            deptIdSets.remove("");
            List<DepartmentModel> DepartmentModels = remoteSecurityService
                    .listDepartmentByDeptIds(Joiner.on(",").join(deptIdSets));
            Map<Long, DepartmentModel> DepartmentModelMap = Maps.uniqueIndex(DepartmentModels, DepartmentModel::getSequenceNbr);
            list.forEach(e -> {
                SpcRiskSource riskSource = iSpcRiskSourceDao.findById(Long.parseLong(e.getRegionId())).get();
                if (riskSource != null) {
                    e.setRegionDepartmentName(riskSource.getName());
                }
                DepartmentModel department = DepartmentModelMap.get(e.getDepartmentId());
                if (department != null) {
                    e.setDepartmentName(department.getDepartmentName());
                }
                e.setCameraCount(iSpcEquipmentCameraBindDao.countByEquipmentId(e.getId()));
                e.setType(EquipmentTypeEnum.getEnumName(e.getType()));
            });
        }
        return page;
    }

    @Override
    public SpcEquipment queryEquipmentDetailById(Long id) {
        return iSpcEquipmentDao.findById(id).get();
    }

    @Override
    public void bindCamera(Long equipId, String[] ids) {
        for (String id : ids) {
            int count = iSpcEquipmentCameraBindDao.countByEquipmentIdAndCameraId(equipId, Long.parseLong(id));
            if (count > 0) {
                continue;
            }
            SpcEquipmentCameraBind bindObj = new SpcEquipmentCameraBind();
            bindObj.setCameraId(Long.parseLong(id));
            bindObj.setEquipmentId(equipId);
            iSpcEquipmentCameraBindDao.save(bindObj);
        }

    }

    @Override
    public void unBindCamera(Long equipId, String[] ids) {
        List<Long> cameraIds = Arrays.stream(ids).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
        iSpcEquipmentCameraBindDao.deleteByEquipmentIdAndCameraIds(equipId, cameraIds);
    }

    @Override
    public List<Map<String, Object>> listOptionByCondition(Long regionId, Long departmentId, String workshopSection,
                                                           String name) {
        Map<String, Object> map = Maps.newHashMap();
        map.put("regionId", regionId);
        map.put("departmentId", departmentId);
        map.put("workshopSection", workshopSection);
        map.put("name", name);
        return equipmentMapper.listOptionByMap(map);
    }

    @Override
    public List<SpcCameras> getBingCamerasByEquipId(Long equipId) {
        // TODO Auto-generated method stub
        List<SpcCameras> list = iSpcCameraDao.getBingCamerasByEquipId(equipId);
        for (SpcCameras camera : list) {
            camera.setType(CamerasEnum.getEnumName(camera.getType()));
        }
        return list;
    }

}
