package com.yeejoin.amos.bank.service.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.stream.Collectors;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.yeejoin.amos.bank.TopographyEvent;
import com.yeejoin.amos.bank.TopographyLine;
import com.yeejoin.amos.bank.TopographyNode;
import com.yeejoin.amos.bank.TopographyNodeDetail;
import com.yeejoin.amos.bank.TopographyTree;
import com.yeejoin.amos.bank.common.enums.AbnormalStateEnum;
import com.yeejoin.amos.bank.dao.repository.ITopographyEventRepository;
import com.yeejoin.amos.bank.dao.repository.ITopographyLineRepository;
import com.yeejoin.amos.bank.dao.repository.ITopographyNodeDetailRepository;
import com.yeejoin.amos.bank.dao.repository.ITopographyNodeRepository;
import com.yeejoin.amos.bank.dao.repository.ITopographyTreeRepository;
import com.yeejoin.amos.bank.dto.TopographyEventDTO;
import com.yeejoin.amos.bank.dto.TopographyLineDTO;
import com.yeejoin.amos.bank.dto.TopographyNodeDTO;
import com.yeejoin.amos.bank.dto.TopographyTreeDTO;
import com.yeejoin.amos.bank.param.AbnormalParam;
import com.yeejoin.amos.bank.param.TopoNodeDetailParam;
import com.yeejoin.amos.bank.service.ITopographyService;
import com.yeejoin.amos.feign.systemctl.model.DictionarieValueModel;
import com.yeejoin.amos.op.core.jpa.GenericManagerImpl;
import com.yeejoin.amos.spc.business.remote.RemoteSecurityService;
import com.yeejoin.amos.spc.business.remote.RemoteWebSocketServer;

import fr.opensagres.xdocreport.template.velocity.internal.Foreach;

/**
 * 拓扑图service
 *
 * @author zhangzhihui
 */
@Service
public class TopographyServiceImpl extends GenericManagerImpl<TopographyTree, String> implements ITopographyService {
	
	//平台字典管理中配置value为key值
	private static final String BNAK_ORG_KEY = "BANK_ORG";
	
	//平台字典管理中配置value为key值
	private static final String BNAK_XIAN_KEY = "xian";
	
	//平台字典管理中配置value为key值
	private static final String BNAK_KEJICHU_KEY = "科技处";

    @Autowired
    private ITopographyTreeRepository iTopographyTreeDao;

    @Autowired
    private ITopographyNodeRepository iTopographyNodeDao;

    @Autowired
    private ITopographyLineRepository iTopographyLineDao;

    @Autowired
    private ITopographyEventRepository iTopographyEventDao;

    @Autowired
    private ITopographyNodeDetailRepository iTopographyNodeDetailDao;

    @Autowired
    private RemoteWebSocketServer webSocketServer;
    
    @Autowired
    RemoteSecurityService remoteSecurityService;

    public TopographyServiceImpl(ITopographyTreeRepository repository) {
        super(repository);
        this.iTopographyTreeDao = repository;
    }

    @Override
    public List<TopographyTreeDTO> getTree(String orgcode,int type) {
    	
        List<TopographyTreeDTO> list = iTopographyTreeDao.findByTypeAndOrgCode(type,orgcode).stream().map(t -> convert(t)).collect(Collectors.toList());
        boolean judgeDeviceAuth = JudgeDeviceAuth(list,orgcode);
        if(false == judgeDeviceAuth) {
        	return new ArrayList<TopographyTreeDTO>();
        }
        Map<String, TopographyTreeDTO> map = list.stream().collect(Collectors.toMap(n -> n.getId(), n -> n));
        list.forEach(n -> {
            if (n.getParent() != null) {
                map.get(n.getParent()).getChildren().add(n);
            }
        });
        return list.stream().filter(n -> n.getParent() == null).collect(Collectors.toList());
    }

    private TopographyTreeDTO convert(TopographyTree node) {
        TopographyTreeDTO topographyTreeDTO = new TopographyTreeDTO();
        topographyTreeDTO.setId(node.getCode());
        topographyTreeDTO.setName(node.getName());
        topographyTreeDTO.setParent(node.getParentCode());
        topographyTreeDTO.setTreeCode(node.getTreeCode());
        topographyTreeDTO.setGroup(node.getGroup());
        topographyTreeDTO.setOrgCode(node.getOrgCode());
        return topographyTreeDTO;
    }

    @Override
    public List<TopographyNodeDTO> getNodes(String orgcode,String treeid, int type) {
        if (treeid == null) {
            return iTopographyNodeDao.findByOrgCodeAndType(orgcode,type).stream().map(t -> convert(t)).collect(Collectors.toList());
        }
        return iTopographyNodeDao.findByOrgCodeAndTreeidAndType(orgcode,treeid, type).stream().map(t -> convert(t)).collect(Collectors.toList());
    }

    private TopographyNodeDTO convert(TopographyNode node) {
        TopographyNodeDTO topographyNodeDTO = new TopographyNodeDTO();
        BeanUtils.copyProperties(node, topographyNodeDTO);
        topographyNodeDTO.setWarnState(node.getState());
        topographyNodeDTO.setPorts(node.getPorts());
        if (node.getState() != 0) {
            topographyNodeDTO.setWarnMessage(node.getNumber() + AbnormalStateEnum.getText(node.getState()));
        }
        return topographyNodeDTO;
    }

    @Override
    public List<TopographyLineDTO> getLinks(String orgcode,String treeid, int type) {
        if (treeid == null) {
            return iTopographyLineDao.findByOrgCodeAndType(orgcode,type).stream().map(t -> convert(t)).collect(Collectors.toList());
        }
        return iTopographyLineDao.findByOrgCodeAndTreeidAndType(orgcode,treeid, type).stream().map(t -> convert(t)).collect(Collectors.toList());
    }

    private TopographyLineDTO convert(TopographyLine node) {
        TopographyLineDTO topographyLineDTO = new TopographyLineDTO();
        BeanUtils.copyProperties(node, topographyLineDTO);
        topographyLineDTO.setWarnState(node.getState());
        if (node.getState() != 0) {
            topographyLineDTO.setWarnMessage(node.getNumber() + AbnormalStateEnum.getText(node.getState()));
        }
        return topographyLineDTO;
    }

    @Override
    public void saveNodes(String orgcode,List<TopographyNodeDTO> nodeData) {
        List<TopographyNode> list = iTopographyNodeDao.findAllById(nodeData.stream().map(n -> n.getId()).collect(Collectors.toList()));
        for (TopographyNode topographyNode : list) {
            TopographyNodeDTO node = nodeData.stream().filter(n -> topographyNode.getId().equals(n.getId())).findFirst().get();
            topographyNode.setLoc(node.getLoc());
            topographyNode.setSize(node.getSize());
            topographyNode.setOrgCode(orgcode);
        }
        iTopographyNodeDao.saveAll(list);
    }
    
    @Override
    public String getSelfOrgCode() {
    	List<DictionarieValueModel> listDictionaryByDictCode = remoteSecurityService.listDictionaryByDictCode(BNAK_ORG_KEY);
    	StringBuffer sb = new StringBuffer();
    	listDictionaryByDictCode.forEach(e ->{
    		if(BNAK_XIAN_KEY.equals(e.getDictDataKey())||BNAK_KEJICHU_KEY.equals(e.getDictDataKey())) {
    			sb.append('#').append(e.getDictDataValue());
    		}
    	});
    	return sb.toString();
    	
    }

    @Override
    public List<TopographyEventDTO> getEvents() {
        return iTopographyEventDao.findAll().stream().map(t -> convert(t)).collect(Collectors.toList());
    }

    private TopographyEventDTO convert(TopographyEvent t) {
        TopographyEventDTO topographyEventDTO = new TopographyEventDTO();
        BeanUtils.copyProperties(t, topographyEventDTO);
        return topographyEventDTO;
    }

    /**
     * @param sourceId
     * @param state    true 确认告警，false 清除告警
     */
    @Override
    public void sendAbnormal(String sourceId, int state) {
        TopographyNode node = iTopographyNodeDao.findBySourceId(sourceId);
        if (node != null) {
            AbnormalParam param = new AbnormalParam();
            param.setNodeid(node.getId());
            if (state > 0) {
                param.setNumber(1);
            } else {
                param.setNumber(0);
            }
            param.setState(state);
            try {
                testAbnormal(param);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void testAbnormal(AbnormalParam param) throws Exception {
        TopographyNode node = iTopographyNodeDao.findById(param.getNodeid() != null ? param.getNodeid() : "").orElse(null);
        TopographyLine link = iTopographyLineDao.findById(param.getLinkid() != null ? param.getLinkid() : "").orElse(null);
        String treeid = null;
        int type = 0;
        if (node != null) {
            node.setState(param.getState());
            node.setNumber(param.getNumber());
            treeid = node.getTreeid();
            iTopographyNodeDao.save(node);
            type = node.getType();
        }
        if (link != null) {
            link.setState(param.getState());
            link.setNumber(param.getNumber());
            treeid = link.getTreeid();
            iTopographyLineDao.save(link);
            type = link.getType();
        }
        //添加点详情
        String nodeDetailItem = param.getNodeDetailItem();
        if (nodeDetailItem != null && !"".equals(nodeDetailItem)) {
            int state = param.getState();
            String nodeid = param.getNodeid() != null ? param.getNodeid() : "";
            TopographyNodeDetail nodeDetail = iTopographyNodeDetailDao.findByNodeid(param.getNodeid() != null ? param.getNodeid() : "");
            if (nodeDetail == null) {
                nodeDetail = new TopographyNodeDetail();
                nodeDetail.setNodeid(nodeid);
                nodeDetail = iTopographyNodeDetailDao.save(nodeDetail);
            }
            String detail = nodeDetail.getNodeDetail();
            List<TopoNodeDetailParam> detailList = JSONArray.parseArray(detail, TopoNodeDetailParam.class);
            if (detailList == null) {
                detailList = new ArrayList<>();
            }
            TopoNodeDetailParam newExcept = new TopoNodeDetailParam(nodeDetailItem, "异常");
            if (state > 0 && !detailList.contains(newExcept)) { //异常，添加点详情信息
                detailList.add(newExcept);
                nodeDetail.setNodeDetail(detailList.toString());
            } else {
                List<TopoNodeDetailParam> newList = detailList.stream().filter(n -> !nodeDetailItem.equals(n.getName())).collect(Collectors.toList());
                nodeDetail.setNodeDetail(newList.toString());
            }
            iTopographyNodeDetailDao.save(nodeDetail);
        }
        if (treeid != null && type == 1) {
            webSocketServer.sendMessage(treeid, "topography");
        }
        if (treeid != null && type == 2) {
            webSocketServer.sendMessage(treeid, "pams");
        }
        if (treeid != null && type == 3) {
            webSocketServer.sendMessage(treeid, "donghuan");
        }
        if (type == 4) {
            webSocketServer.sendMessage("refresh", "shebei");
        }
    }

    @Override
    public void sendAbnormalByType(String sourceId, int state, int type, String nodeDetailItem) {
        TopographyNode node = iTopographyNodeDao.findBySourceIdAndType(sourceId, type);
        if (node != null) {
            AbnormalParam param = new AbnormalParam();
            param.setNodeid(node.getId());
            Integer number = node.getNumber();
            if (state > 0) {
                number = number != null ? number + 1 : 1;
                param.setNumber(number);

            } else {
                number -= 1;
                param.setNumber(number < 0 ? 0 : number);
                if (number > 0) {
                    state = 2;
                }
            }
            param.setState(state);
            if (nodeDetailItem != null) {
                param.setNodeDetailItem(nodeDetailItem);
            }
            try {
                testAbnormal(param);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void delWebTree(String treeid) {
        List<TopographyTree> trees = iTopographyTreeDao.findByType(1).stream().filter(tree -> Arrays.asList(tree.getTreeCode().split("\\*")).contains(treeid)).collect(Collectors.toList());
        iTopographyTreeDao.deleteAll(trees);
    }

    @Override
    public void addWebTree(TopographyTreeDTO tree) {
        TopographyTree t = new TopographyTree();
        t.setId(UUID.randomUUID().toString().replaceAll("-", ""));
        t.setCode(Math.abs(new Random().nextInt()) + "");
        t.setParentCode(tree.getParent());
        t.setName(tree.getName());
        t.setOrgCode(tree.getOrgCode());
        if (tree.getParent() != null) {
            TopographyTree parent = iTopographyTreeDao.findByCodeAndType(tree.getParent(), 1);
            t.setTreeCode(parent.getTreeCode() + "*" + t.getCode());
        } else {
            t.setTreeCode(t.getCode());
        }
        t.setType(1);
        iTopographyTreeDao.save(t);
    }

    @Override
    public Object getNodeDetail(String id) {
        TopographyNode node = iTopographyNodeDao.findById(id).get();
        TopographyNodeDetail detail = iTopographyNodeDetailDao.findByNodeid(id);
        if (detail != null) {
            List<Map> list = JSON.parseArray(detail.getNodeDetail(), Map.class);
            for (Map map : list) {
                if (map.get("name").equals("监测状态") || map.get("name").equals("健康状态")) {
                    if (node.getState() == 0) {
                        map.put("value", "正常");
                    }
                }
                if (map.get("name").equals("室内温度")) {
                    if (node.getState() == 0) {
                        map.put("value", "22.8°C");
                    }
                }
            }
            return list;
        } else {
            return null;
        }
    }

    @Override
    public List<Map> getDeviceNodes(String orgcode) {
        List<TopographyNode> nodes = iTopographyNodeDao.findByOrgCodeAndType(orgcode, 4);
        List<Map> results = new ArrayList<>();
        for (TopographyNode node : nodes) {
            TopographyNodeDTO nodeDTO = convert(node);
            Map map = JSON.parseObject(JSON.toJSONString(nodeDTO), Map.class);
            try {
                Map ports = JSON.parseObject(node.getPorts(), Map.class);
                map.putAll(ports);
            } catch (Exception e) {
            }
            results.add(map);
        }
        return results;
    }
    

    @Override
    public List<TopographyLineDTO> getDeviceLinks(String orgcode) {
        return getLinks(orgcode,null, 4);
    }

    @Override
    public void sendPMTSAbnormal(String fromSourceId, String toSourceId, String groupName, Long treeId, int state) {
        String treeIdStr = String.valueOf(treeId);

        TopographyNode group = iTopographyNodeDao.findByTextAndTreeidAndIsGroup(groupName, treeIdStr, true);
        String groupKey = group.getKey();

        TopographyNode fromNode = iTopographyNodeDao.findBySourceIdAndTreeidAndGroup(fromSourceId, treeIdStr, groupKey);
        TopographyNode toNode = iTopographyNodeDao.findBySourceIdAndTreeidAndGroup(toSourceId, treeIdStr, groupKey);
        if (fromNode != null && toNode != null) {
            TopographyLine line = iTopographyLineDao.findByFromAndToAndTreeid(fromNode.getKey(), toNode.getKey(), treeIdStr);
            if (line != null) {
                line.setState(state);
                iTopographyLineDao.save(line);

                try {
                    webSocketServer.sendMessage(line.getTreeid(), "pams");
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

    }
    
	/**
	 * 判断该用户登选择公司有无查看该拓扑的权限
	 * @param list
	 * @param userOrgCode
	 * @return
	 */
	private boolean JudgeDeviceAuth(List<TopographyTreeDTO> list, String orgCode) {
		
		String userOrgCode = null;
		if(orgCode.startsWith("%#")) {
		  userOrgCode = truncateHeadString(orgCode,2);	
		}else {
		  userOrgCode = orgCode;	
		}
		
			
		for (TopographyTreeDTO topographyTreeDTO : list) {

			if (!StringUtils.isEmpty(topographyTreeDTO.getOrgCode())) {
				if (topographyTreeDTO.getOrgCode().contains("#")) {
					String[] array = topographyTreeDTO.getOrgCode().split("#");
					for (int i = 0; i < array.length; i++) {
						String str = array[i];
						if(str.contains("-")) {
							String[] split = str.split("-");
							for (int j = 0; j < split.length; j++) {
								if (userOrgCode.equals(split[j])) {
									return true;
								}
							}
						} else {
							if(userOrgCode.equals(str)) {
								return true;
							}
						}
					}
					

				} else {
					String[] arr = topographyTreeDTO.getOrgCode().split("-");
					for (int i = 0; i < arr.length; i++) {
						if (userOrgCode.equals(arr[i])) {
							return true;
						}
					}

				}
			}
		}
		
		return false;

	}
	
	/**
	 * 去除字符串前n个字符
	 * @param origin 要操作的字符串
	 * @param count 去掉字符串的数量
	 * @return
	 */
	public static String truncateHeadString(String origin, int count) {
	    if (origin == null || origin.length() < count) {
	        return null;
	    }
	    char[] arr = origin.toCharArray();
	    char[] ret = new char[arr.length - count];
	    for (int i = 0; i < ret.length; i++) {
	        ret[i] = arr[i + count];
	    }
	    return String.copyValueOf(ret);
	}
	
}
