package com.yeejoin.amos.spc.business.service.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.assertj.core.util.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import com.yeejoin.amos.feign.privilege.model.AgencyUserModel;
import com.yeejoin.amos.feign.privilege.model.DepartmentModel;
import com.yeejoin.amos.op.core.common.query.DaoCriteria;
import com.yeejoin.amos.op.core.common.response.CommonResponse;
import com.yeejoin.amos.op.core.util.CommonResponseUtil;
import com.yeejoin.amos.spc.business.constants.FasConstant;
import com.yeejoin.amos.spc.business.dao.repository.ISpcEquipmentDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcRiskFactorDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcRiskSourceDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcTaskworkContentDao;
import com.yeejoin.amos.spc.business.dao.repository.ISpcTaskworkDao;
import com.yeejoin.amos.spc.business.remote.RemoteSecurityService;
import com.yeejoin.amos.spc.business.service.intfc.ISpcTaskworkService;
import com.yeejoin.amos.spc.common.enums.TaskworkStateEnum;
import com.yeejoin.amos.spc.core.common.request.CommonPageable;
import com.yeejoin.amos.spc.core.util.StringUtil;
import com.yeejoin.amos.spc.core.util.query.BaseQuerySpecification;
import com.yeejoin.amos.spc.dao.entity.SpcEquipment;
import com.yeejoin.amos.spc.dao.entity.SpcRiskFactor;
import com.yeejoin.amos.spc.dao.entity.SpcRiskSource;
import com.yeejoin.amos.spc.dao.entity.SpcTaskwork;
import com.yeejoin.amos.spc.dao.entity.SpcTaskworkContent;
import com.yeejoin.amos.spc.exception.YeeException;

@Service
public class SpcTaskworkServiceImpl implements ISpcTaskworkService {

    @Autowired
    private ISpcTaskworkDao iSpcTaskworkDao;

    @Autowired
    private ISpcTaskworkContentDao iSpcTaskworkContentDao;

    @Autowired
    private ISpcRiskFactorDao iSpcRiskFacotrDao;

    @Autowired
    private RemoteSecurityService remoteSecurityService;

    @Autowired
    private ISpcRiskSourceDao iSpcRiskSourceDao;

    @Autowired
    private ISpcEquipmentDao iSpcEquipmentDao;

    @Transactional
    @Override
    public CommonResponse save(SpcTaskwork spcTaskwork) {
        List<SpcTaskwork> eqList = iSpcTaskworkDao.findByNameAndOrgCode(spcTaskwork.getName(),
                spcTaskwork.getOrgCode());
        boolean isExistCode = false;
        if (!eqList.isEmpty() && eqList.size() > 1) {
            isExistCode = true;
        } else if (!eqList.isEmpty() && eqList.size() == 1) {
            if (StringUtil.isNotEmpty(spcTaskwork.getId())) {
                if (eqList.get(0).getId() != spcTaskwork.getId()) {
                    isExistCode = true;
                }
            } else {
                isExistCode = true;
            }
        }
        if (isExistCode) {
            throw new YeeException("作业活动【" + spcTaskwork.getName() + "】已存在，请重新填写！");
        }
        SpcTaskwork newSpcTaskwork = iSpcTaskworkDao.saveAndFlush(spcTaskwork);
        List<SpcTaskworkContent> contentList = spcTaskwork.getContentList();
        if (null != contentList) {
            contentList.forEach(e -> {
                e.setTaskworkId(newSpcTaskwork.getId());
            });
            iSpcTaskworkContentDao.saveAll(contentList);
        }
        return CommonResponseUtil.success();
    }

    @Transactional
    @Override
    public CommonResponse update(SpcTaskwork spcTaskwork) {
        iSpcTaskworkDao.save(spcTaskwork);
        iSpcTaskworkContentDao.deleteByTaskworkId(spcTaskwork.getId());
        List<SpcTaskworkContent> contentList = spcTaskwork.getContentList();
        if (null != contentList) {
            contentList.forEach(e -> {
                e.setTaskworkId(spcTaskwork.getId());
            });
            iSpcTaskworkContentDao.saveAll(contentList);
        }
        return CommonResponseUtil.success();
    }

    @Override
    public CommonResponse list(List<DaoCriteria> daoCriteriaList, CommonPageable commonPageable) {
        BaseQuerySpecification<SpcTaskwork> specification = new BaseQuerySpecification<>(daoCriteriaList);
        Page<SpcTaskwork> page = iSpcTaskworkDao.findAll(specification, commonPageable);
        List<SpcTaskwork> taskworkList = page.getContent();
        if (!CollectionUtils.isEmpty(taskworkList)) {
            Set<String> userIdSet = Sets.newHashSet();
            Set<String> departmentIdSet = Sets.newHashSet();
            taskworkList.forEach(taskwork -> {
                userIdSet.add(taskwork.getApplyUserId());
                userIdSet.add(taskwork.getCreatorId());
                departmentIdSet.add(taskwork.getApplyDepartmentId());
                departmentIdSet.add(taskwork.getDepartmentId());
                departmentIdSet.addAll(Arrays.asList(taskwork.getGroupId().split(",")));
            });
            userIdSet.remove(null);
            userIdSet.remove("");
            departmentIdSet.remove(null);
            departmentIdSet.remove("");
            List<DepartmentModel> DepartmentModels = remoteSecurityService
                    .listDepartmentByDeptIds(Joiner.on(",").join(departmentIdSet));
            List<AgencyUserModel> AgencyUserModels = remoteSecurityService.listUserByUserIds(Joiner.on(",").join(userIdSet));
            Map<String, AgencyUserModel> AgencyUserModelMap = Maps.uniqueIndex(AgencyUserModels, AgencyUserModel::getUserId);
            Map<Long, DepartmentModel> DepartmentModelMap = Maps.uniqueIndex(DepartmentModels, DepartmentModel::getSequenceNbr);
            taskworkList.forEach(taskwork -> {
                AgencyUserModel applyAgencyUserModel = AgencyUserModelMap.get(taskwork.getApplyUserId());
                if (applyAgencyUserModel != null) {
                    taskwork.setApplyUserName(applyAgencyUserModel.getRealName());
                }
                AgencyUserModel createAgencyUserModel = AgencyUserModelMap.get(taskwork.getCreatorId());
                if (createAgencyUserModel != null) {
                    taskwork.setCreateUserName(createAgencyUserModel.getRealName());
                }
                SpcRiskSource spcRiskSource = iSpcRiskSourceDao.findById(Long.parseLong(taskwork.getRegionId())).get();
                if (spcRiskSource != null) {
                    taskwork.setRegionName(spcRiskSource.getName());
                }
                DepartmentModel department = DepartmentModelMap.get(taskwork.getDepartmentId());
                if (department != null) {
                    taskwork.setDepartmentName(department.getDepartmentName());
                }
                DepartmentModel applyDepartment = DepartmentModelMap.get(taskwork.getApplyDepartmentId());
                if (applyDepartment != null) {
                    taskwork.setApplyDepartmentName(applyDepartment.getDepartmentName());
                }
                List<String> groupNameList = new ArrayList<>();
                for (String groupId : taskwork.getGroupId().split(",")) {
                    DepartmentModel groupDepartment = DepartmentModelMap.get(groupId);
                    if (groupDepartment != null) {
                        groupNameList.add(groupDepartment.getDepartmentName());
                    }
                }
                taskwork.setGroupName(StringUtils.join(groupNameList, ","));

                List<SpcTaskworkContent> contentList = iSpcTaskworkContentDao.findAllByTaskworkId(taskwork.getId());
                taskwork.setContentList(contentList);
            });
        }
        return CommonResponseUtil.success(page);
    }

    @Transactional
    @Override
    public List<SpcTaskwork> delete(List<Long> taskworkIds) {
        List<SpcTaskwork> delTaskworkList = iSpcTaskworkDao.findByIds(taskworkIds);
        if (delTaskworkList.size() == 0) {
            return null;
        }
        List<SpcTaskwork> reject1 = new ArrayList<>();
        List<SpcTaskwork> reject2 = new ArrayList<>();
        // 验证每个作业活动能否删除
        delTaskworkList.forEach(delTaskwork -> {
            // 验证该作业活动下的步骤有没有关联在风险因素
            List<SpcTaskworkContent> taskworkContents = iSpcTaskworkContentDao.findAllByTaskworkId(delTaskwork.getId());
            List<Long> contentIds = new ArrayList<Long>();
            taskworkContents.forEach(content -> {
                contentIds.add(content.getId());
            });
            int bindedRiskFactorCount = 0;
            if (contentIds.size() > 0) {
                bindedRiskFactorCount = iSpcRiskFacotrDao
                        .countByControlObjectIdsAndType(FasConstant.CONTROL_OBJECT_TASKWORK, contentIds);
            }
            int status = delTaskwork.getStatus();
            if (bindedRiskFactorCount == 0 && status == TaskworkStateEnum.未启动.getCode()) {// 未关联风险因素 && 未启动
                iSpcTaskworkContentDao.deleteByTaskworkId(delTaskwork.getId());
                iSpcTaskworkDao.delete(delTaskwork);
            } else if (bindedRiskFactorCount > 0) {
                reject1.add(delTaskwork);
            } else if (status > TaskworkStateEnum.未启动.getCode()) {// 作业活动已启动
                reject2.add(delTaskwork);
            }
        });
        if (reject1.size() > 0) {
            throw new YeeException("该作业活动已关联风险因素，请先解绑后再删除");
        }
        if (reject2.size() > 0) {
            throw new YeeException("该作业活动已启动，不能删除");
        }
        delTaskworkList.removeAll(reject1);
        delTaskworkList.removeAll(reject2);
        return delTaskworkList;
    }

    @Transactional
    @Override
    public SpcTaskworkContent deleteTaskworkContent(Long contentId) {
        SpcTaskworkContent spcTaskworkContent = iSpcTaskworkContentDao.findById(contentId).get();
        // 判断步骤是否可删除
        List<SpcRiskFactor> riskFactors = iSpcRiskFacotrDao
                .findByControlObjectIdAndType(FasConstant.CONTROL_OBJECT_TASKWORK, contentId);
        if (null != riskFactors && riskFactors.size() == 0) {// 未关联风险因素可删除
            if (iSpcTaskworkContentDao.findById(contentId).get() != null) {
                iSpcTaskworkContentDao.deleteById(contentId);
            }
        } else {
            throw new YeeException("该作业活动步骤已关联风险因素，请先解绑后再删除");
        }
        return spcTaskworkContent;
    }

    @Override
    public SpcTaskwork getTaskworkDetail(Long taskworkId) {
        SpcTaskwork taskwork = iSpcTaskworkDao.findById(taskworkId).get();
        if (taskwork != null) {
            if (taskwork.getEquipmentId() != null) {
                SpcEquipment equipment = iSpcEquipmentDao.findById(taskwork.getEquipmentId()).get();
                if (equipment != null) {
                    taskwork.setWorkshopSection(equipment.getWorkshopSection());
                    taskwork.setEquipmentCode(equipment.getCode());
                    taskwork.setEquipmentName(equipment.getName());
                }
            }
            List<SpcTaskworkContent> contentList = iSpcTaskworkContentDao.findAllByTaskworkId(taskworkId);
            taskwork.setContentList(contentList);
        }
        return taskwork;
    }

    @Override
    public List<SpcTaskwork> listAll(List<DaoCriteria> criterias) {
        BaseQuerySpecification<SpcTaskwork> specification = new BaseQuerySpecification<>(criterias);
        List<SpcTaskwork> list = iSpcTaskworkDao.findAll(specification);
        return list;
    }

    @Override
    public SpcTaskwork getTaskworkContentIdDetail(Long taskworkContentId) {
        SpcTaskworkContent content = iSpcTaskworkContentDao.findById(taskworkContentId).get();
        if (content.getTaskworkId() != null) {
            SpcTaskwork taskwork = iSpcTaskworkDao.findById(content.getTaskworkId()).get();
            List<SpcTaskworkContent> contentList = new ArrayList<>();
            contentList.add(content);
            taskwork.setContentList(contentList);
            return taskwork;
        } else {
            return null;
        }
    }

}
