package com.yeejoin.amos.spc.business.controller;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONObject;
import com.yeejoin.amos.op.core.common.response.CommonResponse;
import com.yeejoin.amos.op.core.util.CommonResponseUtil;
import com.yeejoin.amos.spc.business.dao.repository.ISpcRiskFactorDao;
import com.yeejoin.amos.spc.business.param.CommonPageInfoParam;
import com.yeejoin.amos.spc.business.param.ReginParams;
import com.yeejoin.amos.spc.business.param.RiskFactorControlObjectParam;
import com.yeejoin.amos.spc.business.service.intfc.ISpcRiskFactorService;
import com.yeejoin.amos.spc.business.service.intfc.ISpcRiskSourceService;
import com.yeejoin.amos.spc.common.enums.RiskFactorsTypeEnum;
import com.yeejoin.amos.spc.core.common.request.CommonPageable;
import com.yeejoin.amos.spc.core.common.request.CommonRequest;
import com.yeejoin.amos.spc.core.common.request.RiskFactorAddRequest;
import com.yeejoin.amos.spc.core.common.request.RiskFactorImproveRequest;
import com.yeejoin.amos.spc.core.util.CommonPageParamUtil;
import com.yeejoin.amos.spc.dao.entity.SpcRiskFactor;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@RestController
@RequestMapping(value = "/api/spcRiskFactor")
@Api(tags = "风险因素")
public class SpcRiskFactorController extends BaseController {
    private final Logger log = LoggerFactory.getLogger(SpcRiskFactorController.class);

    @Autowired
    private ISpcRiskSourceService spcRiskSourceService;

    @Autowired
    private ISpcRiskFactorDao spcRiskFactorDao;

    @Autowired
    private ISpcRiskFactorService iSpcRiskFactorService;

    /**
     * 验证危险因素名称是否存在
     */
    @ApiOperation(httpMethod = "GET", value = "验证危险因素名称是否存在", notes = "验证危险因素名称是否存在")
    @GetMapping(value = "/verifyName", produces = "application/json;charset=UTF-8")
    public CommonResponse verifyName(
            @ApiParam(value = "危险源id", required = true) @RequestParam Long riskSourceId,
            @ApiParam(value = "危险因素名称") @RequestParam(required = false) String name) {
        return CommonResponseUtil.success(spcRiskSourceService.isExistFactorByName(riskSourceId, name));
    }

    /**
     * 获取危险因素类型数据
     */
    @ApiOperation(httpMethod = "GET", value = "获取风险源类型", notes = "获取风险源类型")
    @GetMapping(value = "/type", produces = "application/json;charset=UTF-8")
    public CommonResponse getRiskType() {
        return CommonResponseUtil.success(RiskFactorsTypeEnum.getEnumList());
    }

    /**
     * 新增编辑风险因素
     */
    @ApiOperation(httpMethod = "POST", value = "新增风险因素", notes = "新增风险因素")
    @PostMapping(value = "/save", produces = "application/json;charset=UTF-8")
    public CommonResponse saveSpcEvaluateModel(@RequestBody RiskFactorAddRequest riskFactors) {
        riskFactors.setCreatorId(getUserId());
        spcRiskSourceService.saveRiskFactor(riskFactors);
        return CommonResponseUtil.success();
    }

    /**
     * 删除风险因素
     */
    @ApiOperation(httpMethod = "DELETE", value = "删除风险因素", notes = "删除风险因素")
    @DeleteMapping(value = "/delete/{ids}", produces = "application/json;charset=UTF-8")
    public CommonResponse delete(@PathVariable String ids) {
        String[] idArray = ids.split(",");
        spcRiskSourceService.deleteRiskFactor(idArray);
        return CommonResponseUtil.success();
    }


    /**
     * 风险因素列表页面查询
     */
    @ApiOperation(httpMethod = "POST", value = "风险因素列表页面查询", notes = "风险因素列表页面查询")
    @PostMapping(value = "/list", produces = "application/json;charset=UTF-8")
    public CommonResponse queryPage(
            @ApiParam(value = "查询条件") @RequestBody(required = false) List<CommonRequest> queryRequests,
            @ApiParam(value = "分页参数", required = true) CommonPageable commonPageable) {
        CommonPageInfoParam param = CommonPageParamUtil.fillCommonPageInfoParam(queryRequests, commonPageable);
        Page<HashMap<String, Object>> riskFactorList = spcRiskSourceService.queryRiskFactorByPage(param);
        return CommonResponseUtil.success(riskFactorList);
    }

    /**
     * 批量启动风险因素评价流程
     */
    @ApiOperation(value = "批量启动风险因素评价流程", notes = "批量启动风险因素评价流程")
    @PostMapping(value = "/startBatchFlow", produces = "application/json;charset=UTF-8")
    public CommonResponse startBatchFlow(@ApiParam(value = "风险因素编号（多个逗号隔开）", required = true) @RequestParam String riskFactorIds) {
        List<Long> ids = Arrays.stream(riskFactorIds.split(",")).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
        ReginParams reginParams = getSelectedOrgInfo();
        String departmentId = String.valueOf(getDepartmentId(reginParams));
        String companyId = getCompanyId(reginParams);
        return iSpcRiskFactorService.startBatchFlow(ids, getUserId(), companyId, departmentId, getToken());
    }

    /**
     * 获取设备设施风险因素信息
     */
    @ApiOperation(value = "获取设备设施风险因素信息", notes = "获取设备设施风险因素信息")
    @GetMapping(value = "/equipment/getByInstanceId")
    public CommonResponse getByInstanceIdOfEquipment(@ApiParam(value = "实例编号", required = true) @RequestParam Long instanceId) {
        return iSpcRiskFactorService.getByInstanceIdOfEquipment(instanceId);
    }

    /**
     * 获取作业活动风险因素信息
     */
    @ApiOperation(value = "获取作业活动风险因素信息", notes = "获取作业活动风险因素信息")
    @GetMapping(value = "/taskwork/getByInstanceId")
    public CommonResponse getByInstanceIdOfTaskwork(@ApiParam(value = "实例编号", required = true) @RequestParam Long instanceId) {
        return iSpcRiskFactorService.getByInstanceIdOfTaskwork(instanceId);
    }


    /**
     * 设备设施风险因素流程回调-交大调用
     */
    @ApiOperation(value = "设备设施风险因素流程回调", notes = "设备设施风险因素流程回调")
    @PostMapping(value = "/equipment/flowCallBack")
    public CommonResponse flowCallBackOfEquipment(@ApiParam(value = "风险因素对象", required = true) @RequestBody RiskFactorImproveRequest request) {
        log.info("设备设施风险因素流程回调========================>" + JSONObject.toJSONString(request));
        return iSpcRiskFactorService.flowCallBackOfEquipment(request, getUserId(), getToken());
    }
    
    /**
     * 作业活动评级回调-交大调用
     */
    @ApiOperation(value = "作业活动危险因素评级回调",notes = "作业活动危险因素评级回调")
    @PostMapping(value = "/taskwork/flowCallBack")
    public CommonResponse flowCallBackOfTaskwork(@ApiParam(value = "风险因素对象",required = true) @RequestBody RiskFactorImproveRequest request){
    	log.info("作业活动风险因素等级评级流程回调==>" + JSONObject.toJSONString(request));
    	return iSpcRiskFactorService.flowCallBackOfTaskwork(request, getUserId(), getToken());
    }

    /**
     * 根据ID查询危险因素详情
     */
    @ApiOperation(value = "查询危险因素详情", notes = "查询危险因素详情")
    @GetMapping(value = "/{id}", produces = "application/json;charset=UTF-8")
    public CommonResponse queryRiskFactorDetailById(@ApiParam(value = "危险因素ID", required = true) @PathVariable Long id) {
        try {
            SpcRiskFactor result = spcRiskFactorDao.findById(id).get();
            return CommonResponseUtil.success(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询危险因素详情失败");
        }
    }

    @ApiOperation(value = "风险因素关联管控对象详情查询", notes = "风险因素关联管控对象详情查询")
    @GetMapping(value = "/controlObject/details/{riskFactorId}", produces = "application/json;charset=UTF-8")
    public CommonResponse queryControlObjectDetailById(@ApiParam(value = "风险因素ID", required = true) @PathVariable Long riskFactorId) {
        try {
            HashMap<String, Object> result = spcRiskSourceService.queryControlObjectDetailById(riskFactorId);
            return CommonResponseUtil.success(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询管控对象详情失败");
        }
    }

    /**
     * 查询危险因素管控措施详情
     */
    @ApiOperation(value = "查询危险因素管控措施详情", notes = "查询危险因素管控措施详情")
    @GetMapping(value = "/controlMeasure/details/{riskFactorId}/{evaluateId}", produces = "application/json;charset=UTF-8")
    public CommonResponse queryControlMeasureDetailByRiskFactorId(@ApiParam(value = "危险因素ID", required = true) @PathVariable Long riskFactorId
    		,@ApiParam(value = "流程ID", required = true) @PathVariable Long evaluateId) {
        try {
            List<HashMap<String, Object>> result = spcRiskSourceService.queryControlMeasureDetailByRiskFactorId(riskFactorId,evaluateId);
            return CommonResponseUtil.success(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询危险因素详情失败");
        }
    }

    /**
     * 关联设备设施到风险因素
     */
    @ApiOperation(value = "关联设备设施到风险因素", notes = "关联设备设施到风险因素")
    @PostMapping(value = "/equipment", produces = "application/json;charset=UTF-8")
    public CommonResponse bindEquipments(@ApiParam(value = "危险因素ID", required = true) @RequestBody List<RiskFactorControlObjectParam> param) {
        try {
            iSpcRiskFactorService.bindEquipment(param);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("关联设备设施失败");
        }
    }
    
    /**
     * 关联作业活动到风险因素
     */
    @ApiOperation(value = "关联作业活动到风险因素", notes = "关联作业活动到风险因素")
    @PostMapping(value = "/taskwork", produces = "application/json;charset=UTF-8")
    public CommonResponse bindTaskworks(@ApiParam(value = "危险因素ID", required = true) @RequestBody List<RiskFactorControlObjectParam> param) {
    	try {
    		iSpcRiskFactorService.bindTaskwork(param);
    		return CommonResponseUtil.success();
    	} catch(Exception e) {
    		log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("关联作业活动失败");
    	}
    }
}
