/* eslint-disable no-magic-numbers */

import * as THREE from 'three';
import { TextureFactory, modelClassifyMapper } from 'amos-3d';
import { backgroundHelpers } from 'amos-3d/lib/helpers';
import { utils } from 'amos-tool';
import { getRegion } from './../../utils/cacheUtils';

const { getInnerSkybox } = backgroundHelpers;

const customSkybox = backgroundHelpers.createBackground({
  basePath: '/threeres/bg/bluesky/',
  names: ['posx.jpg', 'negx.jpg', 'posy.jpg', 'negy.jpg', 'posz.jpg', 'negz.jpg']
});

const getMaterial = () => {
  const groundTexture = TextureFactory.createTextureByUrl('/threeres/mesh/grid-gray.jpg', {
    wrapS: THREE.RepeatWrapping,
    wrapT: THREE.RepeatWrapping,
    anisotropy: 16
  });
  groundTexture.repeat.set(25, 25);
  const groundMaterial = new THREE.MeshLambertMaterial({ map: groundTexture });
  return groundMaterial;
};

const getSeriesModel = (key, type, name, count = 1) => {
  const result = [];
  for (let i = 1; i < count + 1; i++) {
    let newKey = i;
    if (i < 10){
      newKey = `${key}_0${i}`;
    }
    result.push({
      key: newKey, type, sortOrder: 0, displayName: `${name}-${i}`
    });
  }
  return result;
};

/**
 * 从数组中组装具体的数据，返回 数组
 * @param {Array} arr
 * @param {string|array} props
 */
export const getPropFromArr = (arr, props) => {
  const result = [];
  arr.forEach(a => {
    if (utils.isArray(props)){
      const item = {};
      props.forEach(p => {
        if (a[p]){
          item[p] = a[p];
        }
      });
      if (!utils.isEmpty(item)){
        result.push(item);
      }
    } else {
      if (a[props]){
        result.push(a[props]);
      }
    }
  });
  return result;
};

/**
 * 视图控制
 */
export const viewController = [
  { key: 'park-contorller', icon: 'shu-changqu', type: 'park', label: '选择建筑' },
  { key: 'burst-contorller', icon: 'zhalie', type: 'burst', singleClick: true, label: '炸裂' },
  { key: 'fitview-contorller', icon: 'zuijiashijiao', type: 'fitview', singleClick: true, label: '初始视图' },
  { key: 'explain-controller', icon: 'jianpan', type: 'explain', singleClick: true, label: '控件' }
];

/**
 * 业务控制
 */
export const bizControls = [
  { key: 'riskSource-point', icon: 'fengxiandian2', type: 'riskSource', label: '风险点', subs: [
    { key: 'level_4', label: '4级' },
    { key: 'level_3', label: '3级' },
    { key: 'level_2', label: '2级' },
    { key: 'level_1', label: '1级' },
    { key: 'level_0' , label: '重大危险源', type: 'majorHazard' }
  ] },
  { key: 'taskwork-point',  icon: 'zuoyehuodong', type: 'taskWork', label: '作业活动', subs: [
    { key: 'level_5', label: '执行中' },
    { key: 'level_6', label: '确认中' }
  ] },
  { key: 'equipment-point',  icon: 'huagongshebei', type: 'equipment', label: '设备设施' },
  { key: 'danger-point', icon: 'yinhuandian', type: 'danger', label: '隐患点', subs: [
    { key: 'level_1', label: '一般隐患' },
    { key: 'level_2', label: '重大隐患' }
  ] }
];


export const bizControlsTypes = [
  { key: 'riskSource', label: '风险点' },
  { key: 'danger', label: '隐患点' },
  { key: 'equipment' , label: '设备设施' }
];

export const riskSourceLevel = [
    { key: '1', label: '1级' },
    { key: '2', label: '2级' },
    { key: '3', label: '3级' },
    { key: '4', label: '4级' }
];

export const dangerStatus = [
  { key: '1', label: '一般隐患' },
  { key: '2', label: '重大隐患' }
];

export const equipmentStatus = [
  { key: '01', label: '生产设备' },
  { key: '02', label: '应急设备' }
];

/**
 * 换流站
 */
export const getStationConfig = () => {
  return {
    cameraEffects: {
      initConf: {
        position: { x: 2436, y: 1376, z: 1001 }
      },
      fitConf: {
        position: [1232, 1228, 27.98],
        target: [-1, -2, -2.5]
      }
    },
    getObjs(){
      return {
        basePath: '/threeres/models/station/',
        objName: 'station.obj',
        mtlName: 'station.mtl'
      };
    }
  };
};

export const getYinanConfig = () => {
  const names = [
    { key: 'all', type: modelClassifyMapper.ALL, sortOrder: 0, displayName: '厂区', ignore: true },
    { key: 'ground', type: modelClassifyMapper.GROUND, sortOrder: 0, displayName: '地面' },
    ...getSeriesModel('building', modelClassifyMapper.BUILDING, '监控室', 4),
    ...getSeriesModel('powerdev', modelClassifyMapper.BUILDING, '电塔', 10)
  ];
  return {
    cameraEffects: {
      initConf: {
        position: { x: 117.20, y: 780.13, z: -944.57 }
      },
      fitConf: {
        position: [-357.67, 318.57, 386.15],
        target: [-1, -2, -2.5]
      }
    },
    layerConfig: {
      // defaultSelects: getPropFromArr(names, 'key'), // 多选
      defaultSelects: 'all',
      layers: getPropFromArr(names.filter(n => n.key !== 'helper'), ['key', 'displayName'])
    },
    getObjs(){
      return names.filter(n => !n.ignore).map(o => {
        return {
          basePath: '/threeres/models/yinan/',
          objName: `${o.key}.obj`,
          mtlName: `${o.key}.mtl`,
          modelClassify: o.type,
          sortOrder: o.sortOrder
        };
      });
    }
  };
};

export const getCurrentObjs = (key) => {
  const region = getRegion();
  const regionObj = JSON.parse(region) || {};
  const cityCode = key || regionObj.cityCode;

  if (cityCode === '济南'){
    //
  } else if (cityCode === '青岛'){
    //
  }

  return getStationConfig();
};

export default {
  ratio: 100,
  complete: 100,
  sceneBg: customSkybox,
  skybox: getInnerSkybox('bluesky'),
  fog: {
    color: 0x89bffb,
    neer: 5000,
    far: 10000
  },
  // fog: null,
  dlight: {
    color: 0xffffff,
    intensity: 1.0
  },
  plight: {
    color: 0xffffff,
    intensity: 0.8
  },
  plugins: {
    OrbitControls: {
      enable: true,
      params: {
        maxPolarAngle: 1.5, // 上下翻转的最大角度 Math.PI
        minPolarAngle: 0.5, // 上下翻转的最小角度 默认 0
        minDistance: 10, // 最近距离，表现在放大
        maxDistance: 5000, // 最远距离，表现在缩小
        rotateSpeed: 0.5, // 旋转速度 1.0
        dampingFactor: 0.2, // 动态阻尼系数 就是鼠标拖拽旋转灵敏度，阻尼越小越灵敏 0.25
        enableZoom: true, // 是否允许缩放
        // autoRotate: true, // 是否自动旋转
        // autoRotateSpeed: 0.2, // 自动旋转速度
        enablePan: true // 是否开启右键拖拽
      }
    },
    CameraHelper: {
      enable: true
    }
  },
  camera: {
    fov: 45,
    near: 1,
    far: 10000
  },
  meshBg: {
    geometry: new THREE.PlaneBufferGeometry(20000, 20000),
    material: getMaterial(),
    position: { y: -30 },
    rotation: { x: -Math.PI / 2 }
  },
  // 展开距离
  spacing: 50,
  link: {
    selectColor: 0x008000,
    fatlineProps: {
      lineStyle: {
        color: 0xff0000,
        linewidth: 3
      }
    },
    normalLineProps: {
      type: 'Line',
      lineStyle: {
        color: 0xff0000,
        linewidth: 1
      }
    }
  },
  // marker 移动耗时
  moveDuration: 3000
};

export const desigerConf = {
  sceneOptions: {
    background: 0x626667
  },
  cameraOptions: {
    fov: 45,
    near: 1,
    far: 5000,
    position: [293.92, 310.55, 1026.64],
    // 是否启用拾取器
    enablePicker: true,
    builtInOrbit: {
      maxPolarAngle: Math.PI * 0.5,
      minDistance: 1,
      maxDistance: 5000
    }
  },
  // 配置信息，支持灯光的配置
  options: {
    light: [
      { type: 'AmbientLight', options: { color: 0xffffff, intensity: 0.15 } },
      { type: 'DirectionalLight', options: { color: 0xffffff, intensity: 0.1 } }
      // { type: 'PointLight', options: { color: 0xffffff, intensity: 0.5, position: { x: -200 } } }
    ],
    floorBoard: {
      visible: true,
      position: { y: -30 },
      showType: 'grid',
      size: 2000,
      divisions: 20,
      colorCenterLine: 0x838a94,
      colorGrid: 0xd3dbe8,
      gridOptions: {
        opacity: 0.2,
        transparent: true
      },

      widthSegments: 1,
      heightSegments: 1,
      boardColor: 0x626667,
      boardOptions: {

      }
    }
  }
};

export const curveConf = {
  parabola: {
    size: 5,
    color: 0xff1c1c
  }
};

export const desigerHelperConfig = {
  cameraEffect: {
    position: [-18, 514.33, 410.65],
    target: [1, 2.5, 3]
  },
  objAnchor: {
    position: [0, 5, 0]
  }
};

export const shortcutKeys = [
  {
    category: '三维移动控制',
    keys: [
      { name: '上移', key: '↑', status: '正常' },
      { name: '下移', key: '↓', status: '正常' },
      { name: '左移', key: '←', status: '正常' },
      { name: '右移', key: '→', status: '正常' }
    ]
  }
];
