import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Select, Input } from 'amos-framework';
import AmosGridTable from './../../common/tableComponent/table/AmosGridTable';
import { getFireEquipmentListAction } from './../../../../services/ledgerService';
import EditableCell from './../../common/editableCell/EditableCell';

const Option = Select.Option;

const EquipClassifyEnum = { 0: '设备类', 1: '耗材类', 2: '视频监控', 3: '灭火器材' };

const fireEquipmentTableColumns = self => {
  return [
    {
      title: '序号',
      render: (text, record, index) => `${index + 1}`
    },
    {
      title: '编号',
      dataIndex: 'code',
      key: 'code',
      width: '10%'
    },
    {
      title: '名称',
      dataIndex: 'name',
      key: 'name',
      width: '14%'
    },
    {
      title: '分类',
      dataIndex: 'equip_classify',
      key: 'equip_classify',
      width: '9%',
      render: (text) => EquipClassifyEnum[text]
    },
    {
      title: '类型',
      dataIndex: 'equip_type',
      key: 'equip_type',
      width: '9%'
    },
    {
      title: '状态',
      dataIndex: 'isBind',
      key: 'isBind',
      width: '10%',
      render: text => text === 'YES' ? '已绑定设备' : ''
    },
    {
      title: '规格型号',
      dataIndex: 'model',
      key: 'model',
      width: '8%'
    },
    {
      title: '厂商',
      dataIndex: 'manufacturer',
      key: 'manufacturer',
      width: '14%'
    },
    {
      title: '数量',
      dataIndex: 'number',
      key: 'number',
      width: '5%'
    },
    {
      title: '添加数量',
      dataIndex: 'numberInStation',
      key: 'numberInStation',
      width: '9%',
      render: (text, record) => record.equip_classify === 1 ? (
        <EditableCell value={text} onChange={(text) => self.onCellChange(record, text)} />
      ) : ''
    }
  ];
};

const defaultHeight = 580;

/**
 * 消防设备
 */
class EquipmentFireEquipmentAdd extends Component {
  constructor(props) {
    super(props);

    this.pageConfig = {
      pageNumber: 0,
      pageSize: 10
    },
    this.state = {
      name: '',
      pagination: true,
      selectedRows: [],
      selectedRowKeys: [],
      dataList: [], //表格数据集合
      size: 'small', //表格大小
      searchParam: { equipClassify: '0,1,2,3' },
      totals: 0, //所有数据总和
      requestParam: [],
      showDetail: true, //是否显示详情
      truckId: 0, //消防车id
      enableConfig: false, //是否打开配置监测点页面
      numberInStation: 1, //添加设备数量
      unitInStation: '',  //添加设备单位
      fireEquipmentId: 0 //设备ID
    };
  }

  onBtnClick() {
    let { searchParam } = this.state;

    this.getFireEquipmentListData(searchParam);
  }

  onSelectChange(index, value) {
    let { searchParam } = this.state;

    searchParam.equipClassify = value.value === -1 ? '0,1,2,3' : value.value;
    searchParam.pageNumber = this.pageConfig.pageNumber;
    searchParam.pageSize = this.pageConfig.pageSize;
    this.setState({
      searchParam
    }, () => this.getFireEquipmentListData(searchParam));
  }

  onChange = (e, value) => {
    let { searchParam } = this.state;
    searchParam.name = value;
    searchParam.pageNumber = this.pageConfig.pageNumber;
    searchParam.pageSize = this.pageConfig.pageSize;
    this.setState({
      searchParam,
      name: value
    }, () => this.getFireEquipmentListData(searchParam));
  };

  onCellChange = (record, text) => {
    let { selectedRowKeys, selectedRows } = this.state;
    let rowKey = record.id;
    let row = record;
    selectedRowKeys.push(rowKey);
    selectedRowKeys = Array.from(new Set([...selectedRowKeys]));
    selectedRows.push(row);

    this.setState({
      numberInStation: parseInt(text),
      unitInStation: record.unit,
      selectedRowKeys,
      selectedRows
    });
  };

  getFireEquipmentListData = param => {
    let { searchParam } = this.state;
    if (param.current != null) {
      //前端的current 即后端的pageNumber
      param.pageNumber = param.current;
    }
    Object.assign(searchParam, { ...param });
    getFireEquipmentListAction(searchParam, param.pageNumber, param.pageSize).then(data => {
      this.setState({
        dataList: data.content,
        totalCount: data.totalElements
      });
    });
  };

  getPanelHeight = () => {
    return defaultHeight;
  };

  setPageConfig = ({ pageSize, current }) => {
    if (pageSize !== undefined) {
      this.pageConfig.pageSize = pageSize;
    }
    if (current !== undefined) {
      this.pageConfig.current = current;
    }
  };

  getRowClassName = (record, index) => {
    let highlight = this.highlight;
    if (highlight && record.id === highlight) {
      return 'highlight-row';
    } else {
      return 'normal-row';
    }
  };

  getSelectedRows = (selectedRows, selectedRowKeys) => {
    this.setState({ selectedRows, selectedRowKeys });
  };

  /**
   * 获取表格刷新方法
   */
  reload = r => {
    this.setState(
      {
        reload: () => {
          r();
          this.setState({ selectedRows: [], selectedRowKeys: [] });
        }
      },
      r()
    );
  };

  render() {
    let { name, selectedRowKeys, dataList, totalCount } = this.state;
    let _true_ = true;

    return (
      <div className="equipment-fire-equipment" style={{ background: '#33333', height: '100%' }}>
        <div className="equipment-fire-equipment-search">
          <div>
            <span>分类：</span>
            <Select onChange={(e, value) => this.onSelectChange(e, value)} defaultValue={-1}>
              <Option value={-1}>全部</Option>
              <Option value={0}>设备类</Option>
              <Option value={1}>耗材类</Option>
              <Option value={2}>视频监控</Option>
              <Option value={3}>灭火器材</Option>
            </Select>
            <span>名称：</span>
            <Input value={name} onChange={e => this.onChange('name', e.target.value)} />
          </div>
        </div>
        <AmosGridTable
          rowKey="id"
          columns={fireEquipmentTableColumns(this)}
          callBack={this.reload}
          dataList={dataList}
          totals={totalCount}
          getTableDataAction={this.getFireEquipmentListData}
          getSelectedRows={this.getSelectedRows}
          rowClassName={this.getRowClassName}
          setPageConfig={this.setPageConfig}
          getPanelHeight={this.getPanelHeight}
          defaultPageConlHeight={this.getPanelHeight}
          isChecked={_true_}
          selectedRowKeys={selectedRowKeys}
        />
      </div>
    );
  }
}

EquipmentFireEquipmentAdd.propTypes = {
  fireEquipmentId: PropTypes.number,
  totalCount: PropTypes.number,
  dataList: PropTypes.array,
  getTableListData: PropTypes.func
};
export default EquipmentFireEquipmentAdd;
