import React, { Component } from 'react';
import PropTypes from 'prop-types';
import AmosEcharts from 'amos-viz/lib/echarts';
import { queryRiskSourceMatrixAction } from './../../../../services/situationService';

/**
 * 发生率/严重度矩阵
 * @class IncidenceSeverityMatrix
 * @extends {Component}
 */
class IncidenceSeverityMatrix extends Component {
  constructor(props) {
    super(props);
    this.state = {
      dataAll: [],
      dangerPoints: [],
      safePoints: [],
      warnPoints: []
    };
  }

  componentDidMount() {
    this.queryRiskSourceMatrix();
  }

  getOptions() {
    return {
      grid: [{ x: '8%', y: '13%', width: '75%', height: '75%' }],
      tooltip: {
        formatter: item => {
          let text = item.value;
          // console.log(JSON.stringify(item));
          return text[2] + '<br>' + 'S:' + text[0] + ' ' + ' O:' + text[1];
        }
      },
      color: ['green', 'red', '#ff6600'],
      xAxis: [
        {
          nameGap: 5,
          gridIndex: 0,
          min: 0,
          max: 10,
          name: 'S(严重度)',
          // nameRotate: 90,
          nameTextStyle: {
            fontWeight: 'bold',
            fontSize: 15
          }
        }
      ],
      yAxis: [
        {
          nameGap: 5,
          gridIndex: 0,
          min: 0,
          max: 10,
          name: 'O(故障率)',
          nameTextStyle: {
            fontWeight: 'bold',
            fontSize: 15
          }
        }
      ],
      // symbol: 'rect',
      legend: {
        top: 5,
        right: 100,
        selectedMode: false,
        data: ['最低线', '最高线', '']
      },
      series: this.getSeries()
    };
  }

  getSeries() {
    let series = [];
    let safetySeries = {
      name: '最低线',
      type: 'scatter',
      xAxisIndex: 0,
      yAxisIndex: 0,
      data: this.state.dataAll[2],
      markLine: this.markLineOpt('safety')
    };
    let dangerSeries = {
      name: '最高线',
      type: 'scatter',
      xAxisIndex: 0,
      yAxisIndex: 0,
      data: this.state.dataAll[0],
      markLine: this.markLineOpt('danger')
    };
    let warnSeries = {
      name: '111',
      type: 'scatter',
      xAxisIndex: 0,
      yAxisIndex: 0,
      data: this.state.dataAll[1]
    };
    series.push(safetySeries);
    series.push(dangerSeries);
    series.push(warnSeries);
    return series;
  }

  markLineOpt(type) {
    var data1 = [];
    let color = 'green';
    if (type === 'safety') {
      color = 'green';
      data1 = [
        {
          coord: [7, 0],
          symbol: 'none'
        },
        {
          coord: [0, 4],
          symbol: 'none'
        }
      ];
    } else if (type === 'danger') {
      color = 'red';
      data1 = [
        {
          coord: [9, 0],
          symbol: 'none'
        },
        {
          coord: [0, 8],
          symbol: 'none'
        }
      ];
    }
    return {
      animation: false,
      label: {
        normal: {
          formatter: '',
          textStyle: {
            align: ''
          }
        }
      },
      lineStyle: {
        normal: {
          type: 'solid',
          color: color
        }
      },
      tooltip: {
        formatter: ''
      },
      data: [data1]
    };
  }

  queryRiskSourceMatrix() {
    queryRiskSourceMatrixAction().then(data => {
      data &&
        data.map(e => {
          this.classifyPoints(e);
        });
      let { dataAll, dangerPoints, safePoints, warnPoints } = this.state;
      dataAll.push(dangerPoints);
      dataAll.push(warnPoints);
      dataAll.push(safePoints);
      this.setState({
        dataAll
      });
    });
  }

  classifyPoints(point) {
    let { dangerPoints, safePoints, warnPoints } = this.state;
    let d = [];
    d.push(point.evaluationSid);
    d.push(point.evaluationOid);
    d.push(point.name);
    if (((8 * point.evaluationSid) + (9 * point.evaluationOid)) > 72) {//红色区域：S>=9或O>=8 (直线右侧区域8x + 9y - 72 > 0)
      dangerPoints.push(d);
    } else if (((4 * point.evaluationSid) + (7 * point.evaluationOid)) < 28) {//绿色区域：S<7且O<4 (直线左侧区域4x + 7y - 28 < 0)
      safePoints.push(d);
    } else {//橙色区域：S<9且O<8&&S>=7且O>=4
      warnPoints.push(d);
    }
    this.setState({
      dangerPoints,
      safePoints,
      warnPoints
    });
  }

  render() {
    const option = this.getOptions();
    return (
      <div className="risk-incidence-severity-matrix">
        <AmosEcharts option={option} />
      </div>
    );
  }
}

IncidenceSeverityMatrix.propTypes = {};

IncidenceSeverityMatrix.defaultProps = {
  data: {},
  legend: [],
  chartStyle: { width: '100%', height: '10rem' }
};

export default IncidenceSeverityMatrix;
