import React, { Component } from 'react';
import { CardPane, Row } from 'amos-framework';
import { Col } from 'amos-framework/lib/grid';
import * as endConf from 'amos-processor/lib/config/endconf';
import moment from 'moment';
import BizIcon from '../../common/icon/BizIcon';
import { getScreenSaverDataAction } from './../../../services/3dService';
import RiskPointPie from './RiskPointPie';

const AmosConfig = endConf.AmosConfig;
const deployDate = AmosConfig.screenSaverConf.deployDate;
const levelCount = 5;

class ScreenSaverView extends Component {
  constructor(props) {
    super(props);

    this.state = {
      equipmentCount: 0,  //重点设备
      detectorCount: 0, //探测器
      monitorCount: 0,  //监控点
      factoryRpn: 0,  //RPNr
      rpnUpOrDown: 'NOCHANGE',  //rpn上升/下降/未变化标识
      riskSourceData: {}, //风险点
      riskSourceCount: 0,  //风险点数量
      patrolPointData: {},  //巡检点
      pointCount: 0 //巡检点数量
    };
  }

  componentWillMount() {
  }

  componentDidMount() {
    getScreenSaverDataAction().then(data => {
      let { equipmentCount, detectorCount, monitorCount, factoryRpn = 0, rpnUpOrDown } = data;
      for (let i = 0; i < data.equipClassify.length; i++) {
        let item = data.equipClassify[i];
        if (item.equip_classify === 0) {  // 探测器
          detectorCount = item.classifyCount;
        } else if (item.equip_classify === 2) { //监控点
          monitorCount = item.classifyCount;
        }
      }
      let riskSourceData = this.parseRiskSourceData(data);
      let patrolPointData = this.parsePatrolPointData(data);

      this.setState({
        equipmentCount,
        detectorCount,
        monitorCount,
        factoryRpn,
        rpnUpOrDown,
        riskSourceData,
        patrolPointData
      });
    });
  }

  parseRiskSourceData(data) {
    let riskSourceData = [];
    riskSourceData.count = data.riskSourceCount;
    riskSourceData.typeName = '风险点';
    riskSourceData.colors = ['#ff0000', '#ff7611', '#ffcc38', '#1e96d9', '#b920ff'];
    let tempData = [];
    for (let j = 1; j <= levelCount; j++) {
      let item = {};
      switch (j) {
        case 1 : {item.color = '#ff0000'; item.value = 0; item.name = '一级'; item.level = '1'; tempData.push(item); break;}
        case 2 : {item.color = '#ff7611'; item.value = 0; item.name = '二级'; item.level = '2'; tempData.push(item); break;}
        case 3 : {item.color = '#ffcc38'; item.value = 0; item.name = '三级'; item.level = '3'; tempData.push(item); break;}
        case 4 : {item.color = '#1e96d9'; item.value = 0; item.name = '四级'; item.level = '4'; tempData.push(item); break;}
        case 5 : {item.color = '#b920ff'; item.value = 0; item.name = '五级'; item.level = '5'; tempData.push(item); break;}
      }
    }

    tempData.map(e => {
      for (let i = 0; i < data.riskLevel.length; i++) {
        let obj = data.riskLevel[i];
        if (obj.riskLevel === e.level) {
          switch (obj.riskLevel) {
            case '1' : {e.value = obj.levelRiskCount; break;}
            case '2' : {e.value = obj.levelRiskCount; break;}
            case '3' : {e.value = obj.levelRiskCount; break;}
            case '4' : {e.value = obj.levelRiskCount; break;}
            case '5' : {e.value = obj.levelRiskCount; break;}
          }
        }
      }
    });
    riskSourceData.data = tempData;
    return riskSourceData;
  }

  parsePatrolPointData(data) {
    let patrolPointData = [];
    patrolPointData.count = data.pointCount;
    patrolPointData.typeName = '巡检点';
    patrolPointData.colors = ['#ff0000', '#ff7611', '#2085fe'];
    let pData = [];
    for (let j = 0; j < data.pointStatus.length; j++) {
      let obj = data.pointStatus[j];
      let item1 = {};
      item1.name = '漏检 ';
      item1.value = obj.omitCount;
      item1.color = '#ff0000';
      let item2 = {};
      item2.name = '不合格';
      item2.value = obj.unqualifiedCount;
      item2.color = '#ff7611';
      let item3 = {};
      item3.name = '超时 ';
      item3.value = patrolPointData.count - obj.omitCount - obj.qualifiedCount - obj.unqualifiedCount;
      item3.color = '#2085fe';
      let item4 = {};
      item4.name = '合格';
      item4.value = obj.qualifiedCount;
      item4.color = '#696969';
      pData.push(item1);
      pData.push(item2);
      pData.push(item3);
      // pData.push(item4);
    }
    patrolPointData.data = pData;
    return patrolPointData;
  }

  getGuardDate() {
    let a = moment(moment().format('YYYY-MM-DD'));
    let b = moment(deployDate);
    return a.diff(b, 'days');
  }

  getRPNContent() {
    let rpnIndex = this.state.rpnUpOrDown;
    let text = '';
    let clsName = '';
    switch (rpnIndex) {
      // case 'NOCHANGE': text = <span style={{ color: 'blue' }} >-</span>; break;
      case 'NOCHANGE': text = <hr className='hr-bar'/>; clsName = 'nochange'; break;
      case 'UP': text = '↑'; clsName = 'up'; break;
      case 'DOWN': text = '↓'; clsName = 'down'; break;
    }
    return <span className={clsName}>{text}</span>;
  }

  render() {

    let { equipmentCount, detectorCount, monitorCount, factoryRpn = 0, riskSourceData, patrolPointData } = this.state;
    return (
      <div className="screen-saver-view">
        <Row gutter={16}>
          <Col className='col' span={4}>
            <CardPane className='screen-saver-view-rpn' bordered={false}>
              <div className='screen-saver-guard' >
                <div className='guard-days'>{this.getGuardDate()}<span style={{ fontSize: '18px' }}>天</span></div>
                <div className='guard-text'>已为您<pre> </pre>安全守卫</div>
              </div>
              <div className='screen-saver-rpn'>
                <div className='guard-days'>{factoryRpn}</div>
                <div className='guard-text'>RPNr {this.getRPNContent()}</div>
              </div>
            </CardPane>
          </Col>
          <Col className='col' span={4}>
            <Row className='row-1'>
              <CardPane className='screen-saver-view-equipment' title={<BizIcon icon='zhongdianshebeixiangqing' />} bordered={false} footer='重点设备'>
                {equipmentCount}
              </CardPane>
            </Row>
            <Row className='row-2'>
              <Col className='col-2-1' span={12}>
                <CardPane className='screen-saver-view-detector' title={<BizIcon icon='shujudian' />} bordered={false} footer='探测器'>
                  {detectorCount}
                </CardPane>
              </Col>
              <Col className='col-2-2' span={12}>
                <CardPane className='screen-saver-view-monitor' title={<BizIcon icon='xunjiandian' />} bordered={false}  footer='监控点'>
                  {monitorCount}
                </CardPane>
              </Col>
            </Row>
          </Col>
          <Col className='col' span={4}>
            <CardPane className='screen-saver-view-riskpoint' title={<BizIcon icon='fengxiandian' />} bordered={false} footer='风险点'>
              <RiskPointPie data={riskSourceData} />
            </CardPane>
          </Col>
          <Col className='col' span={4}>
            <CardPane className='screen-saver-view-patrolpoint' title={<BizIcon icon='dianwei-zhongdianshebei' />} bordered={false} footer='巡检点'>
              <RiskPointPie data={patrolPointData} />
            </CardPane>
          </Col>
        </Row>
      </div>
    );
  }
}

export default ScreenSaverView;
