import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Connect } from 'amos-framework';
import SingleLayerSelect from './SingleLayerSelect';
import { eventTopics } from '../consts';
import InnateControlItem from './InnateControlItem';

const eventConnect = Connect.eventConnect;

/**
 * 三维视图自身操作
 *
 * @class InnateControlPane
 * @extends {Component}
 */
@eventConnect
class InnateControlPane extends Component {
  constructor(props) {
    super(props);
    this.state = {
      innateActives: [],
      showLabel: false, // 是否显示提示
      position: { top: '', left: '' } // 提示相对鼠标坐标位置
    };
  }

  onItemClick = item => {
    let alreadyActive;
    // 两次点击效果不同
    if (!item.singleClick){
      alreadyActive = this.isActiveInnate(item);
      const newKeys = this.getInnateActive(item, alreadyActive);
      this.setState({
        innateActives: newKeys
      });
    }
    this.props.trigger(eventTopics.innate_operater, {
      key: item.type,
      value: !alreadyActive
    });
  };

  onMulitLayerItemClick = (item, toggle) => {
    this.props.trigger(eventTopics.innate_layer, {
      key: item.key,
      value: !toggle
    });
  };

  onLayerItemClick = item => {
    this.props.trigger(eventTopics.innate_layer, {
      key: item.key
    });
  };

  getInnateActive = (item, alreadyActive) => {
    const { innateActives } = this.state;
    return alreadyActive ? innateActives.filter(ac => ac !== item.key) : [...innateActives, item.key];
  };

  isActiveInnate = item => {
    const { key } = item;
    if (item.singleClick){
      return false;
    }
    const { innateActives } = this.state;
    return innateActives && innateActives.includes(key);
  };

  changeLabelState = (e, type) => {
    if (type === 'over') {// mouse over
      this.setState({ showLabel: true });
    } else if (type === 'out'){// mouse out
      this.setState({ showLabel: false });
    } else {// mouse move
      const x = 15;
      const y = 10;
      let top = `${e.nativeEvent.layerY + y}px`;
      let left = `${e.nativeEvent.layerX + x}px`;
      this.setState({
        position: {
          top,
          left
        }
      });
    }
  }

  renderOperate = (item = {}) => {
    const cls = this.isActiveInnate(item) ? 'control-innate-item control-active' : 'control-innate-item';
    return (<InnateControlItem key={item.key} item={item} cls={cls} onItemClick={this.onItemClick} />);
  };

  render() {
    const { controls = [], layerConfig } = this.props;
    return (
      <div className="control-pane-innate">
        {controls.map(c => {
          return c.type === 'layer' ?
            <SingleLayerSelect key={c.key} {...c} {...layerConfig} onLayerItemClick={this.onLayerItemClick} />
            : this.renderOperate(c);
        })}
      </div>
    );
  }
}

InnateControlPane.propTypes = {
  controls: PropTypes.arrayOf(
    PropTypes.shape({
      key: PropTypes.string.isRequired,
      type: PropTypes.string.isRequired,
      icon: PropTypes.string,
      label: PropTypes.string,
      // 提示信息
      title: PropTypes.string,
      // 两次点击效果是否一样，当设置 singleClick 为 true时，表示无论怎么点击，效果都一样
      // 默认为 false 或 undefined，表示，第一次点击与第二次点击为不同事件效果
      singleClick: PropTypes.bool
    })
  ),
  layerConfig: PropTypes.shape({
    defaultSelects: PropTypes.oneOfType([PropTypes.arrayOf(PropTypes.string), PropTypes.string]),
    layers: PropTypes.arrayOf(
      PropTypes.shape({
        key: PropTypes.string.isRequired,
        displayName: PropTypes.string
      })
    )
  }),
  onItemClick: PropTypes.func,
  subscribe: PropTypes.func,
  trigger: PropTypes.func
};

export default InnateControlPane;
