import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { message } from 'amos-framework';
import AmosWebSocket from 'amos-websocket';
import TopologyComponent from './TopologyComponent';
import TreeComponent from './TreeComponent';
import { getNetTopoAction, updateNetTopoAction } from './../../../services/netTopologyService';
import SysWsURL from './../../../consts/wsUrlConsts';
import PamsTable from './pamsTable/index';
import { UUID } from 'amos-tool';
import classnames from 'classnames';

class NetTopology extends Component {
  constructor(props) {
    super(props);
    this.state = {
      selectedKeys: [],
      wSelectedKeys: '',
      topologyData: {}, //拓扑图数据
      pamsDisplay: false,
      pamsTableContent: [],
      isTreeDisplay: true
    };
  }

  // 拓扑图数据初始化
  initTopologyData = (id, value = '') => {
    getNetTopoAction(id).then(data => {
      if (value === 'refresh') {
        message.success('拓扑图刷新成功!');
      }
      this.setState({
        topologyData: data
      });
    });
  };

  // 树选择
  onSelect = selectedKeys => {
    this.initTopologyData(selectedKeys);
    this.setState({
      selectedKeys,
      wSelectedKeys: ''
    });
  };

  // 节点数据改变
  nodeChange = e => {
    // console.log(e.model.toJson());
    this.updateTopologyData = JSON.parse(e.model.toJson());
  };

  // 保存节点数据
  saveTopologyData = () => {
    const { selectedKeys } = this.state;
    const { nodeDataArray, linkDataArray } = this.updateTopologyData;
    const params = {
      treeid: selectedKeys,
      nodeData: nodeDataArray,
      linkData: linkDataArray
    };
    updateNetTopoAction(params).then(data => {
      message.success('拓扑图保存成功!');
    });
  };

  // 刷新节点
  refreshTopologyData = () => {
    const { selectedKeys } = this.state;
    this.initTopologyData(selectedKeys, 'refresh');
  };

  // 选择节点
  nodeSelectionChanged = e => {
    console.log('e', e);
    if (e.isSelected) {
      this.setState({ pamsDisplay: true });
      console.log(e.data, 'device');
      const pamsTableContent = [<PamsTable data={e.data} key={UUID.uuid('16')}></PamsTable>];
      this.setState({ pamsTableContent });
    } else {
      console.log('没进来');
      this.setState({ pamsDisplay: false });
    }
  };

  //pams底部表
  renderPamsTable = () => {
    const { pamsTableContent } = this.state;
    return pamsTableContent;
  };

  // // webSocket 接收消息
  handleData = data => {
    this.onSelect(data);
    this.setState({
      wSelectedKeys: data
    });
  };

  // 视图切换
  toggleView = value => {
    if (value === 'left') {
      this.setState({
        isTreeDisplay: true,
        activeView: 'left'
      });
    } else {
      this.setState({
        isTreeDisplay: false,
        activeView: 'detail'
      });
    }
    //拓扑图刷新
    const { selectedKeys } = this.state;
    this.initTopologyData(selectedKeys[0], '');
  };

  render() {
    const { topologyData, wSelectedKeys, selectedKeys, pamsDisplay, isTreeDisplay } = this.state;
    return (
      <div className="net-wrapper">
        <AmosWebSocket ref={node => (this.aws = node)} url={SysWsURL.netTopoURI} onMessage={this.handleData} reconnect debug />
        <div className={classnames({ 'left-content': isTreeDisplay, 'left-content-close': !isTreeDisplay })}>
          <div className={classnames({ 'view-header': isTreeDisplay, 'view-header-close': !isTreeDisplay })}>
            <div className="left-view">
              <img className="image-sytle" src="/src/assets/bizView/netTopology/left-view.png" alt="左视图" onClick={() => this.toggleView('left')} />
            </div>
            <div className="right-view">
              <img className="image-sytle" src="/src/assets/bizView/netTopology/detail-view.png" alt="详细视图" onClick={() => this.toggleView('detail')} />
            </div>
          </div>
          <TreeComponent onSelect={this.onSelect} wSelectedKeys={wSelectedKeys} />
        </div>
        <div className="right-content">
          <div className="net-topo-header">
            <div className="net-topo-save">
              <img src="/src/assets/bizView/netTopology/save.png" alt="保存" onClick={this.saveTopologyData} />
            </div>
            <div className="net-topo-refresh">
              <img src="/src/assets/bizView/netTopology/refresh.png" alt="刷新" onClick={this.refreshTopologyData} />
            </div>
          </div>
          <div className="net-topo-content" style={{ height: pamsDisplay ? 'calc(100% - 342px)' : 'calc(100% - 42px) ' }}>
            <TopologyComponent topologyData={topologyData} nodeChange={this.nodeChange} nodeSelectionChanged={this.nodeSelectionChanged} />
          </div>
          {pamsDisplay ? this.renderPamsTable() : null}
        </div>
      </div>
    );
  }
}

NetTopology.propTypes = {};

export default NetTopology;
