import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Connect } from 'amos-framework';
import { connect3D } from 'amos-3d/lib/designer';
import { RiskPoint, PatrolPoint, PersonPoint3D, MonitorPoint, ProbePoint, EquipmentPoint, PointStatistics, DynamicRingDataPoint } from './points';
import { eventTopics, isPointEvent , isLevelFilter } from './consts';
import { parseMarkers, parseLevelFilter } from './dataProcessor';

const eventConnect = Connect.eventConnect;

const Shuttle = (props) => props.children;

/**
 * Markers 池，所有的 markers 均在本类进行加载
 *
 * @export
 * @class PointsPool
 * @extends {Component}
 */
@connect3D
@eventConnect
class PointsPool extends Component {

  constructor(props) {
    super(props);
    this.markerList = {};
  }

  componentDidMount() {
    this.props.subscribe(eventTopics.base3d_view, (topic, data) => {
      if (isPointEvent(topic)){
        parseMarkers(this, topic, data);
      } else if (isLevelFilter(topic)) {
        parseLevelFilter(this, topic, data);
      }
    });
  }

  /**
   * 点创建完成
   *
   * @memberof PointsPool
   */
  onMarkersCreated = (type, { markersCache }) => {
    this.markerList[type] = markersCache;
    (markersCache || []).forEach(mc => {
      const extData = mc.getExtData();
      extData.orgCode =  this.props.orgCode;
      // console.log(extData);
    });
  }

  updateMarkers = (data) => {
    this.props.updateMarker(data);
  }

  render() {
    const { markers, ...rest } = this.props;

    return (
      <Shuttle>
        <RiskPoint
          {...rest}
          markers={markers.riskSource}
          onCreated={(val) => this.onMarkersCreated('riskSource', val)}
        />
        <PatrolPoint
          {...rest}
          markers={markers.patrol}
          onCreated={(val) => this.onMarkersCreated('patrol', val)}
        />
        <PersonPoint3D
          {...rest}
          markers={markers.person}
          onCreated={(val) => this.onMarkersCreated('person', val)}
        />
        <EquipmentPoint
          {...rest}
          markers={markers.impEquipment}
          onCreated={(val) => this.onMarkersCreated('impEquipment', val)}
        />
        <DynamicRingDataPoint
          {...rest}
          markers={markers.dynamicRingData}
          onCreated={(val) => this.onMarkersCreated('dynamicRingData', val)}
        />
        <ProbePoint
          {...rest}
          markers={markers.monitorEquipment}
          onCreated={(val) => this.onMarkersCreated('monitorEquipment', val)}
        />
        <MonitorPoint
          {...rest}
          markers={markers.video}
          onCreated={(val) => this.onMarkersCreated('video', val)}
        />
        <PointStatistics
          {...rest}
          markers={markers.statistics}
          onCreated={(val) => this.onMarkersCreated('statistics', val)}
        />
      </Shuttle>
    );
  }
}

PointsPool.propTypes = {
  subscribe: PropTypes.func,
  updateMarker: PropTypes.func,
  markers: PropTypes.object
};

export default PointsPool;
