import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Form, Input, Radio, Select, Button, Toast, CopyToClipboard } from 'amos-framework';
// eslint-disable-next-line max-len
import { getRiskSoureData, updateRiskSourePosition3d, getPointData, updatePointPosition3d, getMonitorsByFloor3d, updateMonitorPosition3d } from 'SERVICES/pickService';

const FormItem = Form.Item;
const Option = Select.Option;
const RadioGroup = Radio.Group;

/**
 * 数据点模型
 *
 * @class PickModal
 * @extends {Component}
 */
class PickModal extends Component {
  constructor(props) {
    super(props);
    this.state = {
      datas: [],
      form: {
      },
      rules: {
      }
    };
  }

  componentDidMount(){

  }

  onInputChange = (key, value) => {
    // console.log(key, value);
    const { form } = this.state;
    form[key] = value;
    this.setState({ form });
  };

  onSelectChange = (value, item) => {
    // console.log(value, item);
    const newForm = Object.assign({}, this.state.form, { type: value, desc: item.children });
    this.setState({
      form: newForm
    });
    switch (value) {
      case 'riskSource':
        this.getRiskSoureData();
        break;
      case 'patrol':
        this.getPatrolData();
        break;
      case 'impEquipment':
        this.setState({ datas: [] });
        break;
      case 'dynamicRingData':
        this.setState({ datas: [] });
        break;
      case 'video':
        this.getMonitorData();
        break;
      default:
        console.log('不支持的类型');
    }
  }

  onRadioChange = (value) => {
    // console.log(value);
    const newForm = Object.assign({}, this.state.form, { id: value });
    this.setState({
      form: newForm
    });
  }

  /**
   * 获取风险点
   *
   * @memberof PickModal
   */
  getRiskSoureData = () => {
    const { orgCode } = this.props;
    const { form } = this.state;

    getRiskSoureData(orgCode, form.floor).then(data => {
      this.setState({ datas: data });
    });
  }

  /**
   * 获取巡检点
   *
   * @memberof PickModal
   */
  getPatrolData = () => {
    const { orgCode } = this.props;
    const { form } = this.state;

    getPointData(orgCode, form.floor).then(data => {
      this.setState({ datas: data });
    });
  }

  /**
   * 获取监控点
   *
   * @memberof PickModal
   */
  getMonitorData = () => {
    const { orgCode } = this.props;
    const { form } = this.state;

    getMonitorsByFloor3d(orgCode, form.floor).then(data => {
      this.setState({ datas: data });
    });
  }

  /**
   * 更新风险点三维坐标
   *
   * @memberof PickModal
   */
  updateRiskSourePosition3d = (pickedPosition) => {
    const { form } = this.state;
    const param = { id: form.id, position3d: pickedPosition.join(',') };
    updateRiskSourePosition3d(param).then(res => {
      Toast.success({ title: `${form.desc}三维坐标更新成功！` });
      this.props.onCancel();
    });
  }

  /**
   * 更新风险点坐标
   *
   * @memberof PickModal
   */
  updatePatrolPosition3d = (pickedPosition) => {
    const { form } = this.state;
    const param = Object.assign({}, form, { position3d: pickedPosition.join(',') });
    updatePointPosition3d(param.id, param.position3d).then(res => {
      Toast.success({ title: `${form.desc}三维坐标更新成功！` });
      this.props.onCancel();
    });
  }

  /**
   * 更新监控点坐标
   *
   * @memberof PickModal
   */
  updateMonitorPosition3d = (pickedPosition) => {
    const { form } = this.state;
    const param = Object.assign({}, form, { position3d: pickedPosition.join(',') });
    updateMonitorPosition3d(param).then(res => {
      Toast.success({ title: `${form.desc}三维坐标更新成功！` });
      this.props.onCancel();
    });
  }

  handleSubmit = e => {
    this.form.validate((valid, dataValues) => {
      if (valid) {
        let { form } = this.state;
        const { pickedPosition } = this.props;
        switch (form.type) {
          case 'riskSource':
            this.updateRiskSourePosition3d(pickedPosition);
            break;
          case 'patrol':
            this.updatePatrolPosition3d(pickedPosition);
            break;
          case 'impEquipment':
            break;
          case 'dynamicRingData':
            break;
          case 'video':
            this.updateMonitorPosition3d(pickedPosition);
            break;
          default:
            console.log('不支持的类型');
        }
      } else {
        return false;
      }
    });
  };

  render() {
    const { pickedPosition, floorType } = this.props;
    // console.log(pickedPosition, floorType, orgCode);
    const { form, rules, datas } = this.state;
    form.floor = floorType === 'floor_3' ? 3 : floorType === 'floor_6' ? 6 : 0;
    let floorName = form.floor === 3 ? '三楼' : form.floor === 6 ? '六楼' : '机柜';

    const formItemLayout = {
      labelCol: {
        xs: { span: 24 },
        sm: { span: 5 },
        className: 'colspanlab'
      },
      wrapperCol: {
        xs: { span: 24 },
        sm: { span: 17 },
        className: 'colspan'
      }
    };

    return (
      <Form model={form} rules={rules} ref={component => this.form = component} className="pick-dialog-body">
        <FormItem label={<span>三维坐标</span>} field="3dPosition" {...formItemLayout}>
          <Input className="full-input" value={pickedPosition} readOnly />&nbsp;
          <CopyToClipboard style={{ marginTop: 10, display: 'inline-block' }} text={pickedPosition} successTip="复制成功！"><Button size="sm">点击复制坐标</Button></CopyToClipboard>
        </FormItem>
        <FormItem label={<span>三维楼层</span>} field="floor" {...formItemLayout}>
          <Input className="full-input" value={floorName} readOnly />
        </FormItem>
        <FormItem label={<span>三维点类型</span>} field="type" {...formItemLayout}>
          <Select className="full-input" value={form.type} onChange={this.onSelectChange}>
            {<Option value="riskSource">风险点</Option>}
            {<Option value="patrol">巡检点</Option>}
            {/* <Option value="impEquipment">网络设备</Option> */}
            {/* {<Option value="dynamicRingData">动环数据</Option>} */}
            {<Option value="video">视频监控</Option>}
          </Select>
        </FormItem>
        <FormItem label={form.desc} field="data" {...formItemLayout}>
          <RadioGroup onChange={this.onRadioChange}>
            { datas.map(item => {
              return (
                <Radio key={item.token} value={item.id}>{item.name}</Radio>
              );
            })}
          </RadioGroup>
        </FormItem>
      </Form>
    );
  }
}

PickModal.propTypes = {
  orgCode: PropTypes.string,
  floorType: PropTypes.string,
  pickedPosition: PropTypes.array,
  onCancel: PropTypes.func
};

export default PickModal;
