import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Connect } from 'amos-framework';
import { connect3D } from 'amos-3d/lib/designer';
import { RiskPoint, EquipmentPoint, DangerPoint, MajorHazardPoint,TaskworkPoint } from './points';
import { eventTopics, isPointEvent , isLevelFilter } from './consts';
import { parseMarkers, parseLevelFilter } from './dataProcessor';

const eventConnect = Connect.eventConnect;
const Shuttle = (props) => props.children;

/**
 * Markers 池，所有的 markers 均在本类进行加载
 *
 * @export
 * @class PointsPool
 * @extends {Component}
 */
@connect3D
@eventConnect
class PointsPool extends Component {

  constructor(props) {
    super(props);
    this.markerList = {};
  }

  componentDidMount() {
    this.props.subscribe(eventTopics.base3d_view, (topic, data) => {
      if (isPointEvent(topic)){
        parseMarkers(this, topic, data);
      } else if (isLevelFilter(topic)) {
        parseLevelFilter(this, topic, data);
      }
    });
  }

  onMarkersCreated = (type, { markersCache }) => {
    (markersCache || []).forEach(mc => {
      const extData = mc.getExtData();
      if (extData.hasOwnProperty('visible')){
        let visible = extData.visible;
        if (visible) {
          mc.show();
        } else {
          mc.hide();
        }
      }
    });
    this.markerList[type] = markersCache;
  }

  updateMarkers = (data) => {
    this.props.updateMarker(data);
  }

  render() {
    const { markers, ...rest } = this.props;
    return (
      <Shuttle>
        <RiskPoint
          {...rest}
          markers={markers.riskSource}
          onCreated={(val) => this.onMarkersCreated('riskSource', val)}
        />
        <EquipmentPoint
          {...rest}
          markers={markers.equipment}
          onCreated={(val) => this.onMarkersCreated('equipment', val)}
        />
        <DangerPoint
          {...rest}
          markers={markers.danger}
          onCreated={(val) => this.onMarkersCreated('danger', val)}
        />
        <MajorHazardPoint
          {...rest}
          markers={markers.majorHazard}
          onCreated={(val) => this.onMarkersCreated('majorHazard', val)}
        />
        <TaskworkPoint
          {...rest}
          markers={markers.taskWork}
          onCreated={(val) => this.onMarkersCreated('taskWork', val)}
        />
      </Shuttle>
    );
  }
}

PointsPool.propTypes = {
  subscribe: PropTypes.func,
  updateMarker: PropTypes.func,
  markers: PropTypes.object
};

export default PointsPool;
