import React, { Component } from 'react';
import { Form, InputNumber, Input, Select, Radio } from 'amos-framework';
import { DatePicker } from 'amos-antd';
import moment from 'moment';
import PropTypes from 'amos-react-router/lib/PropTypes';
import { getFireEquipmentAction } from '../../../../../services/ledgerService';

const FormItem = Form.Item;
const Option = Select.Option;
const RadioGroup  = Radio.Group;
const TextArea = Input.TextArea;

const floorData = [{ id: '1', name: '1层' }, { id: '2', name: '2层' }, { id: '3', name: '3层' }, { id: '4', name: '4层' }, { id: '5', name: '5层' }];

/**
 * 消防设备新增及编辑
 */
class FireEquipmentModel extends Component {
  constructor(props) {
    super(props);
    this.state = {
      searchParam: [],
      form: {
        name: '',
        code: '',
        brand: '',
        effectiveDate: moment().format('YYYY-MM-DD'),
        maintenanceCycle: '',
        equipClassify: '',
        manufacturer: '',
        model: '',
        productionArea: '',
        number: 1,
        productionDate: moment().format('YYYY-MM-DD'),
        equipCode: '',
        equipType: '',
        unit: '',
        floor3d: '1',
        position3d: '',
        room: '',
        remark: '',
        isIndoor: true
      },
      _disabled: true,
      chargeDeptData: [],
      chargeUserData: [],
      fireStationData: [],
      rules: {
        code: [{ required: true, message: '编码不能为空' }],
        name: [{ required: true, message: '名称不能为空' }],
        equipType: [{ required: true, message: '装备类型名称不能为空' }],
        position3d: [{ validator: this.validPositon3d }]
      }
    };
  }

  componentWillMount = () => {
    const { fireEquipmentId, operateFlag } = this.props || {};
    if (operateFlag === 1) {  //编辑
      this.getFireEquipmentData(fireEquipmentId);
    }
  };

  onInputChange = (key, value) => {
    const { form } = this.state;
    form[key] = value;
    this.setState({ form });
  };

  onSelectChange(key, value) {
    const { form } = this.state;
    form[key] = value;
    this.setState({ form });
    if (value === 1) {
      this.setState({ _disabled: false });
    } else {
      this.setState({ _disabled: true });
    }
  }

  onDateChange = (key, date) => {
    const { form } = this.state;
    form[key] = date ? moment(date).format('YYYY-MM-DD') : null;
    this.setState({ form });
  };

  getFireEquipmentData = (fireEquipmentId) => {
    getFireEquipmentAction(fireEquipmentId).then(data => {
      this.setState({ form: data });
    });
  }

  validPositon3d = (rule, value, callback) => {
    let tempArry = value.split(',') || [];
    if ( tempArry && tempArry.length === 3 ){
      const patt = /(^([-]?)[1-9]([0-9]+)?(\.[0-9]+)?$)|(^([-]?)(0){1}$)|(^([-]?)[0-9]\.[0-9]+?$)/;
      let i = 0;
      for (let key of tempArry){
        if (!patt.test(key)){
          i = i + 1;
          callback('非法格式');
          break;
        }
      }
      i === 0 ? callback() : '';
    } else if (!value) {
      callback();
    } else {
      callback('非法格式');
    }
  }

  render() {
    const { form, rules } = this.state;
    let { _disabled } = this.state;
    const formItemLayout = {
      labelCol: {
        xs: { span: 24 },
        sm: { span: 7 },
        className: 'colspanlab'
      },
      wrapperCol: {
        xs: { span: 24 },
        sm: { span: 14 },
        className: 'colspan'
      }
    };

    if (form && form.equipClassify === 1) {
      _disabled = false;
    }
    return (
      <div>
        <Form model={form} rules={rules} ref={component => this.form = component}>
          <div className='important-equipment-edit-left'>
            <FormItem label={<span>装备编号</span>} field="code" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.code} onChange={e => this.onInputChange('code', e.target.value)} />
            </FormItem>
            <FormItem label={<span>装备分类</span>} field="equipClassify" {...formItemLayout}>
              <Select className="risk_factor_select" value={form.equipClassify} onChange={e => this.onSelectChange('equipClassify', e)} closeOnScroll>
                <Option value={0}>设备类</Option>
                <Option value={1}>耗材类</Option>
                <Option value={3}>灭火器材</Option>
              </Select>
            </FormItem>
            <FormItem label={<span>装备编码</span>} field="equipCode" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.equipCode} onChange={e => this.onInputChange('equipCode', e.target.value)} />
            </FormItem>
            <FormItem label={<span>数量</span>} field="number" {...formItemLayout} >
              <InputNumber className="risk_factor_input" required value={form.number} onChange={value => this.onInputChange('number', value)} defaultValue={1} disabled={_disabled} min={1} />
            </FormItem>
            <FormItem label={<span>品牌</span>} field="brand" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.brand} onChange={e => this.onInputChange('brand', e.target.value)} />
            </FormItem>
            <FormItem label={<span>生产日期</span>} field="productionDate" {...formItemLayout}>
              <DatePicker
                value={form.productionDate ? moment(form.productionDate) : ''}
                allowClear
                className="risk_factor_input"
                defaultValue={moment()}
                format="YYYY-MM-DD"
                onChange={e => this.onDateChange('productionDate', e)}
              />
            </FormItem>
            <FormItem label={<span>保养周期</span>} field="maintenanceCycle" {...formItemLayout}>
              <InputNumber className="risk_factor_input" required value={form.maintenanceCycle} onChange={value => this.onInputChange('maintenanceCycle', value)} />
            </FormItem>
            <FormItem label={<span>3维坐标</span>} field="position3d" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.position3d} onChange={e => this.onInputChange('position3d', e.target.value)} />
            </FormItem>
            <FormItem label={<span>3维楼层</span>} field="floor3d" {...formItemLayout}>
              <Select
                className="fire-equip-select"
                data={floorData}
                defaultValue='1'
                value={form.floor3d}
                renderOption={item => <Option value={item.id}>{item.name}</Option>}
                onChange={e => this.onInputChange('floor3d', e)}
              />
            </FormItem>
            <FormItem label={<span>是否室内</span>} field="isIndoor" {...formItemLayout}>
              <RadioGroup defaultValue={form.isIndoor} value={form.isIndoor} onChange={e => this.onInputChange('isIndoor', e)}>
                <Radio value={false}>否</Radio>
                <Radio value={true}>是</Radio>
              </RadioGroup>
            </FormItem>
          </div>
          <div className='important-equipment-edit-right'>
            <FormItem label={<span>装备名称</span>} field="name" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.name} onChange={e => this.onInputChange('name', e.target.value)} />
            </FormItem>
            <FormItem label={<span>类型名称</span>} field="equipType" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.equipType} onChange={e => this.onInputChange('equipType', e.target.value)} />
            </FormItem>
            <FormItem label={<span>规格型号</span>} field="model" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.model} onChange={e => this.onInputChange('model', e.target.value)} />
            </FormItem>
            <FormItem label={<span>单位</span>} field="unit" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.unit} onChange={e => this.onInputChange('unit', e.target.value)} />
            </FormItem>
            <FormItem label={<span>厂商</span>} field="manufacturer" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.manufacturer} onChange={e => this.onInputChange('manufacturer', e.target.value)} />
            </FormItem>
            <FormItem label={<span>有效日期</span>} field="effectiveDate" {...formItemLayout}>
              <DatePicker
                value={form.effectiveDate ? moment(form.effectiveDate) : ''}
                allowClear
                className="risk_factor_input"
                defaultValue={moment()}
                format="YYYY-MM-DD"
                onChange={e => this.onDateChange('effectiveDate', e)}
              />
            </FormItem>
            <FormItem label={<span>区域</span>} field="productionArea" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.productionArea} onChange={e => this.onInputChange('productionArea', e.target.value)} />
            </FormItem>
            <FormItem label={<span>房间号</span>} field="room" {...formItemLayout}>
              <Input className="risk_factor_input" required value={form.room} onChange={e => this.onInputChange('room', e.target.value)} />
            </FormItem>
            <FormItem label={<span>备注</span>} field="remark" {...formItemLayout}>
              <TextArea rows={4} className="risk_factor_input" required value={form.remark} onChange={e => this.onInputChange('remark', e.target.value)} />
            </FormItem>
          </div>
        </Form>
      </div>
    );
  }
}

FireEquipmentModel.propTypes = {
  equipmentId: PropTypes.number,
  toolbarFlag: PropTypes.number
};
export default FireEquipmentModel;
