import React, { Component } from 'react';
import PropTypes from 'prop-types';
import moment from 'moment';
import { Modal, AmosAlert } from 'amos-framework';
import { browserHistory } from 'amos-react-router';
import AmosGridTable from './../../../common/tableComponent/table/AmosGridTable';
import { evaluateModelDeleteAction } from '../../../../../services/bizService';
import EvaluationModel from './EvaluationModel';
import { pathMapping } from './../../../../../routes/customRoutes';

const mgtCtrlModelUrl = pathMapping.mgtCtrlModel;
const activeKeyMap = { 'level': '3', 'criteria': '4' };
const getColumns = (self) => {
  return [
    {
      title: '序号',
      render: (text, record, index) => `${index + 1}`
    },
    {
      title: '风险评价方法',
      dataIndex: 'name',
      key: 'name',
      width: '20%',
      className: 'uant-table-thead'
    },
    {
      title: '风险等级',
      dataIndex: 'level',
      key: 'level',
      width: '15%',
      className: 'uant-table-thead',
      render: (text, record, index) => {
        return (
          <span>
            <a href="javascript:;" onClick={() => {self.onDetailClick(record, 'level');}}>查看</a>
          </span>
        );
      }
    },
    {
      title: '风险评价准则',
      dataIndex: 'criteria',
      key: 'criteria',
      width: '15%',
      className: 'uant-table-thead',
      render: (text, record, index) => {
        return (
          <span>
            <a href="javascript:;" onClick={() => {self.onDetailClick(record, 'criteria');}}>查看</a>
          </span>
        );
      }
    },
    {
      title: '风险管控模型',
      dataIndex: 'ctrlModel',
      key: 'ctrlModel',
      width: '15%',
      className: 'uant-table-thead',
      render: (text, record, index) => {
        return (
          <span>
            <a href="javascript:;" onClick={() => {self.onDetailClick(record, 'ctrlModel');}}>查看</a>
          </span>
        );
      }
    },
    {
      title: '维护人员',
      dataIndex: 'createUserName',
      key: 'createUserName',
      width: '10%',
      className: 'uant-table-thead'
    },
    {
      title: '维护日期',
      dataIndex: 'maintenanceDate',
      key: 'maintenanceDate',
      width: '15%',
      render: text => moment(text).format('YYYY-MM-DD HH:mm:ss'),
      className: 'uant-table-thead'
    }
  ];
};

/**
 * 风险评价模型
 *
 * @class RiskEvaluationModel
 * @extends {Component}
 */
class RiskEvaluationModel extends Component {
  constructor(props) {
    super(props);
    this.pageConfig = {
      current: 1,
      pageSize: 10
    };
    this.state = {
      pagination: true,
      isChecked: true,
      modal: false,
      selectedRows: [],
      selectedRowKeys: [],
      data: [],
      searchParam: []
    };
  }

  componentWillReceiveProps(newProps) {
    let { toolBarFlag, activeKey, initToolbarFlag } = newProps;
    if (activeKey && activeKey === '2' && toolBarFlag) {
      this.checkData(toolBarFlag);
      initToolbarFlag();
    }
  }

  onDetailClick = (record, key) => {
    console.log(key);
    if (key === 'ctrlModel') {//跳转到管控模型页面
      let path = {
        pathname: mgtCtrlModelUrl,
        state: record
      };
      browserHistory.push(path);
    }
    let activeKey = activeKeyMap[key];
    this.props.onTabChange(record, activeKey);
  }

  checkData = toolBarFlag => {
    let { selectedRowKeys } = this.state;
    if (toolBarFlag === '1') {
      //新增
      this.setState({ modal: true, toolBarFlag });
    } else if (toolBarFlag === '2') {
      //编辑
      if (selectedRowKeys === undefined || selectedRowKeys.length <= 0) {
        AmosAlert.warning('提示', '请先点击选择编辑的行');
        return;
      } else {
        this.setState({ modal: true, toolBarFlag });
      }
    } else if (toolBarFlag === '3') {
      //删除
      if (selectedRowKeys === undefined || selectedRowKeys.length <= 0) {
        AmosAlert.warning('提示', '请先点击选择删除的行');
        return;
      } else {
        this.evaluateModelDelete(selectedRowKeys);
      }
    }
  };

  /**
   * 分页设置参数
   */
  setPageConfig = ({ pageSize, current }) => {
    if (pageSize !== undefined) {
      this.pageConfig.pageSize = pageSize;
    }
    if (current !== undefined) {
      this.pageConfig.pageNumber = current;
    }
  };

  /**
   * 获取表格所选则的行数据
   */
  getSelectedRows = (selectedRows, selectedRowKeys) => {
    this.setState({ selectedRows, selectedRowKeys });
  };

  getRowClassName = (record, index) => {
    let { selectedRowKeys } = this.state;
    if (selectedRowKeys.findIndex(item => item === record.id) >= 0) {
      return 'highlight-row';
    } else {
      return 'normal-row';
    }
  };

  onRowClick = record => {
    let { selectedRowKeys, selectedRows } = this.state;
    let exist = false;
    selectedRows.find(item => {
      if (item.id === record.id) {
        exist = true;
      }
    });
    if (exist) {
      selectedRows.splice(selectedRows.findIndex(item => item.id === record.id), 1);
      selectedRowKeys.splice(selectedRowKeys.findIndex(item => item === record.id), 1);
    } else {
      selectedRows.push(record);
      selectedRowKeys.push(record.id);
    }
    this.setState({
      selectedRows,
      selectedRowKeys
    });
  };

  /**
   * 获取表格刷新方法
   */
  reload = r => {
    this.setState(
      {
        reload: () => {
          r();
          this.setState({ selectedRows: [], selectedRowKeys: [] });
        }
      },
      r()
    );
  };

  clearSelectedRows = () => {
    this.setState({ selectedRows: [], selectedRowKeys: [], modal: false, toolBarFlag: '' });
  };

  reloadPage = () => {
    this.clearSelectedRows();
    this.state.reload();
  };

  getContext = toolBarFlag => {
    if (toolBarFlag) {
      if (toolBarFlag === '1') {
        //新增
        return <EvaluationModel reloadPage={this.reloadPage} ref={component => this.model = component} />;
      } else {
        const { selectedRows } = this.state;
        return <EvaluationModel reloadPage={this.reloadPage} selectRowData={selectedRows[0]} ref={component => this.model = component} />;
      }
    }
  };

  cancel = () => {
    this.setState({ modal: false });
  };

  evaluateModelDelete = ids => {
    AmosAlert.confirm('提示', '确定要删除么?', {
      callback: flag => {
        if (flag) {
          // 确定
          evaluateModelDeleteAction(ids).then(
            data => {
              AmosAlert.success('提示', '删除成功');
              this.reloadPage();
            },
            err => {
              AmosAlert.error('错误', err);
            }
          );
        }
      }
    });
  };

  onOk = () => {
    this.model.handleSubmit();
  };

  render() {
    let { pagination, isChecked, selectedRowKeys, modal, toolBarFlag } = this.state;
    let { dataList, totalCount, fetchData } = this.props;
    let defaultPageConfig = this.pageConfig;
    return (
      <div className="patrol-page-table">
        <AmosGridTable
          columns={getColumns(this)}
          callBack={this.reload}
          isPageable={pagination}
          dataList={dataList}
          totals={totalCount}
          getTableDataAction={param => fetchData(param, this)}
          selectedRowKeys={selectedRowKeys}
          getSelectedRows={this.getSelectedRows}
          rowClassName={this.getRowClassName}
          setPageConfig={this.setPageConfig}
          defaultPageConfig={defaultPageConfig}
          onRowClick={this.onRowClick}
          isChecked={isChecked}
        />
        <Modal
          header={toolBarFlag === '1' ? '添加风险评价模型' : '编辑风险评价模型'}
          visible={modal}
          destroyContent
          className="param-model"
          content={this.getContext(toolBarFlag)}
          onCancel={() => this.cancel()}
          onOk={() => this.onOk()}
          okText="确定"
          cancelText="取消"
          outterClosable={false}
        />
      </div>
    );
  }
}

RiskEvaluationModel.propTypes = {
  dataList: PropTypes.array,
  totalCount: PropTypes.number,
  fetchData: PropTypes.func,
  onTabChange: PropTypes.func
};
export default RiskEvaluationModel;
