import React, { Component } from 'react';
import PropTypes from 'prop-types';
import moment from 'moment';
import { Form, AmosAlert, Input, InputNumber, Radio, Select } from 'amos-framework';
import { riskLevelSaveAction } from '../../../../../services/bizService';

const FormItem = Form.Item;
const TextArea = Input.TextArea;
const Option = Select.Option;
const RadioGroup = Radio.Group;

const riskLevelData = [{ id: 1, name: '1级' }, { id: 2,name: '2级' },{ id: 3, name: '3级' },{ id: 4,name: '4级' },{ id: 5,name: '5级' }];
const colorEum = { 1: '红色', 2: '橙色', 3: '黄色', 4: '蓝色', 5: '蓝色' };

/**
 * 风险等级编辑
 */
class RiskLevelModel extends Component {
  constructor(props) {
    super(props);
    this.state = {
      searchParam: [],
      form: {
        level: '',
        code: '',
        upper: '',
        lower: '',
        describe: '',
        isAcceptable: 1
      },
      rules: {
        code: [{ required: true, message: '风险级等级不能为空' }],
        upper: [{ required: true, validator: this.levelValueValidator }],
        lower: [{ required: true, validator: this.levelValueValidator }],
        describe: [{ validator: this.describeLengthValidator }]
      }
    };
  }

  componentWillMount = () => {
    const { selectRowData } = this.props || {};
    if (selectRowData) {
      let form = {};
      Object.assign(form, selectRowData);
      this.setState({ form });
    }
  };

  onSelectChange = (key, value,item) => {
    const { form } = this.state;
    form[key] = value;
    if (key === 'code') {
      form.level = item.name;
      form.code = item.id;
    }
    this.setState({ form });
  };

  onInputChange = (key, value) => {
    const { form } = this.state;
    form[key] = value;
    this.setState({ form });
  };

  onRadioChange = (value) => {
    const { form } = this.state;
    form['isAcceptable'] = value;
    this.setState({ form });
  }

  onDateChange = (key, date) => {
    const { form } = this.state;
    form[key] = date ? moment(date).format('YYYY-MM-DD HH:mm:ss') : null;
    this.setState({ form });
  };

  handleSubmit = evaModelId => {
    this.form.validate((valid, dataValues) => {
      if (valid) {
        let { form } = this.state;
        form.evaluateModelId = evaModelId;
        this.riskLevelEdit(form);
      } else {
        return false;
      }
    });
  };

  riskLevelEdit = body => {
    riskLevelSaveAction(body).then(
      data => {
        AmosAlert.success('提示', '操作成功');
        this.props.reloadPage();
      },
      err => {
        AmosAlert.error('提示', err);
      }
    );
  };

  levelValueValidator = (rule, value, callback) => {
    let { form } = this.state;
    if (form.lower && form.lower > value) {
      callback('最大值不能小于最小值');
    } else if (form.upper && form.upper < value) {
      callback('最小值不能大于最大值');
    } else if (rule.field === 'upper' && !form.upper) {
      callback('风险度上限不能为空');
    } else if (rule.field === 'lower' && (form.lower === 0 ? false : !form.lower)) {
      callback('风险度下限不能为空');
    } else if ((form.upper||form.lower) && value.toString().length>6) {
      callback('最小值或者最大值不能超过6位数');
    } else {
      callback();
    }
  }

  describeLengthValidator = (rule, value, callback) => {
    let { form } = this.state;
    if (form.describe.length > 50) {
      callback('输入长度不能大于50字符');
    } else {
      callback();
    }
  }

  render() {
    const { form, rules } = this.state;
    const formItemLayout = {
      labelCol: {
        xs: { span: 24 },
        sm: { span: 7 },
        className: 'colspanlab'
      },
      wrapperCol: {
        xs: { span: 24 },
        sm: { span: 14 },
        className: 'colspan'
      }
    };

    return (
      <div>
        <Form model={form} rules={rules} ref={component => this.form = component}>
          <FormItem label={<span>风险等级</span>} field="code" {...formItemLayout}>
            <Select
              className="risk_factor_input"
              data={riskLevelData}
              renderOption={item => <Option value={item.id}>{item.name}</Option>}
              value={parseInt(form.code)}
              onChange={(value,item)=> this.onSelectChange('code', value,item)}
            />
          </FormItem>
          <FormItem label={<span>风险度上限</span>} field="upper" {...formItemLayout}>
            <InputNumber className='risk_factor_input' value={form.upper} onChange={value => this.onInputChange('upper', value)} />
          </FormItem>
          <FormItem label={<span>风险度下限</span>} field="lower" {...formItemLayout}>
            <InputNumber className='risk_factor_input' value={form.lower} onChange={value => this.onInputChange('lower', value)} />
          </FormItem>
          <FormItem label={<span>是否可接受</span>} field="isAcceptable" {...formItemLayout}>
            <RadioGroup defaultValue={1} value={form.isAcceptable} onChange={value => this.onRadioChange(value)}>
              <Radio value={0}>否</Radio>
              <Radio value={1}>是</Radio>
            </RadioGroup>
          </FormItem>
          <FormItem label={<span>影响描述</span>} field="describe" {...formItemLayout}>
            <TextArea rows={10} cols={20} className="risk_factor_input" value={form.describe} onChange={e => this.onSelectChange('describe', e.target.value)} />
          </FormItem>
        </Form>
      </div>
    );
  }
}
RiskLevelModel.propTypes = {
  reloadPage: PropTypes.func
};
export default RiskLevelModel;
