import React, { Component } from 'react';
import PropTypes from 'prop-types';
import moment from 'moment';
import { Modal, AmosAlert } from 'amos-framework';
import AmosGridTable from './../../../common/tableComponent/table/AmosGridTable';
import { riskLevelDeleteAction } from '../../../../../services/bizService';
import RiskLevelModel from './RiskLevelModel';

const isAcceptableEnum = { 0: '否', 1: '是' };

const getColumns = () => {
  return [
    {
      title: '序号',
      render: (text, record, index) => `${index + 1}`
    },
    {
      title: '风险等级',
      dataIndex: 'level',
      key: 'level',
      width: '10%',
      className: 'uant-table-thead'
    },
    {
      title: '风险度上限',
      dataIndex: 'upper',
      key: 'upper',
      width: '8%',
      className: 'uant-table-thead'
    },
    {
      title: '风险度下限',
      dataIndex: 'lower',
      key: 'lower',
      width: '8%',
      className: 'uant-table-thead'
    },
    {
      title: '影响描述',
      dataIndex: 'describe',
      key: 'describe',
      width: '36%',
      className: 'uant-table-thead'
    },
    {
      title: '是否可接受',
      dataIndex: 'isAcceptable',
      key: 'isAcceptable',
      width: '8%',
      className: 'uant-table-thead',
      render: (text) => isAcceptableEnum[text]
    },
    {
      title: '维护人员',
      dataIndex: 'createUserName',
      key: 'createUserName',
      width: '10%',
      className: 'uant-table-thead'
    },
    {
      title: '维护日期',
      dataIndex: 'updateDate',
      key: 'updateDate',
      width: '10%',
      render: text => moment(text).format('YYYY-MM-DD HH:mm:ss'),
      className: 'uant-table-thead'
    }
  ];
};


/**
 * 风险等级
 */
class RiskLevel extends Component {
  constructor(props) {
    super(props);
    this.pageConfig = {
      current: 1,
      pageSize: 10
    };
    this.state = {
      pagination: true,
      isChecked: true,
      modal: false,
      selectedRows: [],
      selectedRowKeys: [],
      data: [],
      searchParam: [],
      evaluateModel: {}  //风险评价模型
    };
  }

  componentWillMount() {}

  componentWillReceiveProps(newProps) {
    let { toolBarFlag, activeKey, evaModel, initToolbarFlag } = newProps;
    if (activeKey && activeKey === '3' && toolBarFlag) {
      this.checkData(toolBarFlag);
      initToolbarFlag();
    }
    //切换风险评价模型 重置selectRow
    if(this.props.evaModel && this.props.evaModel.id != evaModel.id){ 
      this.clearSelectedRows();
    }
    this.setState({ evaluateModel: evaModel });
  }

  checkData = toolBarFlag => {
    let { selectedRowKeys } = this.state;
    if (toolBarFlag === '1') {
      //新增
      this.setState({ modal: true, toolBarFlag });
    } else if (toolBarFlag === '2') {
      //编辑
      if (selectedRowKeys === undefined || selectedRowKeys.length <= 0) {
        AmosAlert.warning('提示', '请先点击选择编辑的行');
        return;
      } else {
        this.setState({ modal: true, toolBarFlag });
      }
    } else if (toolBarFlag === '3') {
      //删除
      if (selectedRowKeys === undefined || selectedRowKeys.length <= 0) {
        AmosAlert.warning('提示', '请先点击选择删除的行');
        return;
      } else {
        this.riskLevelDelete(selectedRowKeys);
      }
    }
  };

  /**
   * 分页设置参数
   */
  setPageConfig = ({ pageSize, current }) => {
    if (pageSize !== undefined) {
      this.pageConfig.pageSize = pageSize;
    }
    if (current !== undefined) {
      this.pageConfig.pageNumber = current;
    }
  };

  /**
   * 获取表格所选则的行数据
   */
  getSelectedRows = (selectedRows, selectedRowKeys) => {
    this.setState({ selectedRows, selectedRowKeys });
  };

  getRowClassName = (record, index) => {
    let { selectedRowKeys } = this.state;
    if (selectedRowKeys.findIndex(item => item === record.id) >= 0) {
      return 'highlight-row';
    } else {
      return 'normal-row';
    }
  };

  onRowClick = record => {
    let { selectedRowKeys, selectedRows } = this.state;
    let exist = false;
    selectedRows.find(item => {
      if (item.id === record.id) {
        exist = true;
      }
    });
    if (exist) {
      selectedRows.splice(selectedRows.findIndex(item => item.id === record.id), 1);
      selectedRowKeys.splice(selectedRowKeys.findIndex(item => item === record.id), 1);
    } else {
      selectedRows.push(record);
      selectedRowKeys.push(record.id);
    }
    this.setState({
      selectedRows,
      selectedRowKeys
    });
  };

  /**
   * 获取表格刷新方法
   */
  reload = r => {
    this.setState(
      {
        reload: () => {
          r();
          this.setState({ selectedRows: [], selectedRowKeys: [] });
        }
      },
      r()
    );
  };

  clearSelectedRows = () => {
    this.setState({ selectedRows: [], selectedRowKeys: [], modal: false, toolBarFlag: '' });
  };

  reloadPage = () => {
    this.clearSelectedRows();
    this.state.reload();
  };

  getContext = toolBarFlag => {
    if (toolBarFlag) {
      if (toolBarFlag === '1') {
        //新增
        return <RiskLevelModel reloadPage={this.reloadPage} ref={component => this.model = component} />;
      } else {
        const { selectedRows } = this.state;
        return <RiskLevelModel selectRowData={selectedRows[0]} reloadPage={this.reloadPage} ref={component => this.model = component} />;
      }
    }
  };

  cancel = () => {
    this.setState({ modal: false, toolBarFlag: '' });
  };

  onOk = () => {
    let { evaluateModel } = this.state;
    if (evaluateModel.id) {
      this.model.handleSubmit(evaluateModel.id);
    } else {
      AmosAlert.warning('提示','请选择风险评价模型后，再进行操作');
    }
  };

  riskLevelDelete = body => {
    AmosAlert.confirm('提示', '确定要删除么?', {
      callback: flag => {
        if (flag) {
          // 确定
          riskLevelDeleteAction(body).then(
            data => {
              AmosAlert.success('提示', '删除成功');
              this.reloadPage();
            },
            err => {
              AmosAlert.error('错误', err);
              this.reloadPage();
            }
          );
        }
      }
    });
  };

  render() {
    let { pagination, isChecked, selectedRowKeys, modal, toolBarFlag } = this.state;
    let { dataList, totalCount, fetchData } = this.props;
    let defaultPageConfig = this.pageConfig;
    return (
      <div className="patrol-page-table">
        <AmosGridTable
          columns={getColumns()}
          callBack={this.reload}
          isPageable={pagination}
          dataList={dataList}
          totals={totalCount}
          getTableDataAction={param => fetchData(param, this)}
          selectedRowKeys={selectedRowKeys}
          getSelectedRows={this.getSelectedRows}
          rowClassName={this.getRowClassName}
          setPageConfig={this.setPageConfig}
          defaultPageConfig={defaultPageConfig}
          onRowClick={this.onRowClick}
          isChecked={isChecked}
        />
        <Modal
          header={toolBarFlag === '1' ? '添加风险等级' : '编辑风险等级'}
          visible={modal}
          className="param-model"
          destroyContent
          content={this.getContext(toolBarFlag)}
          onCancel={() => this.cancel()}
          onOk={() => this.onOk()}
          okText="确定"
          cancelText="取消"
          outterClosable={false}
        />
      </div>
    );
  }
}

RiskLevel.propTypes = {
  dataList: PropTypes.array,
  totalCount: PropTypes.number,
  fetchData: PropTypes.func
};
export default RiskLevel;
