import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Modal, message } from 'amos-framework';
import LibHeader from './LibHeader';
import LibList from './LibList';
import CreateLibForm from './CreateLibForm';
import TopoTree from './tree/RuleTree';

import {
  removeSubObjectAction,
  startUseSubObjectAction,
  startUseObjectAction,
  addSubObjectAction,
  updateSubObjectAction
} from './../../../services/ruleDesignService';

class RuleManageView extends Component {

  static propTypes = {
    onPushToRule: PropTypes.func.isRequired,
    ruleLib: PropTypes.array,
    reload: PropTypes.func,
    onSearchRuleLib: PropTypes.func
  };

  static defaultPrps = {
    ruleLib: [],
    reload(){}
  };

  constructor(props) {
    super(props);
    this.state = {
      visible: false,
      editObj: {},
      orgCode: 1
    };
    // 标识是编辑规则
    this.isEditRulelib = false;
  }

  /**
   * 点击添加
   */
  onCreateObj = () => {
    this.toggleEditOrCreate(false);
    this.setState({ visible: true });
  }

  /**
   * 点击设置
   */
  onSetting = (obj) => {
    this.toggleEditOrCreate(true);
    this.setState({
      visible: true,
      editObj: obj
    });
  }

  /**
   * 删除单个规则
   *
   * @memberof RuleManageView
   */
  onRemove = (ruleLib) => {
    if (ruleLib.status === '2' || ruleLib.isUser){
      Modal.confirm({
        title: '操作提示',
        content: '此规则正在应用，确定要删除吗？',
        onOk: () => {
          this.deleteRuleLib(ruleLib);
        }
      });
    } else {
      this.deleteRuleLib(ruleLib);
    }
  }

  /**
   * 启用单个对象
   */
  onToggleUse = (obj, checked) => {
    startUseSubObjectAction(obj.id).then((data) =>{
      this.props.reload();
    });
  }

  /**
   * toggle 发布
   */
  onTogglePublish = (obj, checked) => {
    obj.status = checked === false ? '1' : '2';
    updateSubObjectAction(obj).then((data) =>{
      this.props.reload();
    });
  }

  /**
   * 启用所有对象
   */
  onStartUseAllSwitchChange = (checked) => {
    let { ruleLib } = this.props;
    if (ruleLib && ruleLib.length > 0) {
      startUseObjectAction(checked).then((data) =>{
        this.props.reload();
      });
    } else {
      message.danger('没有可操作的数据');
    }
  }

  /**
   * 保存添加或编辑
   */
  onSave = (obj, resetFunc) => {
    let { orgCode } = this.state;
    Object.assign(obj,{ orgCode });
    const isEdit = this.isEditRulelib;
    if (isEdit) {
      updateSubObjectAction(obj).then((data) =>{
        // resetFunc();
        this.props.reload();
        this.onCancel();
      }, (error) => {
        message.danger(error || '编辑失败');
      });
    } else {
      addSubObjectAction(obj).then((data) => {
        // resetFunc();
        this.props.reload();
        this.onCancel();
      }, (error) => {
        message.danger(error || '添加失败');
      });
    }
  }

  /**
   * 取消添加或编辑
   */
  onCancel = () => {
    this.setState({
      visible: false,
      editObj: {}
    });
  }

  /**
   *
   *
   * @memberof RuleLibrary
   */
  setCurrentSubviewId = (id) => {
    this.setState({ orgCode: id });
  }

  deleteRuleLib = (ruleLib) => {
    removeSubObjectAction(ruleLib.id).then((data) =>{
      this.props.reload();
    });
  }

  /**
   * 切换编辑或者创建
   * @param {boolean} flag true时为编辑rulelib，false代表是创建新的rulelib
   */
  toggleEditOrCreate = (flag = false) => {
    this.isEditRulelib = flag;
  }


  render() {
    const { visible, editObj } = this.state;
    const { ruleLib, onPushToRule, onSearchRuleLib } = this.props;
    let allChecked = ruleLib && (ruleLib.filter(e => e.isUser).length === ruleLib.length && ruleLib.length !== 0);

    let modalHeader = '新增规则';
    const libFormProps = {
      onSave: this.onSave,
      onCancel: this.onCancel
    };

    if (this.isEditRulelib){
      libFormProps.libObj = editObj;
      modalHeader = editObj.name;
    }

    return (
      <div className="rule-manage-root">
        <LibHeader
          title="规则库管理"
          startUseAllSwitchChange={this.onStartUseAllSwitchChange}
          checked={allChecked}
          onSearch={onSearchRuleLib}
        />
        <TopoTree setCurrentSubviewId={this.setCurrentSubviewId} {...this.props} />
        <LibList
          ruleLibList={ruleLib}
          toggleUse={this.onToggleUse}
          togglePublish={this.onTogglePublish}
          onDoubleClick={onPushToRule}
          onCreateObj={this.onCreateObj}
          onRemove={this.onRemove}
          onSetting={this.onSetting}
        />
        <Modal
          header={modalHeader}
          width={400}
          visible={visible}
          className="rule-lib-modal"
          onCancel={this.onCancel}
          noDefaultFooter
          destroyContent
          content={<CreateLibForm cancelAction={this.onCancel} {...libFormProps} />}
        />
      </div>
    );
  }
}

export default RuleManageView;
