import React, { Component } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { parseText, utils } from 'amos-tool';
import enhancePortal from 'dt2react/lib/react/enhancePortal';
import { eleCatalogIconPrefix, isDragEle } from './../modelConf/stageConf';

const catalogList = {
  // object: { type: 'object', text: '对象' },
  status: { type: 'status', text: '状态' },
  trigger: { type: 'trigger', text: '触发器' },
  flowAction: { type: 'flowAction', text: '动作' },
  note: { type: 'note', text: '注释' },
  lineDashed: { type: 'lineDashed', text: '虚线' },
  lineSolid: { type: 'lineSolid', text: '实线' }
};

/**
 * 元素操作目录
 *
 * @class EleCatalogPanel
 * @extends {Component}
 */
class EleCatalogPanel extends Component {

  static propTypes = {
    catalogs: PropTypes.array,
    visible: PropTypes.bool,
    // 关闭流程模态框时，销毁内容
    destroyContent: PropTypes.bool,
    getContainer: PropTypes.func,
    position: PropTypes.object,
    sourceId: PropTypes.string,
    zIndex: PropTypes.number
  };

  static defaultProps = {
    destroyContent: false,
    getContainer: () => document.body
  };

  constructor(props) {
    super(props);
    this.state = {
      currentCatalogText: ''
    };
  }

  onCatalogItemClick = (text) => {
    this.setState({
      currentCatalogText: text
    });
  }

  dragStart = (eleType, sourceId, ev) => {
    const data = `${eleType}#${sourceId}`;
    // ev.dataTransfer.setData('FlowChartNode', eleType);
    ev.dataTransfer.setData('FlowChartNode', data);
  }

  renderItem = ({ type, icon, text }, sourceId) => {
    return (
      <div
        key={type}
        className="catalog-item"
        onClick={() => this.onCatalogItemClick(text)}
        onDragStart={(e) => this.dragStart(type, sourceId, e)}
      >
        <img src={icon} alt="" />
      </div>
    );
  }

  render() {

    const { currentCatalogText } = this.state;
    const { catalogs, visible, destroyContent, position, sourceId, zIndex } = this.props;

    const cls = classNames({
      'ele-catalog-portal': true,
      'ele-catalog-portal-open': visible,
      'ele-catalog-portal-closed': !visible
    });

    const catalogArr = [];
    catalogs.forEach(c => {
      const item = catalogList[c];
      if (item){
        item.icon = parseText(eleCatalogIconPrefix, { icon: item.type });
        catalogArr.push(catalogList[c]);
      }
    });

    const content = (
      <div className="ele-catalog-content">
        <div className="catalog-list">
          {catalogArr.map(c => this.renderItem(c, sourceId))}
        </div>
        <div className="catalog-explain">{currentCatalogText}</div>
      </div>
    );

    let style = {};

    if (position) {
      style.left = position.left || 0;
      style.top = position.top || 0;
    }

    if (!utils.isNil(zIndex)){
      style.zIndex = zIndex;
    }

    return (
      <div className={cls} style={style}>
        {visible || !destroyContent ? content : null}
      </div>
    );
  }
}

export default enhancePortal(EleCatalogPanel, {
  key: 'ele-catalog-modal'
});
