import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'amos-react-router';
import { utils } from 'amos-tool';
import Icon from 'amos-icon';
import { Icon as AFIcon } from 'amos-framework';
import classNames from 'classnames';
import SubMenuItem from './SubMenuItem';
import connectMenu from './connectMenu';

/**
 * 菜单 Item
 *
 * @class MenuItem
 * @extends {PureComponent}
 */
@connectMenu
class MenuItem extends PureComponent {
  constructor(props) {
    super(props);
  }

  isActive = () => {
    const { menuKey, activeKey } = this.props;
    return menuKey === activeKey;
  }

  genHiddenStyle = () => {
    const { height = 0 } = this.props;
    if (height > 0){
      return {};
    }
    return {
      display: 'none'
    };
  }

  genStyle = () => {
    const { menuWidth } = this.props;
    if (menuWidth){
      return {
        width: menuWidth
      };
    }
    return {};
  }

  renderItemLink = (item) => {
    const { iconPrefix } = this.props;
    const NavIcon = item.icon && <Icon icon={item.icon} prefix={iconPrefix} className="menu-item-icon" />;
    let Item;
    if (item.isBlank) {
      Item = (
        <a href={item.path} target="_blank" rel="noopener noreferrer" className="menu-item-link">
          {NavIcon}
          {item.permissionName}
        </a>
      );
    } else {
      Item = (
        <Link to={item.path} className="menu-item-link">
          {NavIcon}
          {item.permissionName}
        </Link>
      );
    }
    return Item;
  }

  renderSub = () => {
    const { menuKey, children, onSubMenuClick, activeSubKey, setSubMenuActive, renderSubmenu, menuWidth, iconPrefix } = this.props;
    const hiddenStyle = this.genHiddenStyle();
    return (
      <ul className="menu-sublist">
        {
          children.map(c => {
            const cls = classNames('menu-subitem', {
              'active': c.sequenceNbr === activeSubKey
            });
            if (renderSubmenu){
              return renderSubmenu({
                ...c,
                className: cls,
                onSubMenuClick,
                setSubMenuActive
              });
            }
            return (
              <SubMenuItem
                {...c}
                key={c.sequenceNbr}
                className={cls}
                iconPrefix={iconPrefix}
                menuKey={c.sequenceNbr}
                rootMenuKey={menuKey}
                subItems={c.children}
                onClick={() => onSubMenuClick(menuKey, c.sequenceNbr)}
                onSubMenuClick={onSubMenuClick}
                setMenuActive={setSubMenuActive}
                menuWidth={menuWidth}
              >
                {this.renderItemLink(c)}
                {c.children && c.children.length > 0 && <AFIcon icon="right" className="menu-subitem-icon" style={hiddenStyle} />}
              </SubMenuItem>
            );
          })
        }
      </ul>
    );
  }

  /** 渲染一个没有二级菜单的块，用于颜色分布 */
  renderNonSub = () => {
    return (
      <ul className="menu-sublist" />
    );
  }

  renderFooter = () => {
    const { footerIcon, extraIcon, icon, iconPrefix } = this.props;
    const hiddenStyle = this.genHiddenStyle();
    const _icon = footerIcon || extraIcon || icon;
    const imgContent = utils.isImageSrc(_icon) ? <img src={_icon} alt="" /> : <Icon prefix={iconPrefix} icon={_icon} />;
    return (
      <div className="nav-menu-item-footer" style={hiddenStyle}>
        <div className="footer-icon">
          {imgContent}
        </div>
        <span className="footer-line" />
      </div>
    );
  }

  render() {
    const { menuKey, permissionName, children, activeKey, height = 0, ...rest } = this.props;
    let itemContent;
    if (children){
      itemContent = <div className="nav-menu-item-label">{permissionName}</div>;
    } else {
      itemContent = <div className="nav-menu-item-label">{this.renderItemLink({ permissionName, ...rest })}</div>;
    }
    const cls = classNames('nav-menu-item', {
      'active': menuKey === activeKey
    });
    const style = this.genStyle();
    return (
      <li className={cls} style={style}>
        {itemContent}
        {children && <AFIcon icon="caretdown" className="nav-menu-item-icon" />}
        <div className="menusub-wrapper" style={{ height, ...style }}>
          {children && this.renderSub()}
          {!children && this.renderNonSub()}
          {this.renderFooter()}
        </div>
      </li>
    );
  }
}

MenuItem.propTypes = {
  height: PropTypes.number,
  menuKey: PropTypes.string,
  permissionName: PropTypes.string.isRequired,
  iconPrefix: PropTypes.string,
  icon: PropTypes.string,
  footerIcon: PropTypes.string,
  extraIcon: PropTypes.string,
  path: PropTypes.string,
  children: PropTypes.arrayOf(PropTypes.shape({
    sequenceNbr: PropTypes.string.isRequired,
    permissionName: PropTypes.string,
    icon: PropTypes.string,
    path: PropTypes.string,
    isBlank: PropTypes.bool
  })),
  activeKey: PropTypes.string,
  activeSubKey: PropTypes.string,
  setMenuActive: PropTypes.func,
  setSubMenuActive: PropTypes.func,
  onSubMenuClick: PropTypes.func,
  onMenuClosed: PropTypes.func,
  toggleSubMenu: PropTypes.func,
  renderSubmenu: PropTypes.func,
  menuWidth: PropTypes.oneOfType([
    PropTypes.string,
    PropTypes.number
  ])
};

export default MenuItem;
