package com.yeejoin.amos.boot.module.statistcs.biz.service.impl;

import cn.hutool.core.date.DateUtil;
import com.yeejoin.amos.boot.biz.common.dto.CountDto;
import com.yeejoin.amos.boot.biz.common.dto.KeyValueDto;
import com.yeejoin.amos.boot.module.common.api.dto.DPFilterParamDto;
import com.yeejoin.amos.boot.module.common.api.enums.IssueMainBodyEnum;
import com.yeejoin.amos.boot.module.common.api.enums.IssueTypeEnum;
import com.yeejoin.amos.boot.module.jg.api.enums.DPMapStatisticsItemEnum;
import com.yeejoin.amos.boot.module.statistics.api.dto.EquipQuestionNumCountDto;
import com.yeejoin.amos.boot.module.statistics.api.dto.SecurityIndexCountItemDto;
import com.yeejoin.amos.boot.module.statistics.api.mapper.AQZSDPStatisticsMapper;
import com.yeejoin.amos.boot.module.statistics.api.mapper.CylinderStatisticsMapper;
import com.yeejoin.amos.boot.module.ymt.api.enums.EquimentEnum;
import com.yeejoin.amos.boot.module.ymt.api.enums.EquipmentClassifityEnum;
import com.yeejoin.amos.boot.module.ymt.api.mapper.EquipTechParamPipelineMapper;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzBaseEnterpriseInfoMapper;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzsUserInfoMapper;
import com.yeejoin.amos.feign.systemctl.model.RegionModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.CountRequest;
import org.elasticsearch.client.core.CountResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 气瓶区域统计表服务实现类
 *
 * @author system_generator
 * @date 2022-03-08
 */
@Service
public class AQZSDPStatisticsServiceImpl {

    private final static String LICENSE_IS_OK = "正常";

    private final static String NO_DATA_STR = "--";

    /**
     * 单位类型-使用单位
     */
    private final static String COMPANY_TYPE_USE = "使用单位";

    /**
     * 单位类型-安装改造维修单位
     */
    private final static String COMPANY_TYPE_MAINTENANCE = "安装改造维修单位";

    /**
     * 单位类型-制造单位
     */
    private final static String COMPANY_TYPE_MANUFACTURE = "制造单位";

    /**
     * 单位类型-充装单位
     */
    private final static String COMPANY_TYPE_FILLING = "充装单位";


    private RestHighLevelClient restHighLevelClient;


    private AQZSDPStatisticsMapper statisticsMapper;


    private TzBaseEnterpriseInfoMapper enterpriseInfoMapper;


    private TzsUserInfoMapper userInfoMapper;

    private CylinderStatisticsMapper cylinderStatisticsMapper;


    private StCommonServiceImpl stCommonService;


    public AQZSDPStatisticsServiceImpl(RestHighLevelClient restHighLevelClient, AQZSDPStatisticsMapper statisticsMapper, TzBaseEnterpriseInfoMapper enterpriseInfoMapper, TzsUserInfoMapper userInfoMapper, CylinderStatisticsMapper cylinderStatisticsMapper, StCommonServiceImpl stCommonService) {
        this.restHighLevelClient = restHighLevelClient;
        this.statisticsMapper = statisticsMapper;
        this.enterpriseInfoMapper = enterpriseInfoMapper;
        this.userInfoMapper = userInfoMapper;
        this.cylinderStatisticsMapper = cylinderStatisticsMapper;
        this.stCommonService = stCommonService;
    }


    private String getLicenseEfficiencyByRegion(String orgCode) {
        List<KeyValueDto> keyValueDtos = statisticsMapper.queryLicenseEfficiencyOfRegion(orgCode);
        int totalUnitNumber = keyValueDtos.stream().mapToInt(KeyValueDto::getIValue).sum();
        int okUnitNumberNumber = keyValueDtos.stream().filter(k -> k.getStrKey().equals(LICENSE_IS_OK)).mapToInt(KeyValueDto::getIValue).sum();
        if (totalUnitNumber == 0) {
            return NO_DATA_STR;
        }
        BigDecimal bigTotalUnitNumber = new BigDecimal(String.valueOf(totalUnitNumber));
        BigDecimal bigOkUnitNumberNumber = new BigDecimal(String.valueOf(okUnitNumberNumber));
        BigDecimal result = bigOkUnitNumberNumber.divide(bigTotalUnitNumber, 4, RoundingMode.HALF_UP).multiply(new BigDecimal("100")).setScale(2,RoundingMode.HALF_UP);
        return result.toPlainString();
    }

    private long searchEsCount(Boolean checkOk, Boolean resultOk, String regionCode, String appId) {
        CountRequest countRequest = new CountRequest("cylinder_filling");
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        if (checkOk) {
            //匹配充装前检查和充装后检查都有数据
            BoolQueryBuilder meBuilder = QueryBuilders.boolQuery();
            meBuilder.must(QueryBuilders.existsQuery("inspectionDateMs"));
            meBuilder.must(QueryBuilders.existsQuery("inspectionDateAfterMS"));
            boolMust.must(meBuilder);
        }
        if (resultOk) {
            //匹配充装前检查和充装后检查都有数据
            BoolQueryBuilder meBuilder = QueryBuilders.boolQuery();
            meBuilder.must(QueryBuilders.termQuery("fillingResult.keyword", "合格"));
            meBuilder.must(QueryBuilders.termQuery("checkResult.keyword", "合格"));
            boolMust.must(meBuilder);
        }

        //匹配行政区划
        if (!ObjectUtils.isEmpty(regionCode)) {
            BoolQueryBuilder query = QueryBuilders.boolQuery();
            query.must(QueryBuilders.wildcardQuery("regionCode", "*" + regionCode + "*"));
            boolMust.must(query);
        }
        //匹配appId
        if (!ObjectUtils.isEmpty(appId)) {
            BoolQueryBuilder query = QueryBuilders.boolQuery();
            query.must(QueryBuilders.matchQuery("appId", appId));
            boolMust.must(query);
        }

        countRequest.query(boolMust);
        try {
            CountResponse countResponse = restHighLevelClient.count(countRequest, RequestOptions.DEFAULT);
            return countResponse.getCount();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Map<String, Object> getChildEarlyWarning(DPFilterParamDto dpFilterParamDto) {
        List<RegionModel> regionModels = stCommonService.setRegionIfRootParent(dpFilterParamDto.getCityCode());
        Map<String, SecurityIndexCountItemDto> regionCodeSecurityIndexMap = getSecurityIndexCountItemDtoMap(regionModels);
        Set<String> legendData = getLegendForCyArea();
        List<String> xuke = new ArrayList<>();
        List<String> shiyongdengji = new ArrayList<>();
        List<String> jianyanchaoqi = new ArrayList<>();
        List<String> jianyanhege = new ArrayList<>();
        List<String> czjc = new ArrayList<>();
        List<String> czjchege = new ArrayList<>();
        //TODO 多线程处理后，regionCodeSecurityIndexMap顺序与regionModels的顺序不一致，需让数据和x轴的顺序保存一致
        List<String> xdata = stCommonService.buildXData(regionModels);
        regionModels.forEach(r -> {
            SecurityIndexCountItemDto countItemDto = regionCodeSecurityIndexMap.get(r.getRegionCode().toString());
            xuke.add(this.castScore2Percent(countItemDto.getXuke()));
            shiyongdengji.add(this.castScore2Percent(countItemDto.getShiyongdengji()));
            jianyanchaoqi.add(this.castScore2Percent(countItemDto.getJianyanchaoqi()));
            jianyanhege.add(this.castScore2Percent(countItemDto.getJianyanhege()));
            czjc.add(this.castScore2Percent(countItemDto.getCzjc()));
            czjchege.add(this.castScore2Percent(countItemDto.getCzjchege()));
        });
        Map<String, Object> result = new HashMap<>();
        result.put("xdata", xdata);
        result.put("legendData", legendData);
        result.put("xuke", xuke);
        result.put("shiyongdengji", shiyongdengji);
        result.put("jianyanchaoqi", jianyanchaoqi);
        result.put("jianyanhege", jianyanhege);
        result.put("czjc", czjc);
        result.put("czjchege", czjchege);
        return result;
    }

    private String castScore2Percent(String score) {
        BigDecimal big = new BigDecimal(score);
        BigDecimal percent = big.divide(new BigDecimal("100"),2,RoundingMode.HALF_UP);
        return percent.toPlainString();
    }

    private Set<String> getLegendForCyArea() {
        Set<String> legendData = new HashSet<>();
        legendData.add("许可有效率");
        legendData.add("使用登记办理率");
        legendData.add("检验超期率");
        legendData.add("检验合格率");
        legendData.add("充装检查率");
        legendData.add("充装检查合格率");
        return legendData;
    }


    public Map<String, Object> getSecurityIndex(DPFilterParamDto dpFilterParamDto) throws Exception {
        List<RegionModel> regionModels = stCommonService.setRegionIfRootParent(dpFilterParamDto.getCityCode());
        Map<String, SecurityIndexCountItemDto> regionCodeSecurityIndexMap = getSecurityIndexCountItemDtoMap(regionModels);
        List<Map<String, String>> data = new ArrayList<>();
        List<Map<String, String>> finalData = data;
        regionCodeSecurityIndexMap.forEach((k, v) -> {
            Map<String, String> item = new HashMap<>();
            item.put("name", v.getRegionName());
            // 计算xuke的加权值
            double xukeWeighted = Double.parseDouble(v.getXuke()) * 0.10;
            // 剩余权重平均分配给四个因素，每个因素大约22.5%（但为了简化，我们这里用25%的剩余权重）
            double remainingWeightPerFactor = 0.90 / 4;
            // 计算其他四个因素的加权值
            double jianyanchaoqiWeighted = Double.parseDouble(v.getJianyanchaoqi()) * remainingWeightPerFactor;
            double jianyanhegeWeighted = Double.parseDouble(v.getJianyanhege()) * remainingWeightPerFactor;
            double czjcWeighted = Double.parseDouble(v.getCzjc()) * remainingWeightPerFactor;
            double czjchegeWeighted = Double.parseDouble(v.getCzjchege()) * remainingWeightPerFactor;
            // 计算总指数
            double totalIndex = xukeWeighted + jianyanchaoqiWeighted + jianyanhegeWeighted + czjcWeighted + czjchegeWeighted;
            item.put("value", String.format("%.2f", totalIndex));
            finalData.add(item);
        });
        data = finalData.stream()
                .filter(map -> map.containsKey("value"))
                .sorted(Comparator.comparingDouble(map -> {
                    // 检查并解析值，如果失败则返回Double.MIN_VALUE（或根据需要处理）
                    try {
                        return Double.parseDouble(map.get("value"));
                    } catch (NumberFormatException e) {
                        // 处理无法解析为double的情况，这里返回Double.MIN_VALUE表示最小
                        // 或者可以抛出自定义异常，记录日志等
                        return Double.MIN_VALUE;
                    }
                }))
                .collect(Collectors.toList());
        Map<String, Object> result = new HashMap<>();
        List<String> name = data.stream().map(map -> map.get("name")).collect(Collectors.toList());
        Collections.reverse(name);
        result.put("xdata", name);
        List<String> value = data.stream().map(map -> map.get("value")).collect(Collectors.toList());
        Collections.reverse(value);
        result.put("ydata", value);
        return result;
    }

    private Map<String, SecurityIndexCountItemDto> getSecurityIndexCountItemDtoMap(List<RegionModel> regionModels) {
        // 多线程处理后，顺序与regionModels的顺序不一致，故生成list
        List<SecurityIndexCountItemDto> countItemDtoList = getSecurityIndexCountDtoList(regionModels);
        return countItemDtoList.stream().collect(Collectors.toMap(SecurityIndexCountItemDto::getRegionCode, Function.identity(), (k1, k2) -> k2));
    }

    private List<SecurityIndexCountItemDto> getSecurityIndexCountDtoList(List<RegionModel> regionModels) {
        return regionModels.parallelStream().map(r -> {
            String orgCode = stCommonService.getAndSetOrgCode(r.getRegionCode().toString());
            SecurityIndexCountItemDto countItemDto = new SecurityIndexCountItemDto();
            countItemDto.setRegionCode(r.getRegionCode().toString());
            countItemDto.setRegionName(r.getRegionName());
            if (ObjectUtils.isEmpty(orgCode)) {
                this.setDefaultZeroData(countItemDto);
            } else {
                // 1.许可有效率
                String value = this.getLicenseEfficiencyByRegion(orgCode);
                if (NO_DATA_STR.equals(value)) {
                    value = "0";
                    countItemDto.setXuke(value);
                } else {
                    countItemDto.setXuke(value);
                }
                // 2.使用登记办理率
                countItemDto.setShiyongdengji(cylinderStatisticsMapper.queryUserRegisterCertAppPercent(orgCode));
                // 3.检验超期率
                countItemDto.setJianyanchaoqi(statisticsMapper.getInspectionExpiredRate(orgCode, null).get("expiredRate").toString());
                // 4.检验合格率
                countItemDto.setJianyanhege(statisticsMapper.getInspectionResultRate(orgCode, null).get("resultRate").toString());
                // 5.充装检查率
                long totalAll = searchEsCount(false, false, orgCode, null);
                long totalCheck = searchEsCount(true, false, orgCode, null);
                long totalResult = searchEsCount(false, true, orgCode, null);
                if (totalAll != 0) {
                    countItemDto.setCzjc(new DecimalFormat("#.00").format(((double) totalCheck / totalAll) * 100));
                } else {
                    countItemDto.setCzjc("0");
                }
                // 6.充装检查合格率
                if (totalAll != 0) {
                    countItemDto.setCzjchege(new DecimalFormat("#.00").format(((double) totalResult / totalAll) * 100));
                } else {
                    countItemDto.setCzjchege("0");
                }
            }
            return countItemDto;
        }).collect(Collectors.toList());
    }

    private void setDefaultZeroData(SecurityIndexCountItemDto countItemDto) {
        countItemDto.setXuke("0");
        countItemDto.setShiyongdengji("0");
        countItemDto.setJianyanhege("0");
        countItemDto.setCzjc("0");
        countItemDto.setCzjchege("0");
        countItemDto.setJianyanchaoqi("0");
    }

    public Map<String, Object> mainBodyCount(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        Map<String, Object> dataMap = new HashMap<>();
        String startDate = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<String> mainBodyNameList = IssueMainBodyEnum.getEnumNameList();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<Map<String, Object>> list = statisticsMapper.selectByOrgAndMainBody(orgCode, mainBodyNameList, startDate, null);
        List<Object> legendDataList = new ArrayList<>();
        for (int i = 0; i < 2; i++) {
            Map<String, Object> map = new HashMap<>();
            if (0 == i) {
                map.put("dataKey", "problemCount");
                map.put("value", "问题数量");
            } else if (1 == i) {
                map.put("dataKey", "problemRate");
                map.put("value", "问题闭环率");
            }
            legendDataList.add(map);
        }
        resultMap.put("legendData", legendDataList);
        List<String> xdata = new ArrayList<>();
        List<Object> problemCount = new ArrayList<>();
        List<Object> problemRate = new ArrayList<>();
        list.stream().forEach(t -> {
            dataMap.put(t.get("sourceType").toString(), t.get("count").toString());
        });
        for (int i = 0; i < mainBodyNameList.size(); i++) {
            if (null != dataMap.get(mainBodyNameList.get(i))) {
                problemCount.add(dataMap.get(mainBodyNameList.get(i)));
                List<String> list1 = new ArrayList<>();
                list1.add(mainBodyNameList.get(i));
                List<Map<String, Object>> maps = statisticsMapper.selectByOrgAndMainBody(orgCode, list1, startDate, "1");
                if (maps.size() > 0) {
                    problemRate.add(new DecimalFormat("0.00").format(Double.parseDouble(maps.get(0).get("count").toString()) / Double.parseDouble(dataMap.get(mainBodyNameList.get(i)).toString())));
                } else {
                    problemRate.add("0");
                }
            } else {
                problemCount.add("0");
                problemRate.add("0");
            }
            xdata.add(mainBodyNameList.get(i));
        }
        resultMap.put("xdata", xdata);
        resultMap.put("problemCount", problemCount);
        resultMap.put("problemRate", problemRate);

        return resultMap;
    }


    public Map<String, Object> equipmentCount(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        Map<String, Object> dataMap = new HashMap<>();
        List<Object> xdata = new ArrayList<>();
        List<Object> ydata = new ArrayList<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<String> enumNameList = IssueTypeEnum.getEnumNameListByMainBody("3");
        String startDate = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<Map<String, Object>> list = statisticsMapper.selectByOrgAndProblemType(orgCode, enumNameList, startDate, IssueMainBodyEnum.EQUIPMENT.getName(), null);
        list.stream().forEach(t -> {
            dataMap.put(t.get("problemType").toString(), t.get("count").toString());
        });
        for (int i = 0; i < enumNameList.size(); i++) {
            xdata.add(enumNameList.get(i));
            ydata.add(dataMap.getOrDefault(enumNameList.get(i), "0"));
        }
        resultMap.put("xdata", xdata);
        resultMap.put("ydata", ydata);
        return resultMap;
    }

    public List<Map<String, Object>> companyCount(String cityCode) {
        List<Map<String, Object>> resultList = new ArrayList<>();
        Map<String, Object> dataMap = new HashMap<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<String> enumNameList = IssueTypeEnum.getEnumNameListByMainBody("2");
        String startDate = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<Map<String, Object>> list = statisticsMapper.selectByOrgAndProblemType(orgCode, enumNameList, startDate, IssueMainBodyEnum.COMPANY.getName(), null);
        list.stream().forEach(t -> {
            dataMap.put(t.get("problemType").toString(), t.get("count").toString());
        });
        for (int i = 0; i < enumNameList.size(); i++) {
            Map<String, Object> map = new HashMap<>();
            if (enumNameList.get(i).equals("许可临期")) {
                map.put("key", "licenseExpirationDateNumber");
                map.put("label", "许可临期数");
                map.put("value", dataMap.getOrDefault(enumNameList.get(i), "0"));
            } else if (enumNameList.get(i).equals("许可超期")) {
                map.put("key", "licenseOverdueNumber");
                map.put("label", "许可超期数");
                map.put("value", dataMap.getOrDefault(enumNameList.get(i), "0"));
            } else if (enumNameList.get(i).equals("维保备案超期")) {
                map.put("key", "maintenanceFilingOverdueNumber");
                map.put("label", "维保备案超期数（仅电梯）");
                map.put("value", dataMap.getOrDefault(enumNameList.get(i), "0"));
            }
            resultList.add(map);
        }
        return resultList;
    }

    public Map<String, Object> personCount(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        Map<String, Object> dataMap = new HashMap<>();
        List<Object> xdata = new ArrayList<>();
        List<Object> ydata = new ArrayList<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<String> enumNameList = IssueTypeEnum.getEnumNameListByMainBody("1");
        String startDate = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<Map<String, Object>> list = statisticsMapper.selectByOrgAndProblemType(orgCode, enumNameList, startDate, IssueMainBodyEnum.PERSON.getName(), null);
        list.stream().forEach(t -> {
            dataMap.put(t.get("problemType").toString(), t.get("count").toString());
        });
        for (int i = 0; i < enumNameList.size(); i++) {
            xdata.add(enumNameList.get(i));
            ydata.add(dataMap.getOrDefault(enumNameList.get(i), "0"));
        }
        resultMap.put("xdata", xdata);
        resultMap.put("ydata", ydata);
        return resultMap;
    }

    public Map<String, Object> cylinderCount(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        Map<String, Object> dataMap = new HashMap<>();
        List<Object> xdata = new ArrayList<>();
        List<Object> ydata = new ArrayList<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<String> enumNameList = IssueTypeEnum.getEnumNameListByMainBody("4");
        String startDate = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<Map<String, Object>> list = statisticsMapper.selectByOrgAndProblemType(orgCode, enumNameList, startDate, IssueMainBodyEnum.EQUIPMENT.getName(), "2300");
        list.stream().forEach(t -> {
            dataMap.put(t.get("problemType").toString(), t.get("count").toString());
        });
        for (int i = 0; i < enumNameList.size(); i++) {
            xdata.add(enumNameList.get(i));
            ydata.add(dataMap.getOrDefault(enumNameList.get(i), "0"));
        }
        resultMap.put("xdata", xdata);
        resultMap.put("ydata", ydata);
        return resultMap;
    }

    public Map<String, Object> issueCountByMonth(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        List<Object> legendData = new ArrayList<>();
        for (int i = 0; i < 2; i++) {
            Map<String, Object> map = new HashMap<>();
            if (i == 0) {
                map.put("key", "problemCount");
                map.put("value", "问题数量");
            } else {
                map.put("key", "processedRate");
                map.put("value", "已处理");
            }
            legendData.add(map);
        }
        resultMap.put("legendData", legendData);
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        int year = LocalDate.now().getYear();
        List<Map<String, Object>> allIssue = statisticsMapper.getIssueCountByMonth(orgCode, String.valueOf(year), null);
        List<Map<String, Object>> endIssue = statisticsMapper.getIssueCountByMonth(orgCode, String.valueOf(year), "1");
        Map<String, Object> allIssueMap = new HashMap<>();
        Map<String, Object> endIssueMap = new HashMap<>();
        allIssue.stream().forEach(t -> {
            allIssueMap.put(t.get("time").toString(), t.get("count"));
        });
        endIssue.stream().forEach(t -> {
            endIssueMap.put(t.get("time").toString(), t.get("count"));
        });
        List<Object> problemCount = new ArrayList<>();
        List<Object> processedRate = new ArrayList<>();
        List<Object> xdata = new ArrayList<>();
        for (int i = 0; i < 12; i++) {
            String time = LocalDate.now().minusMonths(i).format(DateTimeFormatter.ofPattern("yyyy-MM"));
            xdata.add(time);
            if (allIssueMap.get(time) == null) {
                problemCount.add("0");
                processedRate.add("0");
            } else {
                problemCount.add(allIssueMap.get(time));
                if (endIssueMap.get(time) == null) {
                    processedRate.add("0");
                } else {
                    processedRate.add(new DecimalFormat("0.00").format(Double.parseDouble(endIssueMap.get(time).toString()) / Double.parseDouble(allIssueMap.get(time).toString())));
                }
            }
        }
        Collections.reverse(xdata);
        Collections.reverse(problemCount);
        Collections.reverse(processedRate);
        resultMap.put("xdata", xdata);
        resultMap.put("problemCount", problemCount);
        resultMap.put("processedRate", processedRate);
        return resultMap;
    }

    public Map<String, Object> issueCountByDay(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        List<Object> legendData = new ArrayList<>();
        for (int i = 0; i < 2; i++) {
            Map<String, Object> map = new HashMap<>();
            if (i == 0) {
                map.put("key", "problemCount");
                map.put("value", "问题数量");
            } else {
                map.put("key", "processedRate");
                map.put("value", "已处理");
            }
            legendData.add(map);
        }
        resultMap.put("legendData", legendData);
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        String year = LocalDate.now().minusDays(29).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
        List<Map<String, Object>> allIssue = statisticsMapper.getIssueCountByDate(orgCode, year, null);
        List<Map<String, Object>> endIssue = statisticsMapper.getIssueCountByDate(orgCode, year, "1");
        Map<String, Object> allIssueMap = new HashMap<>();
        Map<String, Object> endIssueMap = new HashMap<>();
        allIssue.stream().forEach(t -> {
            allIssueMap.put(t.get("time").toString(), t.get("count"));
        });
        endIssue.stream().forEach(t -> {
            endIssueMap.put(t.get("time").toString(), t.get("count"));
        });
        List<Object> problemCount = new ArrayList<>();
        List<Object> processedRate = new ArrayList<>();
        List<Object> xdata = new ArrayList<>();
        for (int i = 0; i < 29; i++) {
            String time = LocalDate.now().minusDays(i).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            xdata.add(time);
            if (allIssueMap.get(time) == null) {
                problemCount.add("0");
                processedRate.add("0");
            } else {
                problemCount.add(allIssueMap.get(time));
                if (endIssueMap.get(time) == null) {
                    processedRate.add("0");
                } else {
                    processedRate.add(new DecimalFormat("0.00").format(Double.parseDouble(endIssueMap.get(time).toString()) / Double.parseDouble(allIssueMap.get(time).toString())));
                }
            }
        }
        Collections.reverse(xdata);
        Collections.reverse(problemCount);
        Collections.reverse(processedRate);
        resultMap.put("xdata", xdata);
        resultMap.put("problemCount", problemCount);
        resultMap.put("processedRate", processedRate);
        return resultMap;
    }

    public Map<String, Object> issueCountByCityCode(String cityCode) {
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        String now = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM"));
        int count = statisticsMapper.countByOrgCode(orgCode, now);
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("issueCount", count);
        return resultMap;
    }


    public List<Map<String, Object>> issueChildrenCityCount(String cityCode) throws Exception {
        List<RegionModel> regionModels = stCommonService.setRegionIfRootParent(cityCode);
        List<Integer> regionCodeList = regionModels.stream().map(m -> m.getRegionCode()).collect(Collectors.toList());
        List<Map<String, String>> orgCodeAndCompanyCodesByCompanyCodes = statisticsMapper.getOrgCodeAndCompanyCodesByCompanyCodes(regionCodeList);
        List<String> orgCodes = orgCodeAndCompanyCodesByCompanyCodes.stream().map(m -> m.get("orgCode")).collect(Collectors.toList());
        String now = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM"));
        Map<String, String> orgCodeAndCompanyCodesMap = new HashMap<>();
        orgCodeAndCompanyCodesByCompanyCodes.stream().forEach(t -> {
            orgCodeAndCompanyCodesMap.put(t.get("companyCode"), t.get("orgCode"));
        });
        List<Map<String, Object>> list = statisticsMapper.countByOrgCodes(orgCodes, now);
        Map<String, Object> dataMap = new HashMap<>();
        list.stream().forEach(t -> {
            dataMap.put(t.get("orgCode").toString(), t.get("count"));
        });

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (RegionModel regionModel : regionModels) {
            Map<String, Object> map = new HashMap<>();
            AtomicInteger count = new AtomicInteger();
            map.put("regionCode", regionModel.getRegionCode());
            map.put("regionName", regionModel.getRegionName());
            if (StringUtils.isNotBlank(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                dataMap.forEach((k, v) -> {
                    if (k.startsWith(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                        if (null != v) {
                            count.addAndGet(Integer.valueOf(v + ""));
                        }
                    }
                });
            }
            map.put("issueCount", count);
            resultList.add(map);
        }

        return resultList;
    }

    public List<Map<String, Object>> issueMonthList(String cityCode) {
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        String time = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM"));
        List<Map<String, Object>> list = statisticsMapper.issueMonthList(orgCode, time);
        return list;
    }

    public Map<String, Object> issueProblemLevelCount(String cityCode) {
        Map<String, Object> resultMap = new HashMap<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        String time = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy"));
        List<Map<String, Object>> list = statisticsMapper.issueProblemLevelCount(orgCode, time);
        List<Object> xdata = new ArrayList<>();
        List<Object> ydata = new ArrayList<>();
        for (int i = 0; i < list.size(); i++) {
            xdata.add(list.get(i).get("problemLevel"));
            ydata.add(list.get(i).get("count"));
        }
        resultMap.put("xdata", xdata);
        resultMap.put("ydata", ydata);
        return resultMap;
    }

    public List<Map<String, Object>> issueCompanyTop(String cityCode) {
        List<Map<String, Object>> resultList = new ArrayList<>();
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        String time = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy"));
        List<Map<String, Object>> list = statisticsMapper.issueCompanyTop(orgCode, time);
        for (int i = 0; i < list.size(); i++) {
            Map<String, Object> map = new HashMap<>();
            map.put("sequenceNbr", (i + 1) + "");
            map.put("region", list.get(i).get("governingBody"));
            map.put("company", list.get(i).get("principalUnit"));
            map.put("count", list.get(i).get("count"));
            resultList.add(map);
        }
        return resultList;
    }

    public List<Map<String, Object>> issueChildrenCityCountTop(String cityCode) throws Exception {
        List<RegionModel> regionModels = stCommonService.setRegionIfRootParent(cityCode);
        List<Integer> regionCodeList = regionModels.stream().map(RegionModel::getRegionCode).collect(Collectors.toList());
        List<Map<String, String>> orgCodeAndCompanyCodesByCompanyCodes = statisticsMapper.getOrgCodeAndCompanyCodesByCompanyCodes(regionCodeList);
        List<String> orgCodes = orgCodeAndCompanyCodesByCompanyCodes.stream().map(m -> m.get("orgCode")).collect(Collectors.toList());
        String now = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy"));
        Map<String, String> orgCodeAndCompanyCodesMap = new HashMap<>();
        orgCodeAndCompanyCodesByCompanyCodes.stream().forEach(t -> {
            orgCodeAndCompanyCodesMap.put(t.get("companyCode"), t.get("orgCode"));
        });
        List<Map<String, Object>> allIssueCount = statisticsMapper.issueChildrenCityCountTop(orgCodes, now, null);
        List<Map<String, Object>> endIssueCount = statisticsMapper.issueChildrenCityCountTop(orgCodes, now, "1");
        Map<String, Object> allIssuedataMap = new HashMap<>();
        Map<String, Object> endIssuedataMap = new HashMap<>();
        allIssueCount.stream().forEach(t -> {
            allIssuedataMap.put(t.get("orgCode").toString(), t.get("count"));
        });

        endIssueCount.stream().forEach(t -> {
            endIssuedataMap.put(t.get("orgCode").toString(), t.get("count"));
        });

        List<Map<String, Object>> resultList = new ArrayList<>();
        for (RegionModel regionModel : regionModels) {
            Map<String, Object> map = new HashMap<>();
            AtomicInteger allCount = new AtomicInteger();
            AtomicInteger endCount = new AtomicInteger();
            map.put("region", regionModel.getRegionName());
            if (StringUtils.isNotBlank(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                allIssuedataMap.forEach((k, v) -> {
                    if (k.startsWith(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                        if (null != v) {
                            allCount.addAndGet(Integer.valueOf(v + ""));
                        }
                    }
                });
            }

            if (StringUtils.isNotBlank(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                endIssuedataMap.forEach((k, v) -> {
                    if (k.startsWith(orgCodeAndCompanyCodesMap.get(regionModel.getRegionCode() + ""))) {
                        if (null != v) {
                            endCount.addAndGet(Integer.valueOf(v + ""));
                        }
                    }
                });
            }
            if (allCount.get() != 0) {
                map.put("rate", new DecimalFormat("0.00").format(Double.parseDouble(endCount + "") / (Double.parseDouble(allCount.get() + ""))));
            } else {
                map.put("rate", "0.00");
            }
            resultList.add(map);
        }
        resultList.sort(Comparator.comparing(m -> Double.valueOf(m.get("rate") + ""), Comparator.reverseOrder()));
        for (int i = 0; i < resultList.size(); i++) {
            resultList.get(i).put("sequenceNbr", (i + 1) + "");
        }
        return resultList;
    }

    public List<Map<String, Object>> issueCountTopByProblemType(String cityCode) {
        String orgCode = stCommonService.getAndSetOrgCode(cityCode);
        List<Map<String, Object>> allList = statisticsMapper.issueCountTopByProblemType(orgCode, null);
        List<Map<String, Object>> endList = statisticsMapper.issueCountTopByProblemType(orgCode, "1");
        Map<String, Object> endIssuedataMap = new HashMap<>();
        endList.forEach(t -> {
            endIssuedataMap.put(t.get("problemTypeCode").toString(), t.get("count"));
        });
        List<Map<String, Object>> resultList = new ArrayList<>();
        for (int i = 0; i < allList.size(); i++) {
            Map<String, Object> map = new HashMap<>();
            map.put("name", IssueTypeEnum.getNameByCode(allList.get(i).get("problemTypeCode").toString()));
            map.put("count", allList.get(i).get("count"));
            if (null != endIssuedataMap.get(allList.get(i).get("problemTypeCode").toString())) {
                map.put("rate", new DecimalFormat("0.00").format(Double.parseDouble(endIssuedataMap.get(allList.get(i).get("problemTypeCode").toString()) + "") / (Double.parseDouble(allList.get(i).get("count") + ""))));
            }
            resultList.add(map);
        }

        return resultList;
    }

    public List<Map<String, Object>> getEquipQuestionList(DPFilterParamDto dpFilterParamDto) {
        this.setDefaultFilter(dpFilterParamDto);
        String orgCode = stCommonService.getAndSetOrgCode(dpFilterParamDto.getCityCode());
        if (StringUtils.isEmpty(orgCode)) {
            return new ArrayList<>();
        }
        List<EquipQuestionNumCountDto> equipQuestionNumCountDtos = statisticsMapper.queryEquipQuestionList(orgCode, dpFilterParamDto);
        return this.calPercentFinishedAndCast2Map(equipQuestionNumCountDtos);
    }

    private List<Map<String, Object>> calPercentFinishedAndCast2Map(List<EquipQuestionNumCountDto> equipQuestionNumCountDtos) {
        return equipQuestionNumCountDtos.stream().map(e -> {
            Map<String, Object> item = new HashMap<>();
            // 设备id
            item.put("key", e.getSourceId());
            // 区域名称
            item.put("regionName", e.getCityName());
            // 设备种类
            item.put("deviceCategoryName", EquipmentClassifityEnum.getNameByCode(e.getEquList()));
            // 监管码/使用登记证编号
            item.put("code", this.buildCodeWithSplit(e.getSupervisoryCode(), e.getUseOrgCode()));
            // 使用单位
            item.put("useUnitName", e.getUseUnitName());
            // 问题次数
            item.put("issuesNumber", e.getTotal());
            // 闭环率
            item.put("closedLoopRate", this.calClosedLoopRate(e.getTotal(), e.getFinished()));
            return item;
        }).collect(Collectors.toList());
    }

    private String calClosedLoopRate(Long total, Long finished) {
        if (total == 0) {
            return "0";
        }
        BigDecimal totalBig = new BigDecimal(total.toString());
        BigDecimal finishedBig = new BigDecimal(finished.toString());
        BigDecimal result = finishedBig.divide(totalBig, 2, RoundingMode.HALF_UP);
        NumberFormat nt = NumberFormat.getPercentInstance();
        nt.setMinimumFractionDigits(0);
        return nt.format(result);
    }

    private String buildCodeWithSplit(String supervisoryCode, String useOrgCode) {
        if (StringUtils.isNotEmpty(supervisoryCode) && StringUtils.isNotEmpty(useOrgCode)) {
            return String.format("%s/%s", supervisoryCode, useOrgCode);
        }
        if (StringUtils.isNotEmpty(supervisoryCode)) {
            return supervisoryCode;
        }
        if (StringUtils.isNotEmpty(useOrgCode)) {
            return useOrgCode;
        }
        return "";
    }

    private void setDefaultFilter(DPFilterParamDto dpFilterParamDto) {
        if (StringUtils.isEmpty(dpFilterParamDto.getBeginDate())) {
            dpFilterParamDto.setBeginDate(cn.hutool.core.date.DateUtil.offsetDay(new Date(), -30).toDateStr());
        }
        if (StringUtils.isEmpty(dpFilterParamDto.getEndDate())) {
            dpFilterParamDto.setEndDate(DateUtil.today());
        }
    }

    public Map<String, Object> getCenterMapCountDataForGlobal(DPFilterParamDto dpFilterParamDto) {
        Map<String, Object> result = new HashMap<>();
        String orgCode = stCommonService.getAndSetOrgCode(dpFilterParamDto.getCityCode());
        if (StringUtils.isEmpty(orgCode)) {
            return new HashMap<>();
        }
        // 1.气瓶数量统计
        long cylinderNum = stCommonService.staticsCenterMapCountDataForCylinder(result, orgCode);
        //1.8大类设备数量统计,压力容器里包括气瓶所以需要特殊处理，在统计压力容器时去掉气瓶的数量
        stCommonService.staticsCenterMapCountDataForEquip(result, cylinderNum, orgCode);
        //2.压力管道长度统计
        stCommonService.staticsCenterMapCountDataForPipeline(result, orgCode);
        //3.单位数量统计
        this.staticsCenterMapCountDataForCompany(result, orgCode);
        //4.人员数量统计
        this.staticsCenterMapCountDataForPerson(result, dpFilterParamDto, orgCode);
        //5.问题数统计
        this.staticsCenterMapCountDataForIssue(result, orgCode);
        return result;
    }

    private void staticsCenterMapCountDataForIssue(Map<String, Object> result, String orgCode) {
        List<Map<String, Object>> list = statisticsMapper.selectByOrg(orgCode);
        Map<String, Object> dataMap = new HashMap<>();
        AtomicReference<Long> issueCount = new AtomicReference<>(0L);
        list.forEach(t -> {
            dataMap.put(t.get("sourceType").toString(), t.get("count"));
            issueCount.updateAndGet(v -> v + Long.parseLong(t.get("count").toString()));
        });
        result.put("issueCount", issueCount.get());
        result.put("deviceIssueCount", dataMap.getOrDefault(DPMapStatisticsItemEnum.EQU_ISSUES_COUNT.getCode(), 0L));
        result.put("companyIssueCount", dataMap.getOrDefault(DPMapStatisticsItemEnum.COMPANY_ISSUES_COUNT.getCode(), 0L));
        result.put("personIssueCount", dataMap.getOrDefault(DPMapStatisticsItemEnum.PERSON_ISSUES_COUNT.getCode(), 0L));

    }

    private void staticsCenterMapCountDataForCompany(Map<String, Object> result, String orgCode) {
        if (orgCode == null) {
            setDefaultCompanyCountData(result);
            return;
        }
        List<CountDto> countDtos = enterpriseInfoMapper.countByUnitTypeAndOrgCode(orgCode);
        result.put(DPMapStatisticsItemEnum.USERS_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_USE)).mapToInt(CountDto::getIntValue).sum());
        result.put(DPMapStatisticsItemEnum.CONSTRUCTION_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_MAINTENANCE)).mapToInt(CountDto::getIntValue).sum());
        result.put(DPMapStatisticsItemEnum.MANUFACTURING_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_MANUFACTURE)).mapToInt(CountDto::getIntValue).sum());
        result.put(DPMapStatisticsItemEnum.GAS_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_FILLING)).mapToInt(CountDto::getIntValue).sum());
    }

    private void setDefaultCompanyCountData(Map<String, Object> m) {
        m.put(DPMapStatisticsItemEnum.USERS_UNITS.getCode(), 0);
        m.put(DPMapStatisticsItemEnum.CONSTRUCTION_UNITS.getCode(), 0);
        m.put(DPMapStatisticsItemEnum.MANUFACTURING_UNITS.getCode(), 0);
        m.put(DPMapStatisticsItemEnum.GAS_UNITS.getCode(), 0);
    }

    private void staticsCenterMapCountDataForPerson(Map<String, Object> result, DPFilterParamDto dpFilterParamDto, String orgCode) {
        if (orgCode == null) {
            result.put(DPMapStatisticsItemEnum.OPERATORS.getCode(), 0);
            return;
        }
        Long num = userInfoMapper.countUserByPostAndAreaCode(orgCode, "6552", dpFilterParamDto.getCityCode());
        result.put(DPMapStatisticsItemEnum.OPERATORS.getCode(), num);
    }

    public List<Map<String, Object>> getCenterMapCountDataForOverview(DPFilterParamDto dpFilterParamDto) throws Exception {
        List<RegionModel> regionModels = stCommonService.setRegionIfRootParent(dpFilterParamDto.getCityCode());
        List<Map<String, Object>> result = regionModels.parallelStream().map(r -> {
            DPFilterParamDto filterParamDto = new DPFilterParamDto();
            filterParamDto.setCityCode(r.getRegionCode().toString());
            Map<String, Object> itemResult = getCenterMapOverviewData(filterParamDto);
            itemResult.put("regionCode", r.getRegionCode());
            itemResult.put("regionName", r.getRegionName());
            return itemResult;
        }).collect(Collectors.toList());
        this.setCompanyDataBatch(result);
        return result;
    }

    private void setCompanyDataBatch(List<Map<String, Object>> result) {
        List<CountDto> countDtos = enterpriseInfoMapper.countByUnitTypeAndOrgCodeNoParam();
        result.forEach(m -> {
            String cityCode = m.get("regionCode").toString();
            String orgCode = stCommonService.getAndSetOrgCode(cityCode);
            if (orgCode != null) {
                m.put(DPMapStatisticsItemEnum.USERS_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_USE) && c.getLabel().contains(orgCode)).mapToInt(CountDto::getIntValue).sum());
                m.put(DPMapStatisticsItemEnum.CONSTRUCTION_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_MAINTENANCE) && c.getLabel().contains(orgCode)).mapToInt(CountDto::getIntValue).sum());
                m.put(DPMapStatisticsItemEnum.MANUFACTURING_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_MANUFACTURE) && c.getLabel().contains(orgCode)).mapToInt(CountDto::getIntValue).sum());
                m.put(DPMapStatisticsItemEnum.GAS_UNITS.getCode(), countDtos.stream().filter(c -> c.getKeyStr().contains(COMPANY_TYPE_FILLING) && c.getLabel().contains(orgCode)).mapToInt(CountDto::getIntValue).sum());
            } else {
                setDefaultCompanyCountData(m);
            }
        });
    }

    private Map<String, Object> getCenterMapOverviewData(DPFilterParamDto dpFilterParamDto) {
        Map<String, Object> result = new HashMap<>();
        String orgCode = stCommonService.getAndSetOrgCode(dpFilterParamDto.getCityCode());
        if (StringUtils.isEmpty(orgCode)) {
            return new HashMap<>();
        }
        // 0. 气瓶数量统计
        long cylinderNum = stCommonService.staticsCenterMapCountDataForCylinder(result, orgCode);
        // 1. 8大类设备数量统计,压力容器里包括气瓶所以需要特殊处理，在统计压力容器时去掉气瓶的数量
        stCommonService.staticsCenterMapCountDataForEquip(result, cylinderNum, orgCode);
        // 2. 压力管道长度统计
        stCommonService.staticsCenterMapCountDataForPipeline(result, orgCode);
        // 3.单位数量统计
        this.staticsCenterMapCountDataForCompany(result, orgCode);
        // 4. 人员数量统计
        this.staticsCenterMapCountDataForPerson(result, dpFilterParamDto, orgCode);
        // 5.问题统计
        this.staticsCenterMapCountDataForIssue(result, orgCode);

        return result;
    }
}