package com.yeejoin.equipmanage.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yeejoin.equipmanage.common.entity.Car;
import com.yeejoin.equipmanage.common.entity.CarSpeedWarningRecord;
import com.yeejoin.equipmanage.common.entity.WlCarMileage;
import com.yeejoin.equipmanage.common.utils.CoordinateUtil;
import com.yeejoin.equipmanage.common.utils.DateUtils;
import com.yeejoin.equipmanage.common.utils.RedisUtil;
import com.yeejoin.equipmanage.controller.Coordinate;
import com.yeejoin.equipmanage.fegin.IotFeign;
import com.yeejoin.equipmanage.listener.CarIotNewListener;
import com.yeejoin.equipmanage.mapper.CarMapper;
import com.yeejoin.equipmanage.mapper.WlCarMileageMapper;
import com.yeejoin.equipmanage.mapper.WlCarSpeedWaringRecordMapper;
import com.yeejoin.equipmanage.service.ICarService;
import com.yeejoin.equipmanage.service.IWlCarMileageService;
import org.apache.commons.lang3.ObjectUtils;
import org.checkerframework.checker.units.qual.Speed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.typroject.tyboot.component.emq.EmqKeeper;
import org.typroject.tyboot.core.restful.utils.ResponseModel;

import javax.annotation.Resource;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 车量里程表 服务实现类
 *
 * @author duanwei
 * @date 2023-02-01
 */

@Service
@EnableScheduling
@EnableAsync
public class WlCarMileageServiceImpl extends ServiceImpl<WlCarMileageMapper, WlCarMileage>
        implements IWlCarMileageService {
    private final String GUIDE_KEY = "8d2ab194d72e88d3636e9d721814333a";
    private final String GUIDE_URL = "https://restapi.amap.com/v4/grasproad/driving?";
    private final String GUIDE_ADDRESS_URL = "https://restapi.amap.com/v3/geocode/regeo?";
    @Autowired
    IotFeign iotFeign;
    @Autowired
    WlCarMileageMapper wlCarMileageMapper;
    @Autowired
    WlCarSpeedWaringRecordMapper wlCarSpeedWaringRecordMapper;
    @Autowired
    CarMapper carMapper;
    @Value("${car.max.speed:100}")
    private Integer CAR_MAX_SPEED;
    @Value("${mileage.parameter}")
    private Double mileageParameter;
    @Resource
    private EmqKeeper emqKeeper;
    @Autowired
    private RedisTemplate redisTemplate;

    public static List<String> getDayByMonth(Date date) {
        List<String> data = new ArrayList<>();
        try {
            Calendar c = Calendar.getInstance();
            c.setTime(date);
            // 获取当前的年份
            int year = c.get(Calendar.YEAR);
            // 获取当前的月份（需要加1才是现在的月份）
            int month = c.get(Calendar.MONTH) + 1;
            // 获取本月的总天数
            int dayCount = c.getActualMaximum(Calendar.DAY_OF_MONTH);
            // 定义时间格式
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            // 开始日期为当前年月拼接1号
            Date startDate = sdf.parse(year + "-" + month + "-01");
            // 结束日期为当前年月拼接该月最大天数
            Date endDate = sdf.parse(year + "-" + month + "-" + dayCount);
            // 设置calendar的开始日期
            c.setTime(startDate);
            // 当前时间小于等于设定的结束时间
            while (c.getTime().compareTo(endDate) <= 0) {
                String time = sdf.format(c.getTime());
                data.add(time);
                // 当前日期加1
                c.add(Calendar.DATE, 1);
            }
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return data;
    }

    @Override
    public Page<WlCarMileage> page(Page<WlCarMileage> page, WlCarMileage wlCarMileage) {
        return this.baseMapper.page(page, wlCarMileage, mileageParameter);
    }

    @Override
    public Double totalMileage(String iotCode) {
        return this.baseMapper.totalMileage(iotCode);
    }


    @Override
    public Map<String, Boolean> getCalender(long id, Date date) {
        List<String> daylist = new ArrayList<String>();
        Map<String, Boolean> map = new HashMap<>();
        Car car = carMapper.selectById(id);
        if (car == null || car.getIotCode() == null || date == null) {
            return map;
        }
        Calendar c = Calendar.getInstance();
        // 获取上一个月
        c.setTime(date);
        c.add(Calendar.MONTH, -1);
        daylist.addAll(getDayByMonth(c.getTime()));

        // 获取当月
        c.setTime(date);
        daylist.addAll(getDayByMonth(c.getTime()));

        // 获取下一个月
        c.setTime(date);
        c.add(Calendar.MONTH, 1);
        daylist.addAll(getDayByMonth(c.getTime()));
//		List<Map<String, Object>> hasList = this.listMaps(new QueryWrapper<WlCarMileage>()
//				.select("COUNT(1) AS count,date").lambda().eq(WlCarMileage::getIotCode, car.getIotCode())
//				.between(WlCarMileage::getDate, daylist.get(0), daylist.get(daylist.size() - 1))
//				.isNotNull(WlCarMileage::getEndTime)
//				.ge(WlCarMileage::getTravel, 0.5)
//				.groupBy(WlCarMileage::getDate));
        Map<String, Object> hasMap = new HashMap<>();
        List<WlCarMileage> wlCarMileageList = this.list(new QueryWrapper<WlCarMileage>().eq("iot_code", car.getIotCode())
                .between("date", daylist.get(0), daylist.get(daylist.size() - 1))
                .isNotNull("end_time")
                .ge("travel", 0.5));
//                .groupBy("date"));
        List<WlCarMileage> wlCarMileageListActiveData = wlCarMileageList.stream().filter(wl -> (!ObjectUtils.isEmpty(wl.getTakeTime()) && !ObjectUtils.isEmpty(wl.getTravel()) && ((wl.getTravel() / (wl.getTakeTime() / 3600000.0)) < CAR_MAX_SPEED))).collect(Collectors.toList());
        List<String> activeDateList = wlCarMileageListActiveData.stream().map(wlCarMileage -> String.valueOf(DateUtils.convertDateToString(wlCarMileage.getDate(), DateUtils.DATE_PATTERN))).collect(Collectors.toList());
        for (String day : daylist) {
            boolean has = false;
            if (activeDateList.contains(day)) {
                has = true;
            }
            map.put(day, has);
        }
        return map;
    }

    @Override
    @Scheduled(cron = "${mileage.segmentation.cron}")
    @Async
    public void mileageSegmentation() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        cal.add(Calendar.DATE, -1);
        String nowDate = new SimpleDateFormat("yyyy-MM-dd").format(cal.getTime());
        List<WlCarMileage> list = this.baseMapper.list(nowDate);

        list.forEach(item -> {
            redisTemplate.delete(item.getIotCode());
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(item.getDate());
            calendar.add(Calendar.DATE, 1);
            Date date = calendar.getTime();
            //iotcode有效
            if (!ObjectUtils.isEmpty(item.getIotCode()) && item.getIotCode().length() > 8) {
                // 查询车辆上报信息
                ResponseModel<List<Object>> result = iotFeign.getLiveData(item.getIotCode().substring(0, 8), item.getIotCode().substring(8),
                        item.getStartTime(), new Date(new Date().getTime() + 2000));
                List<Object> list1 = result.getResult();
                // 获取最后一个有坐标的数据
                JSONObject lastObj = null;
                // 过滤空坐标
                List<Object> filterList = new ArrayList<Object>();
                for (int i = 0; i < list1.size(); i++) {
                    JSONObject Obj = JSONObject.parseObject(JSONObject.toJSONString(list1.get(i)));
                    if (Obj.get("FireCar_Longitude") != null && Obj.get("FireCar_Latitude") != null
                            && Obj.getDoubleValue("FireCar_Longitude") != 0
                            && Obj.getDoubleValue("FireCar_Latitude") != 0) {
                        filterList.add(list1.get(i));
                        // 获取第一个不为空的坐标
                        if (lastObj == null) {
                            lastObj = Obj;
                        }
                    }
                }
                if (lastObj == null) {
                    lastObj = new JSONObject();
                    lastObj.put("FireCar_Longitude", 0.0);
                    lastObj.put("FireCar_Latitude", 0.0);
                    lastObj.put("time", 0);
                    lastObj.put("FireCar_Speed", 0);
                }
                // 当前坐标
                double startLongitude = lastObj.getDoubleValue("FireCar_Longitude");
                double startLatitude = lastObj.getDoubleValue("FireCar_Latitude");
                // 当前速度
                Double v = lastObj.getDoubleValue("FireCar_Speed");
                item.setStartSpeed(v.intValue());
                double travel = 0.0;
                // 获取里程
                for (int i = 0; i < filterList.size() - 1; i++) {
                    JSONObject start = JSONObject.parseObject(JSONObject.toJSONString(filterList.get(i)));
                    JSONObject end = JSONObject.parseObject(JSONObject.toJSONString(filterList.get(i + 1)));
                    travel += CoordinateUtil.distance(start.getDoubleValue("FireCar_Latitude"),
                            start.getDoubleValue("FireCar_Longitude"), end.getDoubleValue("FireCar_Latitude"),
                            end.getDoubleValue("FireCar_Longitude"));
                }

                // 里程耗时
                long takeTime = (date.getTime()) - (item.getStartTime().getTime());

                // 查询车辆最新位置

                // 修改0点未结束里程记录
                item.setEndSpeed(v.intValue());
                item.setEndTime(date);
                item.setEndLatitude(startLatitude);
                item.setEndLongitude(startLongitude);
                item.setEndName("无效坐标");
                if (travel > 0.5) {
                    String address = getAddress(startLongitude, startLatitude);
                    item.setEndName(address);
                }
                item.setTravel(new BigDecimal(travel / 1000).setScale(1, BigDecimal.ROUND_HALF_UP).doubleValue());
                item.setTakeTime(takeTime);
                this.getBaseMapper().updateById(item);
                }
        });
        //删除过期的告警数据
        Date endTime = DateUtil.offsetDay(new Date(),-15);
        String endTimeStr = DateUtil.format(endTime, "yyyy-MM-dd HH:mm:00");
        wlCarSpeedWaringRecordMapper.delete(new QueryWrapper<CarSpeedWarningRecord>().le("waring_date",endTimeStr));
    }

    @Scheduled(cron = "${mileage.splitmileage.cron}")
    @Async
    public void endCarMileageTask() {
        List<WlCarMileage> list = wlCarMileageMapper.selectList(new QueryWrapper<WlCarMileage>().isNull("end_time").isNotNull("start_time"));
        list.forEach(item -> {
            if (ObjectUtils.isEmpty(redisTemplate.opsForValue().get(item.getIotCode()))) {
                if (!ObjectUtils.isEmpty(item.getIotCode()) && item.getIotCode().length() > 8) {
                    // 查询车辆上报信息
                    Date date = new Date();
                    ResponseModel<List<Object>> result = iotFeign.getLiveData(item.getIotCode().substring(0, 8), item.getIotCode().substring(8),
                            item.getStartTime(), new Date(new Date().getTime() + 2000));
                    List<Object> list1 = result.getResult();
                    // 获取最后一个有坐标的数据
                    JSONObject lastObj = null;
                    // 过滤空坐标
                    List<Object> filterList = new ArrayList<Object>();
                    for (int i = 0; i < list1.size(); i++) {
                        JSONObject Obj = JSONObject.parseObject(JSONObject.toJSONString(list1.get(i)));
                        if (Obj.get("FireCar_Longitude") != null && Obj.get("FireCar_Latitude") != null
                                && Obj.getDoubleValue("FireCar_Longitude") != 0
                                && Obj.getDoubleValue("FireCar_Latitude") != 0) {
                            filterList.add(list1.get(i));
                            // 获取第一个不为空的坐标
                            if (lastObj == null) {
                                lastObj = Obj;
                            }
                        }
                    }
                    if (lastObj == null) {
                        lastObj = new JSONObject();
                        lastObj.put("FireCar_Longitude", 0.0);
                        lastObj.put("FireCar_Latitude", 0.0);
                        lastObj.put("time", 0);
                        lastObj.put("FireCar_Speed", 0);
                    }
                    // 当前坐标
                    double startLongitude = lastObj.getDoubleValue("FireCar_Longitude");
                    double startLatitude = lastObj.getDoubleValue("FireCar_Latitude");
                    // 当前速度
                    Double v = lastObj.getDoubleValue("FireCar_Speed");
                    item.setStartSpeed(v.intValue());
                    double travel = 0.0;
                    // 获取里程
                    for (int i = 0; i < filterList.size() - 1; i++) {
                        JSONObject start = JSONObject.parseObject(JSONObject.toJSONString(filterList.get(i)));
                        JSONObject end = JSONObject.parseObject(JSONObject.toJSONString(filterList.get(i + 1)));
                        travel += CoordinateUtil.distance(start.getDoubleValue("FireCar_Latitude"),
                                start.getDoubleValue("FireCar_Longitude"), end.getDoubleValue("FireCar_Latitude"),
                                end.getDoubleValue("FireCar_Longitude"));
                    }

                    // 里程耗时
                    long takeTime = (date.getTime()) - (item.getStartTime().getTime());
                    // 查询车辆最新位置
                    // 修改0点未结束里程记录
                    item.setEndSpeed(v.intValue());
                    item.setEndTime(date);
                    item.setEndLatitude(startLatitude);
                    item.setEndLongitude(startLongitude);
                    item.setEndName("无效坐标");
                    if (travel > 0.5) {
                        String address = getAddress(startLongitude, startLatitude);
                        item.setEndName(address);
                    }
                    item.setTravel(new BigDecimal(travel / 1000).setScale(1, BigDecimal.ROUND_HALF_UP).doubleValue());
                    item.setTakeTime(takeTime);
                    if (travel > 0.5) {
                        this.getBaseMapper().updateById(item);
                    } else {
                        this.getBaseMapper().deleteById(item.getId());
                    }
                    }
            }
        });

        //删除无效的告警数据
        Date endTime = DateUtil.offsetMinute(new Date(),-10);
        String endTimeStr = DateUtil.format(endTime, "yyyy-MM-dd HH:mm:00");
        wlCarSpeedWaringRecordMapper.delete(new QueryWrapper<CarSpeedWarningRecord>().le("waring_date",endTimeStr).lt("over_speed_count",5));
    }

    @Override
    public Boolean getUncompleteMileagByIotCode(String iotCode) {
        Integer integer = wlCarMileageMapper.getUncompleteMileagByIotCode(iotCode);
        return integer <= 0;
    }

    private String getAddress(double longitude, double lantitude) {
        StringBuilder api = new StringBuilder(GUIDE_ADDRESS_URL);
        api.append("key=").append(GUIDE_KEY).append("&location=").append(longitude).append(",").append(lantitude)
                .append("&radius=1000").append("&batch=false").append("&extensions=base").append("&roadlevel=0")
                .append("&batch=false");
        StringBuilder res = new StringBuilder();
        BufferedReader in = null;
        try {
            System.out.println(api.toString());
            URL url = new URL(api.toString());
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setDoOutput(true);
            conn.setRequestMethod("POST");
            in = new BufferedReader(new InputStreamReader(conn.getInputStream(), "UTF-8"));
            String line;
            while ((line = in.readLine()) != null) {
                res.append(line).append("\n");
            }
            JSONObject object = JSONObject.parseObject(res.toString());
            JSONObject regeocode = object.getJSONObject("regeocode");
            String address = regeocode.getString("formatted_address");
            if ("[]".equals(address)) {
                address = "无效坐标";
            }
            res = new StringBuilder(address);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return res.toString();
    }

    @Override
    public Double getTotalTravelByIotCodeAndDate(String date, String iotCode) {
        if (StringUtils.isEmpty(date)) {
            date = DateUtil.format(new Date(), "yyyy-MM");
        }
        Double totalTravel = 0.0;
        LambdaQueryWrapper<WlCarMileage> wrapper = new LambdaQueryWrapper<WlCarMileage>();
        wrapper.eq(WlCarMileage::getIotCode, iotCode);
        wrapper.ge(WlCarMileage::getTravel, 0.5);
        wrapper.isNotNull(WlCarMileage::getEndTime);
        wrapper.isNotNull(WlCarMileage::getStartTime);
        wrapper.like(WlCarMileage::getStartTime, date);
        List<WlCarMileage> list = this.list(wrapper);
        for (WlCarMileage wl : list) {
            //判断当前车辆全程是否异常,如果时速小于阈值 则对数据进行返回 否则不予返回
            if (!org.springframework.util.ObjectUtils.isEmpty(wl.getTakeTime()) && !org.springframework.util.ObjectUtils.isEmpty(wl.getTravel()) && ((wl.getTravel() / (wl.getTakeTime() / 3600000.0)) < CAR_MAX_SPEED)) {
                totalTravel += wl.getTravel();
            }
        }
        return totalTravel;
    }

    @Override
    public Double getTotalTravelByIotCodeAndDateProd(String startDate, String endDate, String iotCode) {
        Double totalTravel = 0.0;
        LambdaQueryWrapper<WlCarMileage> wrapper = new LambdaQueryWrapper<WlCarMileage>();
        wrapper.eq(WlCarMileage::getIotCode, iotCode);
        wrapper.ge(WlCarMileage::getTravel, 0.5);
        wrapper.isNotNull(WlCarMileage::getEndTime);
        wrapper.isNotNull(WlCarMileage::getStartTime);
        wrapper.ge(WlCarMileage::getDate, startDate);
        wrapper.le(WlCarMileage::getDate, endDate);
        List<WlCarMileage> list = this.list(wrapper);
        for (WlCarMileage wl : list) {
            //判断当前车辆全程是否异常,如果时速小于阈值 则对数据进行返回 否则不予返回
            if (!org.springframework.util.ObjectUtils.isEmpty(wl.getTakeTime()) && !org.springframework.util.ObjectUtils.isEmpty(wl.getTravel()) && ((wl.getTravel() / (wl.getTakeTime() / 3600000.0)) < CAR_MAX_SPEED)) {
                totalTravel += wl.getTravel();
            }
        }
        return totalTravel;
    }

    /**
     * @param objects        纠偏下之前的数据
     * @param startLatitude  纠偏之后的经度
     * @param startLongitude 纠偏之后的纬度
     * @return 当前数据应该展示的速度
     * @descrip 获取纠偏后数据应该展示的数据
     */
    public Double getSpeedByOriginalData(JSONArray objects, Double startLatitude, Double startLongitude) {
        Double speed = 0.0;
        for (int i = 0; i < objects.size() - 1; i++) {
            JSONObject jsonObject1 = objects.getJSONObject(i);
            JSONObject jsonObject2 = objects.getJSONObject(i + 1);

            Double travel1 = CoordinateUtil.distance(startLatitude, startLongitude, jsonObject1.getDoubleValue("x"), jsonObject1.getDoubleValue("y"));
            Double travel2 = CoordinateUtil.distance(startLatitude, startLongitude, jsonObject2.getDoubleValue("x"), jsonObject2.getDoubleValue("y"));

            if (travel2 > travel1) {
               speed = jsonObject1.getDoubleValue("sp");
                break;
            } else {
                speed = jsonObject2.getDoubleValue("sp");
            }
        }
        return speed;
    }

    /**
     * @param originalPoints 原始数据点位
     * @param aMapPoints     高德纠偏之后的数据点位
     * @return List<Coordinate> 处理之后的数据
     */
    public ArrayList<Coordinate> giveSpeedToCoordinate(JSONArray originalPoints, JSONArray aMapPoints) {
        ArrayList<Coordinate> result = new ArrayList<>();
        Double rate = originalPoints.size() / (aMapPoints.size() * 1.0);
        for (int i = 0; i < aMapPoints.size(); i++) {
            Integer speedIndex = Double.valueOf(i * rate).intValue();
            JSONObject jsonObject1 = JSONObject.parseObject(aMapPoints.get(i).toString());
            List<Double> doubles = new ArrayList<>();
            Coordinate coordinate = new Coordinate();
            doubles.add(Double.valueOf(jsonObject1.get("x").toString()));
            doubles.add(Double.valueOf(jsonObject1.get("y").toString()));
            coordinate.setLnglat(doubles);
            coordinate.setSpeed(originalPoints.getJSONObject(speedIndex).getDoubleValue("sp"));
            result.add(coordinate);
        }
        return result;
    }
}
