package com.yeejoin.amos.boot.module.ys.biz.service.impl;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.yeejoin.amos.boot.module.common.biz.service.impl.DynamicFormInstanceServiceImpl;
import com.yeejoin.amos.boot.module.jg.api.entity.SafetyProblemTracing;
import com.yeejoin.amos.boot.module.jg.api.mapper.SafetyProblemTracingMapper;
import com.yeejoin.amos.boot.module.ymt.api.enums.EquipmentClassifityEnum;
import com.yeejoin.amos.boot.module.ys.api.entity.YsEmergencyPlan;
import com.yeejoin.amos.boot.module.ys.api.entity.YsEmergencyRehearsal;
import com.yeejoin.amos.boot.module.ys.api.enums.MaintenanceExpiredEarlyWarningEnum;
import com.yeejoin.amos.boot.module.ys.api.vo.SortVo;
import com.yeejoin.amos.boot.module.ys.biz.service.CommonService;
import com.yeejoin.amos.feign.privilege.Privilege;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;
import org.apache.commons.lang3.RandomUtils;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.CountRequest;
import org.elasticsearch.client.core.CountResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 公共服务实现类
 *
 * @author system_generator
 * @date 2024-09-24
 */
@Service
public class CommonServiceImpl implements CommonService {

    /**
     * 气瓶设备类别
     */
    public final static String EQU_CATEGORY_CYLINDER = "2300";

    /**
     * 应急保障四项统计使用 - 饼图
     */
    public final static String PIECHART = "pieChart";
    /**
     * 应急保障四项统计使用 - 饼图模板
     */
    public final static String PIECHART_TEMPLATE = "[{\"name\": \"%s\",\"value\": %s},{\"name\": \"其他\",\"value\": %s}]";
    /**
     * 应急保障四项统计使用 - 富文本
     */
    public final static String RICHTEXT = "richText";
    /**
     * 应急保障四项统计使用 - 富文本模板
     */
    public final static String RICHTEXT_TEMPLATE = "{\"total\": %s,\"%s\": %s}";

    /**
     * 安全追溯-综合统计-严重违法 模板
     */
    public final static String SERIOUS_VIOLATION_OF_LAW = "{\"seriesData\": [%s, %s, %s],\"axisData\": [\"红码警示企业数\", \"红码警示设备数\", \"红码警示人员数\"]}";

    /**
     * 安全追溯-综合统计-异常名录 模板
     */
    public final static String EXCEPTION_DIRECTORY = "[{\"name\": \"问题企业数量\",\"value\": %s},{\"name\": \"非问题企业数量\",\"value\": %s}]";

    /**
     * 安全追溯-综合统计-信用监管 模板
     */
    public final static String CREDIT_SUPERVISION = "[{\"name\": \"绿码\",\"value\": %s},{\"name\": \"红码\",\"value\": %s},{\"name\": \"橙码\",\"value\": %s},{\"name\": \"灰码\",\"value\": %s}]";

    /**
     * 应急保障四项统计使用 - 业务类型
     */
    public final static JSONObject businessTypeJSON = new JSONObject();
    private static final Logger log = LoggerFactory.getLogger(CommonServiceImpl.class);

    @Autowired
    private RestHighLevelClient restHighLevelClient;

    @Autowired
    private DynamicFormInstanceServiceImpl dynamicFormInstanceService;

    @Autowired
    private YsEmergencyRehearsalServiceImpl ysEmergencyRehearsalService;

    @Autowired
    private YsEmergencyPlanServiceImpl ysEmergencyPlanService;

    @Autowired
    private SafetyProblemTracingMapper safetyProblemTracingMapper;

    static {
        businessTypeJSON
                .fluentPut("education", "教育培训数").fluentPut("construct", "体系建设数")
                .fluentPut("rehearsal", "应急演练数").fluentPut("prePlan", "应急预案数");
    }

    @Value("classpath:/json/analysisCityMapData.json")
    private Resource analysisCityMapData;

    private JsonNode jsonData;

    @PostConstruct
    public void init() {
        try {
            jsonData = new ObjectMapper().readTree(analysisCityMapData.getInputStream());
        } catch (IOException e) {
            throw new RuntimeException("Failed to read or parse JSON data", e);
        }
    }

    /**
     * 排序  ：页面列表排序功能支持，将 "字段,ascend" 或 "字段,descend" 转化为对应JSONObject
     *
     * @param sort "字段,ascend" 或 "字段,descend"
     * @return JSONObject
     */
    public SortVo sortFieldConversion(String sort) {
        Optional<String> optionalSort = Optional.ofNullable(sort);
        Optional<SortVo> optionalSortMap = optionalSort.filter(s -> !s.isEmpty())
                .map(s -> {
                    String[] sortParts = s.split(",");
                    if (sortParts.length == 2) {
                        String field = sortParts[0];
                        String sortSituation = sortParts[1].contains("asc") ? "ASC" : "DESC";
                        return SortVo.builder()
                                .field(convertToUnderline(field))
                                .sortType(sortSituation)
                                .build();
                    }
                    return null;
                });
        return optionalSortMap.orElse(null);
    }

    /**
     * 驼峰转下划线
     *
     * @param str
     * @return
     */
    public static String convertToUnderline(String str) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (Character.isUpperCase(c)) {
                sb.append("_").append(Character.toLowerCase(c));
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /**
     * 气瓶数量统计
     *
     * @param orgCode 区域code
     * @return
     */
    public long staticsCenterMapCountDataForCylinder(String orgCode) {
        long num = 0;
        CountRequest request = new CountRequest();
        request.indices("idx_biz_view_jg_all");
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        // 按照管辖机构区域信息模糊查询
        boolMust.must(QueryBuilders.wildcardQuery("ORG_BRANCH_CODE.keyword", QueryParser.escape(orgCode) + "*"));
        // 设备类别精确查询气瓶
        boolMust.must(QueryBuilders.termQuery("EQU_CATEGORY_CODE", EQU_CATEGORY_CYLINDER));
        // 只统计已纳管设备
        boolMust.must(QueryBuilders.termQuery("IS_INTO_MANAGEMENT", Boolean.TRUE));
        // 只统计有监管码的、状态不是草稿且不是已拒领的
        boolMust.must(QueryBuilders.existsQuery("SUPERVISORY_CODE"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "草稿"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "已拒领"));
        request.query(boolMust);
        try {
            CountResponse response = restHighLevelClient.count(request, RequestOptions.DEFAULT);
            num = response.getCount();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return num;
    }

    /**
     * 八大类设备数量统计
     *
     * @param equTypeCode
     * @param cylinderNum
     * @param orgCode
     * @return
     */
    public long staticsCenterMapCountDataForEquip(String equTypeCode, long cylinderNum, String orgCode) {
        AtomicLong result = new AtomicLong(0L);
        SearchRequest request = new SearchRequest();
        request.indices("idx_biz_view_jg_all");
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        // 按照管辖机构区域信息模糊查询
        boolMust.must(QueryBuilders.wildcardQuery("ORG_BRANCH_CODE.keyword", QueryParser.escape(orgCode) + "*"));
        // 只统计已纳管设备
        boolMust.must(QueryBuilders.termQuery("IS_INTO_MANAGEMENT", Boolean.TRUE));
        // 只统计有监管码的、状态不是草稿且不是已拒领的
        boolMust.must(QueryBuilders.existsQuery("SUPERVISORY_CODE"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "草稿"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "已拒领"));
        boolMust.must(QueryBuilders.termQuery("EQU_LIST_CODE", equTypeCode));
        SearchSourceBuilder builder = new SearchSourceBuilder();
        builder.query(boolMust);
        // 默认size为10 ，数据库有脏数据，所以需要多查询一些
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("count_by_equ_list_code").field("EQU_LIST_CODE").size(20);
        builder.aggregation(aggregationBuilder);
        request.source(builder);
        try {
            SearchResponse response = restHighLevelClient.search(request, RequestOptions.DEFAULT);
            Terms terms = response.getAggregations().get("count_by_equ_list_code");

            for (Terms.Bucket bucket : terms.getBuckets()) {
                if (equTypeCode.equals(bucket.getKeyAsString())) {
                    // 压力容器里包括气瓶，所以需要特殊处理
                    long count = bucket.getDocCount();
                    result.set(bucket.getKeyAsString().equals(EquipmentClassifityEnum.YLRQ.getCode())
                            ? count - cylinderNum
                            : count);
                    break;
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result.get();
    }

    /**
     * 应急保障四项统计
     *
     * @param businessType  业务类型（教育培训：education、体系建设：construct、应急演练：rehearsal、应急预案：prePlan）
     * @param componentType 组件类型（饼图：pieChart、富文本：richText）
     * @return 模板数据
     */
    @Override
    public Object emergencySupportStatistics(String businessType, String componentType) {
        // 教育培训数
        long educationNum = dynamicFormInstanceService.pageList(1, 1, MaintenanceExpiredEarlyWarningEnum.JYPX.getGroupCode()).getTotal();
        // 体系建设数
        long constructNum = dynamicFormInstanceService.pageList(1, 1, MaintenanceExpiredEarlyWarningEnum.TXJS.getGroupCode()).getTotal();
        // 应急演练数
        int rehearsalNum = ysEmergencyRehearsalService.getBaseMapper().selectCount(new LambdaQueryWrapper<YsEmergencyRehearsal>().eq(YsEmergencyRehearsal::getIsDelete, Boolean.FALSE));
        // 应急预案数
        int prePlanNum = ysEmergencyPlanService.getBaseMapper().selectCount(new LambdaQueryWrapper<YsEmergencyPlan>().eq(YsEmergencyPlan::getIsDelete, Boolean.FALSE));
        // 四项总数
        BigDecimal totalNum = NumberUtil.add(educationNum, constructNum, rehearsalNum, prePlanNum);
        // 其他数量
        BigDecimal otherNum = new BigDecimal(0);
        // 请求类型数量
        BigDecimal valueNum = new BigDecimal(0);
        switch (businessType) {
            case "education":
                valueNum = BigDecimal.valueOf(educationNum);
                otherNum = NumberUtil.sub(totalNum, educationNum);
                break;
            case "construct":
                valueNum = BigDecimal.valueOf(constructNum);
                otherNum = NumberUtil.sub(totalNum, constructNum);
                break;
            case "rehearsal":
                valueNum = BigDecimal.valueOf(rehearsalNum);
                otherNum = NumberUtil.sub(totalNum, rehearsalNum);
                break;
            case "prePlan":
                valueNum = BigDecimal.valueOf(prePlanNum);
                otherNum = NumberUtil.sub(totalNum, prePlanNum);
                break;
        }
        if (PIECHART.equals(componentType)) {
            // 组装 饼图 模板数据
            return JSONArray.parseArray(String.format(PIECHART_TEMPLATE, businessTypeJSON.get(businessType).toString(), valueNum, otherNum));
        }
        if (RICHTEXT.equals(componentType)) {
            // 组装 富文本 模板数据
            return JSONObject.parseObject(String.format(RICHTEXT_TEMPLATE, totalNum, businessType, valueNum));
        }
        return null;
    }

    /**
     * 安全追溯-综合分析统计
     *
     * @param statisticalType 统计类型 creditSupervision（信用监管） 、 exceptionDirectory（异常名录）、seriousViolationOfLaw（严重违法）
     * @return result
     */
    @Override
    public Object safetyTraceabilitySynthesis(String statisticalType) {
        // 严重违法 统计红码的企业、设备、人员数量
        if ("seriousViolationOfLaw".equals(statisticalType)) {
            // 红码警示企业数
            long comp = safetyProblemTracingMapper.selectCount(new LambdaQueryWrapper<SafetyProblemTracing>()
                    .eq(SafetyProblemTracing::getProblemStatusCode, 0)
                    .eq(SafetyProblemTracing::getSourceTypeCode, 2));
            // 红码警示设备数
            long equ = safetyProblemTracingMapper.selectCount(new LambdaQueryWrapper<SafetyProblemTracing>()
                    .eq(SafetyProblemTracing::getProblemStatusCode, 0)
                    .eq(SafetyProblemTracing::getSourceTypeCode, 3));
            // 红码警示人员数
            long peo = safetyProblemTracingMapper.selectCount(new LambdaQueryWrapper<SafetyProblemTracing>()
                    .eq(SafetyProblemTracing::getProblemStatusCode, 0)
                    .eq(SafetyProblemTracing::getSourceTypeCode, 1));
            return JSONObject.parseObject(String.format(SERIOUS_VIOLATION_OF_LAW, comp, equ, peo));
        }
        // 异常名录 统计企业主体问题数量
        if ("exceptionDirectory".equals(statisticalType)) {
            // 问题企业数量
            long comp = safetyProblemTracingMapper.selectList(new LambdaQueryWrapper<SafetyProblemTracing>()
                            .eq(SafetyProblemTracing::getProblemStatusCode, 0)
                            .eq(SafetyProblemTracing::getSourceTypeCode, 2)
                            .groupBy(SafetyProblemTracing::getPrincipalUnitCode))
                    .size();
            // 非问题企业数量 (企业总数-问题企业数量)
            long total = Privilege.companyClient.queryForCompanyList(null, null).getResult().size();
            long other = (long) NumberUtil.sub(total, comp);
            return JSONArray.parseArray(String.format(EXCEPTION_DIRECTORY, comp, other));
        }
        // 信用监管 统计绿、红、橙、灰码数量及占比
        if ("creditSupervision".equals(statisticalType)) {
            // 绿码
            long greenCode = safetyProblemTracingMapper.selectCount(new LambdaQueryWrapper<SafetyProblemTracing>()
                    .eq(SafetyProblemTracing::getProblemStatusCode, 1));
            // 红码
            long redCode = safetyProblemTracingMapper.selectCount(new LambdaQueryWrapper<SafetyProblemTracing>()
                    .eq(SafetyProblemTracing::getProblemStatusCode, 0));
            // 橙码
            long orangeCode = 0L;
            // 灰码
            long grayCode = 0L;
            return JSONArray.parseArray(String.format(CREDIT_SUPERVISION, greenCode, redCode, orangeCode, grayCode));
        }
        return null;
    }

    @Override
    public Object analysisCityMapData(String firstMenuKey, String secondMenuKey) {
        return new JSONObject().fluentPut("records", Collections.singletonList(jsonData.get(firstMenuKey).get(secondMenuKey)));
    }

    private static final Random random = new Random();

    public static List<Integer> generateIntegers(int min, int max, int count) {
        return random.ints(count, min, max + 1)
                .boxed()
                .collect(Collectors.toList());
    }

    private static Result generatingFakeData() {
        List<Integer> allNumbers = generateIntegers(0, 3, 24);

        List<Integer> lineData = allNumbers.subList(0, 8);
        List<Integer> barData1 = allNumbers.subList(8, 16);
        List<Integer> barData2 = allNumbers.subList(16, 24);

        String lineDataStr = lineData.stream().map(String::valueOf).collect(Collectors.joining(","));
        String barData1Str = barData1.stream().map(String::valueOf).collect(Collectors.joining(","));
        String barData2Str = barData2.stream().map(String::valueOf).collect(Collectors.joining(","));
        return new Result(lineDataStr, barData1Str, barData2Str);
    }

    @Data
    @Getter
    private static class Result {
        private String lineDataStr;
        private String barData1Str;
        private String barData2Str;

        public Result(String lineDataStr, String barData1Str, String barData2Str) {
            this.lineDataStr = lineDataStr;
            this.barData1Str = barData1Str;
            this.barData2Str = barData2Str;
        }
    }

    @Override
    public Object analysisTwoLists(String type) {
        Result fakeData = generatingFakeData();
        String twoListsTemplate = "{\"seriesData\":[{\"data\":[%s],\"name\":\"正常点\",\"type\":\"line\",\"key\":\"default\",\"yAxisIndex\":1,\"smooth\":false},{\"barWidth\":18,\"stack\":\"巡检点\",\"data\":[%s],\"name\":\"异常点\",\"type\":\"bar\",\"key\":\"default\",\"smooth\":false},{\"barWidth\":18,\"stack\":\"巡检点\",\"data\":[%s],\"type\":\"bar\",\"key\":\"default\",\"smooth\":false}],\"axisData\":[\"使用\\n单位\",\"安全检\\n察部门\",\"专委会\\n成员单\",\"检验检\\n测机构\",\"人员考\\n试机构\",\"行业\\n协会\",\"生产\\n单位\",\"经营\\n单位\"]}";
        String result = String.format(twoListsTemplate, fakeData.getLineDataStr(), fakeData.getBarData1Str(), fakeData.getBarData2Str());
        return JSONObject.parseObject(result);
    }

    @Override
    public Object analysisSpecialRectification(String type) {
        Result fakeData = generatingFakeData();
        String specialRectificationTemplate = "{\"seriesData\":[{\"data\":[%s],\"name\":\"正常点\",\"type\":\"line\",\"key\":\"default\",\"yAxisIndex\":1,\"smooth\":false},{\"barWidth\":18,\"stack\":\"巡检点\",\"data\":[%s],\"name\":\"异常点\",\"type\":\"bar\",\"key\":\"default\",\"smooth\":false},{\"barWidth\":18,\"stack\":\"巡检点\",\"data\":[%s],\"type\":\"bar\",\"key\":\"default\",\"smooth\":false}],\"axisData\":[\"电梯\",\"锅炉\",\"压力\\n容器\",\"压力\\n管道\",\"大型游\\n乐设施\",\"索道\",\"场内机\\n动车\",\"起重\\n机械\"]}";
        String result = String.format(specialRectificationTemplate, fakeData.getLineDataStr(), fakeData.getBarData1Str(), fakeData.getBarData2Str());
        return JSONObject.parseObject(result);
    }

    @Override
    public Object analysisSafetySupervision(String type) {
        String template = "{\"seriesData\":[%s]}";
        return JSONObject.parseObject(String.format(template, "0".equals(type) ? "88.3" : "92.1"));
    }
}