package com.yeejoin.amos.boot.module.common.api.converter;

import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.property.ExcelContentProperty;
import com.google.common.collect.Lists;
import com.yeejoin.amos.boot.biz.common.utils.RedisUtils;
import org.springframework.stereotype.Component;
import org.typroject.tyboot.core.foundation.utils.ValidationUtil;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;

/**
 * 通用的自定义字段转换器
 * 用于处理Excel导入导出时的字段转换逻辑
 */
@Component
public class CommonCustomConverter {

    private static RedisUtils redisUtils;

    public CommonCustomConverter(RedisUtils redisUtils) {
        CommonCustomConverter.redisUtils = redisUtils;
    }

    /**
     * 年月日格式转换器
     */
    public static class Y_M_D_DateConverter implements Converter<Date> {

        @Override
        public Class supportJavaTypeKey() {
            return null;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return null;
        }

        @Override
        public Date convertToJavaData(CellData cellData, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) throws Exception {
            return null;
        }

        @Override
        public CellData convertToExcelData(Date value, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) throws Exception {
            if (value == null) {
                return new CellData("");
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日");
            return new CellData(sdf.format(value));
        }
    }

    public static class CompanyCodeConverter implements Converter<String> {

        public static List<LinkedHashMap<String, Object>> cacheResult = Lists.newArrayList();

        // 使用静态初始化块加载数据字典
        static {
            refreshCache();
        }

        /**
         * 刷新数据缓存
         */
        @SuppressWarnings("unchecked")
        public static void refreshCache() {
            cacheResult = (List<LinkedHashMap<String, Object>>) redisUtils.get("ADMINISTRATION_UNIT_TREE");
        }

        @SuppressWarnings("unchecked")
        private static LinkedHashMap<String, Object> findCompanyByCode(List<LinkedHashMap<String, Object>> tree, String companyCode) {
            for (LinkedHashMap<String, Object> node : tree) {
                if (node.get("companyCode").equals(companyCode)) {
                    return node;
                }
                if (node.containsKey("children") && node.get("children") instanceof List) {
                    LinkedHashMap<String, Object> foundNode = findCompanyByCode((List<LinkedHashMap<String, Object>>) node.get("children"), companyCode);
                    if (foundNode != null) {
                        return foundNode;
                    }
                }
            }
            return null;
        }

        private static LinkedHashMap<String, Object> findCompanyByPath(String path) {
            String[] pathElements = path.split("#");
            return findCompanyByPathRecursive(cacheResult, pathElements, 0);
        }

        @SuppressWarnings("unchecked")
        private static LinkedHashMap<String, Object> findCompanyByPathRecursive(List<LinkedHashMap<String, Object>> tree, String[] pathElements, int index) {
            if (index >= pathElements.length) {
                return null;
            }

            String currentName = pathElements[index];
            for (LinkedHashMap<String, Object> node : tree) {
                if (currentName.equals(node.get("companyName"))) {
                    // 如果是路径的最后一个元素，直接返回该节点
                    if (index == pathElements.length - 1) {
                        return node;
                    }
                    // 如果不是最后一个元素，继续在子节点中查找
                    if (node.containsKey("children") && node.get("children") instanceof List) {
                        return findCompanyByPathRecursive((List<LinkedHashMap<String, Object>>) node.get("children"), pathElements, index + 1);
                    }
                    return null;
                }
            }
            return null;
        }

        private static LinkedHashMap<String, Object> findAllCompaniesByPartialPath(String path) {
            List<LinkedHashMap<String, Object>> result = new ArrayList<>();
            String[] pathElements = path.split("#");
            findAllCompaniesByPartialPathRecursive(cacheResult, pathElements, 0, result);
            return !result.isEmpty() ? result.get(0) : null;
        }

        @SuppressWarnings("unchecked")
        private static void findAllCompaniesByPartialPathRecursive(List<LinkedHashMap<String, Object>> tree, String[] pathElements, int index, List<LinkedHashMap<String, Object>> result) {
            if (index >= pathElements.length) {
                return;
            }

            String currentName = pathElements[index];
            for (LinkedHashMap<String, Object> node : tree) {
                if (currentName.equals(node.get("companyName"))) {
                    // 如果是路径的最后一个元素，添加该节点到结果中
                    if (index == pathElements.length - 1) {
                        result.add(node);
                    }
                    // 如果不是最后一个元素，继续在子节点中查找
                    if (node.containsKey("children") && node.get("children") instanceof List) {
                        findAllCompaniesByPartialPathRecursive((List<LinkedHashMap<String, Object>>) node.get("children"), pathElements, index + 1, result);
                    }
                }
                // 继续在子节点中查找起始路径
                if (node.containsKey("children") && node.get("children") instanceof List) {
                    findAllCompaniesByPartialPathRecursive((List<LinkedHashMap<String, Object>>) node.get("children"), pathElements, index, result);
                }
            }
        }

        public static String getCompanyCodeByName(String name) {
            if (ValidationUtil.isEmpty(name)) {
                return "";
            }
            LinkedHashMap<String, Object> company = findAllCompaniesByPartialPath(name);
            if (company == null) {
                return "610100"; // 针对西安电梯导入没有属地监管部门则默认给西安市场监督管理局
            }
            return company.get("companyCode").toString();
        }

        public static String getCompanyOrgCodeByName(String name) {
            LinkedHashMap<String, Object> company = findAllCompaniesByPartialPath(name);
            if (company == null) {
                return "";
            }
            return company.get("orgCode").toString();
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellDataStr = cellData.getStringValue();
            return CompanyCodeConverter.getCompanyCodeByName(cellDataStr);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            return null;
        }

        public static String getNameByCode(String code) {
            LinkedHashMap<String, Object> company = findCompanyByCode(cacheResult, code);
            if (company == null) {
                return "";
            }
            return company.get("companyName").toString();
        }
    }
}
