package com.yeejoin.equipmanage.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yeejoin.equipmanage.common.dto.AnalysisReportLogDto;
import com.yeejoin.equipmanage.common.entity.AnalysisReportLog;
import com.yeejoin.equipmanage.common.entity.FireFightingSystemEntity;
import com.yeejoin.equipmanage.common.entity.publics.BaseEntity;
import com.yeejoin.equipmanage.common.enums.AnalysisReportEnum;
import com.yeejoin.equipmanage.common.utils.DateUtils;
import com.yeejoin.equipmanage.mapper.AnalysisReportLogMapper;
import com.yeejoin.equipmanage.mapper.AnalysisReportMonthMapper;
import com.yeejoin.equipmanage.mapper.AnalysisReportSummaryMapper;
import com.yeejoin.equipmanage.service.IAnalysisReportLogService;
import com.yeejoin.equipmanage.service.IFireFightingSystemService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.util.*;

/**
 * 报告流水表 服务实现类
 *
 * @author duanwei
 * @date 2021-05-26
 */
@Service
public class AnalysisReportLogServiceImpl extends ServiceImpl<AnalysisReportLogMapper, AnalysisReportLog> implements IAnalysisReportLogService {

    @Autowired
    IFireFightingSystemService fireFightingSystemService;

    @Autowired
    AnalysisReportMonthMapper analysisReportMonthMapper;

    @Autowired
    AnalysisReportSummaryMapper analysisReportSummaryMapper;

    @Autowired
    AnalysisReportLogMapper analysisReportLogMapper;

    @Override
    public IPage<AnalysisReportLog> listPage(Page page, AnalysisReportLog analysisReportLog) {

        LambdaQueryWrapper<AnalysisReportLog> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(AnalysisReportLog::getReportType, analysisReportLog.getReportType());
        if (analysisReportLog.getStartDate() != null && analysisReportLog.getEndDate() != null) {
            queryWrapper.ge(AnalysisReportLog::getStartDate, analysisReportLog.getStartDate());
            queryWrapper.le(AnalysisReportLog::getEndDate, analysisReportLog.getEndDate());
        }
        queryWrapper.select().orderByDesc(BaseEntity::getCreateDate);

        return this.baseMapper.selectPage(page, queryWrapper);
    }

    @Override
    public void generateReport(AnalysisReportEnum reportEnum, Date beginDate, Date endDate) {
        this.saveAnalysisReportLog(reportEnum, beginDate, endDate);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void generateMonthReport(AnalysisReportEnum reportEnum, Date beginDate, Date endDate) throws ParseException {
        // 0.保存日志
        this.saveAnalysisReportLog(reportEnum, beginDate, endDate);
        // 创建月分析统计报告数据
        // 1、 查询消防系统表，捞出所有系统，新增字段，存放自定义用的告警指标模糊查询指标key，逗号分隔
        List<FireFightingSystemEntity> fightingSystemEntityList = fireFightingSystemService.getBaseMapper().selectList(
                new LambdaQueryWrapper<FireFightingSystemEntity>()
                        .isNotNull(FireFightingSystemEntity::getAnalysisIndexKey));
        // 2、循环插入 wl_analysis_report_month、wl_analysis_report_summary
        String beginDateStr = DateUtils.dateFormat(beginDate,DateUtils.DATE_PATTERN);
        String endDateStr = DateUtils.dateFormat(endDate,DateUtils.DATE_PATTERN);
        fightingSystemEntityList.forEach(f -> {
            analysisReportMonthMapper.insertSystemMonthData(new ArrayList<>(Arrays.asList(f.getAnalysisIndexKey().split(","))), beginDateStr, endDateStr, f.getId());
            analysisReportSummaryMapper.insertSystemMonthSummaryData(new ArrayList<>(Arrays.asList(f.getAnalysisIndexKey().split(","))), beginDateStr, endDateStr, f.getId());
        });
    }

    private void saveAnalysisReportLog(AnalysisReportEnum reportEnum, Date beginDate, Date endDate) {
        AnalysisReportLog reportLog = new AnalysisReportLog();
        reportLog.setName(reportEnum.getName());
        reportLog.setReportType(reportEnum.getType());
        reportLog.setStartDate(beginDate);
        reportLog.setEndDate(endDate);
        reportLog.setReportDateDesc(buildReportDateDesc(reportEnum, beginDate, endDate));
        this.save(reportLog);
    }

    private String buildReportDateDesc(AnalysisReportEnum reportEnum, Date beginDate, Date endDate) {
        StringBuilder desc = new StringBuilder();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(beginDate);
        switch (reportEnum) {
            case DAY_REPORT:
                desc.append(calendar.get(Calendar.YEAR));
                desc.append("年");
                desc.append(calendar.get(Calendar.MONTH) + 1);
                desc.append("月");
                desc.append(calendar.get(Calendar.DAY_OF_MONTH));
                desc.append("日");
                break;
            case WEEK_REPORT:
                desc.append(calendar.get(Calendar.YEAR));
                desc.append("年");
                desc.append(calendar.get(Calendar.MONTH) + 1);
                desc.append("月");
                desc.append(calendar.get(Calendar.DAY_OF_MONTH));
                desc.append(" - ");
                calendar.setTime(endDate);
                desc.append(calendar.get(Calendar.YEAR));
                desc.append("年");
                desc.append(calendar.get(Calendar.MONTH) + 1);
                desc.append("月");
                desc.append(calendar.get(Calendar.DAY_OF_MONTH));
                desc.append("日");
                break;
            case MONTH_REPORT:
                desc.append(calendar.get(Calendar.YEAR));
                desc.append("年");
                desc.append(calendar.get(Calendar.MONTH) + 1);
                desc.append("月");
                break;
            default:
                break;
        }
        return desc.toString();
    }

    @Override
    public IPage<AnalysisReportLogDto> iotReportPage(Page page, AnalysisReportLogDto dto) {
        return analysisReportLogMapper.iotReportPage(page, dto);
    }
}
