package com.yeejoin.amos.boot.module.common.biz.refresh.listener;

import com.alibaba.fastjson.JSONObject;
import com.yeejoin.amos.boot.module.common.api.entity.TzsDataRefreshMessage;
import com.yeejoin.amos.boot.module.common.api.service.IDataRefreshDispatch;
import com.yeejoin.amos.boot.module.common.api.service.IDataRefreshHandler;
import com.yeejoin.amos.boot.module.common.biz.constats.Constants;
import com.yeejoin.amos.boot.module.common.biz.refresh.DataRefreshEvent;
import com.yeejoin.amos.boot.module.common.biz.refresh.factory.RefreshHandlerFactory;
import com.yeejoin.amos.boot.module.common.biz.service.impl.TzsDataRefreshMessageServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.event.TransactionalEventListener;

import javax.annotation.PostConstruct;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

@Component
@Slf4j
@RequiredArgsConstructor
public class DataRefreshListener {

    @Value("${data.opEvent.deal.thread.number:1}")
    private int threadNumber;

    private final BlockingQueue<DataRefreshEvent> queue = new LinkedBlockingQueue<>();

    private ExecutorService executorService;

    private final IDataRefreshDispatch dataRefreshService;

    private final RefreshHandlerFactory refreshHandlerFactory;

    private final TzsDataRefreshMessageServiceImpl tzsDataRefreshMessageService;

    @TransactionalEventListener(value = DataRefreshEvent.class)
    @Async
    public void onEquipCreateOrEdit(DataRefreshEvent event) {
        log.info("收到数据刷新消息:{}", JSONObject.toJSONString(event));
        queue.add(event);
    }

    @PostConstruct
    public void init() {
        executorService = Executors.newFixedThreadPool(threadNumber);
        IntStream.range(0, threadNumber).forEach(i -> {
            executorService.execute(() -> {
                while (true) {
                    try {
                        DataRefreshEvent event = queue.take();
                        processEvent(event);
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                    }
                }
            });
        });
    }


    private void processEvent(DataRefreshEvent event) {
        List<IDataRefreshHandler> handlers = refreshHandlerFactory.getRefreshHandler(event.getDataType());
        if (!handlers.isEmpty()) {
            // 1.记录 message
            List<TzsDataRefreshMessage> messages = createMsg(event);
            // 2.调用更新处理-异步
            dataRefreshService.doDispatch(handlers, messages);
        }
    }

    public List<TzsDataRefreshMessage> createMsg(DataRefreshEvent event) {
        List<TzsDataRefreshMessage> messages = event.getDataIds().stream().filter(Objects::nonNull).map(dataId -> {
            TzsDataRefreshMessage message = new TzsDataRefreshMessage();
            message.setDataId(dataId);
            message.setDataType(event.getDataType());
            message.setOperation(event.getOperation().name());
            message.setRecDate(new Date());
            message.setStatus(Constants.REFRESH_STATUS_DEALING); // 流程中
            return message;
        }).collect(Collectors.toList());
        tzsDataRefreshMessageService.saveBatch(messages);
        return messages;
    }
}
