package com.yeejoin.amos.speech.util;

import com.alibaba.nls.client.protocol.InputFormatEnum;
import com.alibaba.nls.client.protocol.NlsClient;
import com.alibaba.nls.client.protocol.SampleRateEnum;
import com.alibaba.nls.client.protocol.asr.SpeechTranscriber;
import com.alibaba.nls.client.protocol.asr.SpeechTranscriberListener;
import com.alibaba.nls.client.protocol.asr.SpeechTranscriberResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import static com.yeejoin.amos.speech.util.SpeechUtil.AppKey;


/**
 * 实时语音识别
 * <p>
 * 支持的输入格式：PCM（无压缩的PCM或WAV文件）、16 bit采样位数、单声道（mono）。
 * 支持的音频采样率：8000 Hz和16000 Hz。
 */
public class SpeechTranscriberDemo {
    private final NlsClient client;
    private final File speechFile;
    private static final Logger logger = LoggerFactory.getLogger(SpeechTranscriberDemo.class);

    public static void main(String[] args) {
        //本案例使用本地文件模拟发送实时流数据。您在实际使用时，可以实时采集或接收语音流并发送到ASR服务端。
        String fileLink = "https://gw.alipayobjects.com/os/bmw-prod/0574ee2e-f494-45a5-820f-63aee583045a.wav";
        //将上面fileLink文件下载到本地后，替换filepath为本地地址测试
        String filepath = "D:\\ffmpeg-4.4-full_build-shared\\bin\\test1.wav";
        SpeechTranscriberDemo transcriberDemo;
        try {
            transcriberDemo = new SpeechTranscriberDemo(new File(filepath));
            transcriberDemo.process();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public SpeechTranscriberDemo(File speechFile) throws IOException {
        if (speechFile != null && speechFile.exists() && speechFile.isFile()) {
            this.speechFile = speechFile;
            // 默认值：wss://nls-gateway.cn-shanghai.aliyuncs.com/ws/v1。
            //应用全局创建一个NlsClient实例，默认服务地址为阿里云线上服务地址。
            client = new NlsClient(SpeechUtil.getToken());
        } else {
            throw new IllegalArgumentException("识别文件异常");
        }
    }

    //根据二进制数据大小计算对应的同等语音长度。
    //sampleRate：支持8000或16000。
    public static int getSleepDelta(int dataSize, int sampleRate) {
        // 仅支持16位采样。
        int sampleBytes = 16;
        // 仅支持单通道。
        int soundChannel = 1;
        return (dataSize * 10 * 8000) / (160 * sampleRate);
    }

    public void process() {
        SpeechTranscriber transcriber = null;
        try {
            //创建实例、建立连接。
            transcriber = new SpeechTranscriber(client, new SpeechTranscriberListener() {
                /**
                 * 语音识别过程中返回的结果。仅当setEnableIntermediateResult为true时，才会返回该消息。
                 */
                @Override
                public void onTranscriptionResultChange(SpeechTranscriberResponse response) {
                    logger.warn("语音识别过程中返回的结果");
                    logger.warn("task_id: " + response.getTaskId() +
                            ", name: " + response.getName() +
                            //状态码“20000000”表示正常识别。
                            ", status: " + response.getStatus() +
                            //句子编号，从1开始递增。
                            ", index: " + response.getTransSentenceIndex() +
                            //当前的识别结果。
                            ", result: " + response.getTransSentenceText() +
                            //当前已处理的音频时长，单位为毫秒。
                            ", time: " + response.getTransSentenceTime());
                }

                /**
                 * 服务端准备好了进行识别
                 */
                @Override
                public void onTranscriberStart(SpeechTranscriberResponse response) {
                    logger.warn("服务端准备好了进行识别");
                    logger.warn("task_id: " + response.getTaskId()
                            + ", name: " + response.getName()
                            + ", status: " + response.getStatus());
                }

                /**
                 * 服务端检测到了一句话的开始
                 */
                @Override
                public void onSentenceBegin(SpeechTranscriberResponse response) {
                    logger.warn("服务端检测到了一句话的开始");
                    logger.warn("task_id: " + response.getTaskId()
                            + ", name: " + response.getName()
                            + ", status: " + response.getStatus());

                }

                /**
                 * 服务端检测到了一句话的结束
                 * 识别出一句话。服务端会智能断句，当识别到一句话结束时会返回此消息。
                 */
                @Override
                public void onSentenceEnd(SpeechTranscriberResponse response) {
                    logger.warn("服务端检测到了一句话的结束");
                    logger.warn("task_id: " + response.getTaskId() +
                            ", name: " + response.getName() +
                            //状态码“20000000”表示正常识别。
                            ", status: " + response.getStatus() +
                            //句子编号，从1开始递增。
                            ", index: " + response.getTransSentenceIndex() +
                            //当前的识别结果。
                            ", result: " + response.getTransSentenceText() +
                            //置信度
                            ", confidence: " + response.getConfidence() +
                            //开始时间
                            ", begin_time: " + response.getSentenceBeginTime() +
                            //当前已处理的音频时长，单位为毫秒。
                            ", time: " + response.getTransSentenceTime());
                }

                /**
                 * 识别结束后返回的最终结果
                 */
                @Override
                public void onTranscriptionComplete(SpeechTranscriberResponse response) {
                    logger.warn("识别结束后返回的最终结果");
                    logger.warn("task_id: " + response.getTaskId()
                            + ", name: " + response.getName()
                            + ", status: " + response.getStatus()
                            + ",result:" + response.getTransSentenceText());
                }

                /**
                 * 失败处理
                 */
                @Override
                public void onFail(SpeechTranscriberResponse response) {
                    logger.error("失败处理");
                    logger.error("task_id: " + response.getTaskId()
                            + ", status: " + response.getStatus()
                            + ", status_text: " + response.getStatusText());
                }
            });
            transcriber.setAppKey(AppKey);
            //输入音频编码方式。
            transcriber.setFormat(InputFormatEnum.PCM);
            //输入音频采样率。
            transcriber.setSampleRate(SampleRateEnum.SAMPLE_RATE_16K);
            //是否返回中间识别结果。
            transcriber.setEnableIntermediateResult(true);
            //是否生成并返回标点符号。
            transcriber.setEnablePunctuation(true);
            //是否将返回结果规整化，比如将一百返回为100。
            transcriber.setEnableITN(false);
            //设置vad断句参数。默认值：800ms，有效值：200ms～2000ms。
            //transcriber.addCustomedParam("max_sentence_silence", 600);
            //设置是否语义断句。
            //transcriber.addCustomedParam("enable_semantic_sentence_detection",false);
            //设置是否开启顺滑。
            //transcriber.addCustomedParam("disfluency",true);
            //设置是否开启词模式。
            //transcriber.addCustomedParam("enable_words",true);
            //设置vad噪音阈值参数，参数取值为-1～+1，如-0.9、-0.8、0.2、0.9。
            //取值越趋于-1，判定为语音的概率越大，亦即有可能更多噪声被当成语音被误识别。
            //取值越趋于+1，判定为噪音的越多，亦即有可能更多语音段被当成噪音被拒绝识别。
            //该参数属高级参数，调整需慎重和重点测试。
            //transcriber.addCustomedParam("speech_noise_threshold",0.3);
            //设置训练后的定制语言模型id。
            //transcriber.addCustomedParam("customization_id","你的定制语言模型id");
            //设置训练后的定制热词id。
            //transcriber.addCustomedParam("vocabulary_id","你的定制热词id");
            //设置是否忽略单句超时。
            transcriber.addCustomedParam("enable_ignore_sentence_timeout", false);
            //vad断句开启后处理。
            //transcriber.addCustomedParam("enable_vad_unify_post",false);
            //此方法将以上参数设置序列化为JSON发送给服务端，并等待服务端确认。
            transcriber.start();
            FileInputStream fis = new FileInputStream(speechFile);
            byte[] b = new byte[3200];
            int len;
            while ((len = fis.read(b)) > 0) {
                logger.info("send data pack length: " + len);
                transcriber.send(b, len);
                //本案例用读取本地文件的形式模拟实时获取语音流并发送的，因为读取速度较快，这里需要设置sleep。
                //如果实时获取语音则无需设置sleep, 如果是8k采样率语音第二个参数设置为8000。
                int deltaSleep = getSleepDelta(len, 16000);
                TimeUnit.MILLISECONDS.sleep(deltaSleep);
            }
            //通知服务端语音数据发送完毕，等待服务端处理完成。
            long now = System.currentTimeMillis();
            logger.info("ASR wait for complete");
            transcriber.stop();
            logger.info("ASR latency : " + (System.currentTimeMillis() - now) + " ms");
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (null != transcriber) {
                transcriber.close();
            }
            if (client != null) {
                client.shutdown();
            }
        }
    }
}
