package com.yeejoin.amos.maintenance.business.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.typroject.tyboot.core.foundation.enumeration.UserType;
import org.typroject.tyboot.core.restful.doc.TycloudOperation;

import com.yeejoin.amos.boot.biz.common.bo.ReginParams;
import com.yeejoin.amos.maintenance.business.param.RoutePageParam;
import com.yeejoin.amos.maintenance.business.service.intfc.IRouteService;
import com.yeejoin.amos.maintenance.business.util.CommonResponse;
import com.yeejoin.amos.maintenance.business.util.CommonResponseUtil;
import com.yeejoin.amos.maintenance.business.util.RoutePageParamUtil;
import com.yeejoin.amos.maintenance.core.common.request.CommonPageable;
import com.yeejoin.amos.maintenance.core.common.request.CommonRequest;
import com.yeejoin.amos.maintenance.core.common.request.RoutePointInputItemRequest;
import com.yeejoin.amos.maintenance.dao.entity.InputItem;
import com.yeejoin.amos.maintenance.dao.entity.Plan;
import com.yeejoin.amos.maintenance.dao.entity.Route;
import com.yeejoin.amos.maintenance.dao.entity.RoutePoint;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@RestController
@RequestMapping(value = "/api/route")
@Api(tags = "巡检路线api")
public class RouteController extends AbstractBaseController {

    private final Logger log = LoggerFactory.getLogger(RouteController.class);

    @Autowired
    private IRouteService routeService;

    /**
     * 增加巡检路线
     *
     * @param route 路线
     * @return CommonResponse
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "新增巡检路线", notes = "新增巡检路线")
    @PostMapping(value = "/addRoute", produces = "application/json;charset=UTF-8")
    public CommonResponse addRoute(@ApiParam(value = "巡检路线", required = true) @RequestBody Route route) {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            String loginOrgCode = getOrgCode(reginParams);
            route.setOrgCode(loginOrgCode);
            route.setIsDelete(false);
            if (routeService.existRouteName(loginOrgCode, route.getName())) {
                return CommonResponseUtil.failure("巡检路线名称重复");
            }
            route.setCreatorId(getUserId());
            routeService.addRoute(route);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("巡检路线新增失败");
        }
    }

    /**
     * 更新巡检路线
     *
     * @param route
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "更新巡检路线", notes = "更新巡检路线")
    @PutMapping(value = "/updateRoute", produces = "application/json;charset=UTF-8")
    public CommonResponse updateRoute(@ApiParam(value = "巡检路线", required = true) @RequestBody Route route) {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            String orgCode = getOrgCode(reginParams);
            route.setOrgCode(orgCode);
            route.setCreatorId(getUserId());
            routeService.updateRoute(route);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("巡检路线更新失败");
        }
    }

    /**
     * 删除巡检路线
     *
     * @param routeIds
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "删除巡检路线", notes = "删除巡检路线")
    @DeleteMapping(value = "/deleteRoute", produces = "application/json;charset=UTF-8")
    public CommonResponse deleteRoute(@ApiParam(value = "巡检路线ID", required = false) @RequestParam List<Long> routeIds) {
        try {
            //校验1：线路上有点时返回删除失败
            Map<Long, String> idNameMap = new HashMap<>();
            for (long id : routeIds) {
                int routePointCount = routeService.countRoutePoint(id);
                if (routePointCount > 0) {
                    Route route = routeService.queryRouteById(id);
                    idNameMap.put(id, route.getName());
                }
            }
            if (idNameMap.size() > 0) {
                return CommonResponseUtil.failure(idNameMap, "删除路线失败：路线上已设置巡检点");
            }
            //校验2：计划在使用，删除失败
            idNameMap.clear();
            for (long id : routeIds) {
                int routePointCount = routeService.countRoutePoint(id);
                if (routePointCount > 0) {
                    List<Plan> planList = routeService.queryPlanByRouteId(id);
                    if (planList != null && planList.size() > 0) {
                        idNameMap.put(id, planList.get(0).getName());
                    }
                }
            }
            if (idNameMap.size() > 0) {
                return CommonResponseUtil.failure(idNameMap, "删除路线失败：计划在使用");
            }
            routeService.delRouteById(routeIds.toArray(new Long[0]));
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("删除巡检路线失败");
        }
    }

    /**
     * 根据ID查询巡检路线信息
     *
     * @param id
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询巡检路线信息", notes = "查询巡检路线信息")
    @GetMapping(value = "/queryRouteById/{id}", produces = "application/json;charset=UTF-8")
    public CommonResponse queryRouteById(
            @ApiParam(value = "巡检路线id", required = true) @PathVariable(name = "id") long id) {
        try {
            Route route = routeService.queryRouteById(id);
            return CommonResponseUtil.success(route);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询巡检路线失败");
        }
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "迁移路线上点到其他路线", notes = "迁移路线上点到其他路线")
    @PostMapping(value = "/migrateRoutePoint", produces = "application/json;charset=UTF-8")
    public CommonResponse migrateRoutePoint(
            @ApiParam(value = "当前巡检路线id", required = true) @RequestParam(name = "sourceRouteId") Long sourceRouteId,
            @ApiParam(value = "巡检点id", required = true) @RequestParam(name = "pointIds") List<Long> pointIds,
            @ApiParam(value = "迁移目标巡检路线id", required = true) @RequestParam(name = "targetRouteId") Long targetRouteId) {
        try {
            routeService.migrateRoutePoint(sourceRouteId, pointIds.toArray(new Long[0]), targetRouteId);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("巡检点迁移失败");
        }
    }

    /**
     * 查询巡检路线信息
     *
     * @return CommonResponse
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询巡检路线信息", notes = "查询巡检路线信息")
    @PostMapping(value = "/list", produces = "application/json;charset=UTF-8")
    public CommonResponse listRoute(
            @ApiParam(value = "查询条件") @RequestBody(required = false) List<CommonRequest> queryRequests,
            @ApiParam(value = "分页参数", required = true, defaultValue = "pageNumber=0&pageSize=10") CommonPageable commonPageable) {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            String loginOrgCode = getOrgCode(reginParams);
            HashMap<String, Object> paramMap = new HashMap<String, Object>();
            paramMap.put("orgCode", loginOrgCode);
            RoutePageParam params = RoutePageParamUtil.fillRouteInfoParam(queryRequests, commonPageable, paramMap);
            return CommonResponseUtil.success(routeService.getRouteInfo(getToken(), getProduct(), getAppKey(), params));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询巡检路线信息失败");
        }
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询当前登录公司的的所有巡检路线", notes = "查询当前登录公司的的所有巡检路线")
    @GetMapping(value = "/All", produces = "application/json;charset=UTF-8")
    public CommonResponse queryAllRoute(
            @RequestParam(required = false) String ownerId, @RequestParam(required = false) Integer orgCode) {
        try {
            //1.组织数据权限信息
            ReginParams reginParams = getSelectedOrgInfo();
            String org = getOrgCode(reginParams);
            //2.查询
            List<Route> routeList;
            if (null != orgCode && 0 == orgCode) {
                routeList = routeService.queryRouteList(ownerId, null);
            } else {
                routeList = routeService.queryRouteList(ownerId, org);
            }
            return CommonResponseUtil.success(routeList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询巡检路线信息失败");
        }
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询所有路线列表<font color='blue'>手机app</font>", notes = "查询所有路线列表<font color='blue'>手机app</font>")
    @GetMapping(value = "/routeList", produces = "application/json;charset=UTF-8")
    public CommonResponse queryRouteList() {
        try {
            ReginParams reginParams = getSelectedOrgInfo();
            String loginOrgCode = getOrgCode(reginParams);
            List<HashMap<String, Object>> routeList = routeService.queryRouteListByOrgCode(loginOrgCode, null, null);
            routeList = routeList.stream().distinct().collect(Collectors.toList());
            return CommonResponseUtil.success(routeList);
        } catch (Exception e) {
            return CommonResponseUtil.failure(e.getMessage());
        }
    }

    /**
     * 查询巡检路线上点列表
     *
     * @param
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询巡检路线上点列表", notes = "查询巡检路线上点列表")
    @PostMapping(value = "/{routeId}/listRoutePoint", produces = "application/json;charset=UTF-8")
    public CommonResponse listRoutePoint(@PathVariable Long routeId) {
        try {
            return CommonResponseUtil.success(routeService.queryRoutePoints(routeId));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("查询巡检路线上点列表失败");
        }
    }


    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询巡检路线上点列表", notes = "查询巡检路线上点列表")
    @PutMapping(value = "/exchangeOrderNumber", produces = "application/json;charset=UTF-8")
    public CommonResponse exchangeRoutePointOrderNumber(@ApiParam(value = "路线巡检点id", required = true) @RequestParam long src,
                                                        @ApiParam(value = "路线巡检点id", required = true) @RequestParam long target) {
        if (src > 0 && target > 0) {
            routeService.exchangeRoutePointOrderNumber(src, target);
        }
        return CommonResponseUtil.success();
    }

    /**
     * 巡检路线上增加巡检点
     *
     * @param
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "巡检路线上增加巡检点", notes = "巡检路线上增加巡检点")
    @PostMapping(value = "/addRoutePoint", produces = "application/json;charset=UTF-8")
    public CommonResponse addRoutePoint(@ApiParam(value = "路线id", required = true) @RequestParam String routeId,
                                        @ApiParam(value = "巡检点ids", required = true) @RequestParam List<String> pointIds) {

        if (pointIds == null || pointIds.size() <= 0) {
            return CommonResponseUtil.failure("巡检点不能为空，请选择");
        }

        try {
            String uid = getUserId();
            ReginParams reginParams = getSelectedOrgInfo();
            String loginOrgCode = getOrgCode(reginParams);
            List<RoutePoint> routePointList = new ArrayList<>();
            for (String pointId : pointIds) {
                RoutePoint routePoint = new RoutePoint();
                routePoint.setRouteId(Long.parseLong(routeId));
                routePoint.setPointId(Long.parseLong(pointId));
                routePoint.setOrgCode(loginOrgCode);
                routePoint.setCreatorId(uid);
                routePointList.add(routePoint);
            }
            routeService.addRoutePoint(routePointList);
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("巡检路线上增加巡检点失败");
        }
    }

    /**
     * 巡检路线上删除巡检点
     *
     * @param
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "巡检路线上删除巡检点", notes = "巡检路线上删除巡检点")
    @DeleteMapping(value = "/deleteRoutePoint", produces = "application/json;charset=UTF-8")
    public CommonResponse deleteRoutePoint(@ApiParam(value = "路线id", required = true) @RequestParam Long routeId,
                                           @ApiParam(value = "巡检点ids", required = true) @RequestParam List<Long> pointIds) {

        try {
            routeService.deleteRoutePoint(routeId, pointIds.toArray(new Long[0]));
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("巡检路线上删除巡检点失败");
        }
    }

    /**
     * 列出巡检线路点的巡检项
     *
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "列出巡检线路点的巡检项", notes = "列出巡检线路点的巡检项")
    @PostMapping(value = "/listRoutePointInputItem", produces = "application/json;charset=UTF-8")
    public CommonResponse listRoutePointInputItem(@ApiParam(value = "路线id", required = true) @RequestBody RoutePointInputItemRequest request) {
        try {
            return CommonResponseUtil.success(routeService.listRoutePointInputItem(request));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("列出巡检线路点的巡检项失败");
        }
    }

    /**
     * 列出巡检线路点的分类及巡检项
     *
     * @param routeId
     * @param pointId
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "列出巡检线路点的分类及巡检项", notes = "列出巡检线路点的分类及巡检项")
    @PostMapping(value = "/listRoutePointInputItemWithClassify", produces = "application/json;charset=UTF-8")
    public CommonResponse listRoutePointInputItemWithClassify(@ApiParam(value = "路线id", required = true) @RequestParam long routeId,
                                                              @ApiParam(value = "巡检点id", required = true) @RequestParam long pointId) {
        try {
            Map<String, List<InputItem>> classifiedInputItemMap = routeService.listRoutePointInputItemWithClassify(routeId, pointId);
            return CommonResponseUtil.success(classifiedInputItemMap);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("列出巡检线路点的分类及巡检项失败");
        }
    }

    /**
     * 修改巡检线路点的巡检项
     *
     * @param routeId
     * @param pointId
     * @param inputItemIds
     * @return
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "修改巡检线路点的巡检项", notes = "修改巡检线路点的巡检项")
    @PostMapping(value = "/updateRoutePointInputItem", produces = "application/json;charset=UTF-8")
    public CommonResponse updateRoutePointInputItem(@ApiParam(value = "巡检路线id", required = true) @RequestParam Long routeId,
                                                    @ApiParam(value = "巡检点id", required = true) @RequestParam Long pointId,
                                                    @ApiParam(value = "巡检点巡检项id", required = true) @RequestParam List<Long> inputItemIds) {
        try {
            routeService.updateRoutePointInputItem(routeId, pointId, inputItemIds.toArray(new Long[0]));
            return CommonResponseUtil.success();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return CommonResponseUtil.failure("修改巡检线路点的巡检项失败");
        }
    }

    /**
     * 根据巡检路线ID获取该路线上的所有点ID
     *
     * @return IDs
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "获取巡检路线上的所有点ID", notes = "获取巡检路线上的所有点ID")
    @RequestMapping(value = "/points/{routeID}", produces = "application/json;charset=UTF-8", method = RequestMethod.POST)
    public CommonResponse getPointIDByRouyeId(@ApiParam(value = "巡检路线ID") @PathVariable(required = true) Long routeID) {
        List<Long> pointIDs = routeService.getPointIDByRouyeId(routeID);
        return CommonResponseUtil.success(pointIDs);
    }

    /**
     * 根据巡检路线ID获取该路线上的所有点ID
     *
     * @return IDs
     */
    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "获取巡检路线上的所有点ID", notes = "获取巡检路线上的所有点ID")
    @RequestMapping(value = "/points", produces = "application/json;charset=UTF-8", method = RequestMethod.POST)
    public CommonResponse getPointIDByRouteId(@ApiParam(value = "巡检路线ID信息") @RequestBody(required = true) Map routeInfo) {
        List<String> routeIdList = (List<String>) routeInfo.get("routeIDs");
        LinkedHashMap<String, Object> routePointIdMap = routeService.getPointIdByRouteId(routeIdList);
        return CommonResponseUtil.success(routePointIdMap);
    }
}
