package com.yeejoin.equipmanage.service.impl;

import com.yeejoin.equipmanage.common.entity.dto.FireResourceStatsDTO;
import com.yeejoin.equipmanage.common.enums.EmergencyEquipTypeEnum;
import com.yeejoin.equipmanage.mapper.FireFightingSystemMapper;
import com.yeejoin.equipmanage.service.IFireResourceSupervisionService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.List;
import java.util.Map;

@Slf4j
@Service
public class IFireResourceSupervisionServiceImpl implements IFireResourceSupervisionService {
    @Autowired
    private FireFightingSystemMapper fireFightingSystemMapper;

    /**
     * 消防系统信息
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireSystemStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectFireFightingSystemStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 监测部件
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getMonitorComponentStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectEquipmentSpecificStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 消防车辆
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireCarStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectCarStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 消防器材 按照需求统计消火栓、灭火器、消防铲、正压式空气呼吸器、消防斧、消防桶
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireEquipStats(String bizOrgCode) {
        List<String> list = EmergencyEquipTypeEnum.getCodeList();
        Map<String, Object> resultMap = fireFightingSystemMapper.getEquipStats(bizOrgCode, list);
        return buildFireResourceStatsDTO(resultMap);
    }

    private FireResourceStatsDTO buildFireResourceStatsDTO(Map<String, Object> resultMap) {
        FireResourceStatsDTO fireResourceStats = new FireResourceStatsDTO();
        fireResourceStats.setTotalCounts(Long.parseLong(resultMap.get("totalCount").toString()));
        fireResourceStats.setYellowCounts(Long.parseLong(resultMap.get("yellowCodeCount").toString()) );
        fireResourceStats.setRedCounts(Long.parseLong(resultMap.get("redCodeCount").toString()));

        long expCount = fireResourceStats.getYellowCounts() + fireResourceStats.getRedCounts();
        double abnormalRatio = 0;
        if (fireResourceStats.getTotalCounts() != 0 && expCount != 0) {
            abnormalRatio = (double) expCount / fireResourceStats.getTotalCounts() * 100;
        }

        if (abnormalRatio == 0) {
            fireResourceStats.setAbnormalRatio(0);
        } else {
            DecimalFormat decimalFormat = new DecimalFormat("#.00");
            decimalFormat.setRoundingMode(RoundingMode.HALF_UP);
            String formattedRatio = decimalFormat.format(abnormalRatio);
            fireResourceStats.setAbnormalRatio(Double.parseDouble(formattedRatio));
        }

        return fireResourceStats;
    }

}
