package com.yeejoin.amos.boot.module.common.api.converter;

import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.property.ExcelContentProperty;
import com.yeejoin.amos.boot.biz.common.entity.DataDictionary;
import com.yeejoin.amos.boot.biz.common.service.impl.DataDictionaryServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 企业信息自定义字段转换器
 * 用于处理Excel导入导出时的字段转换逻辑
 */
@Component
public class CompanyFieldCustomConverter {

    private static DataDictionaryServiceImpl dataDictionaryService;

    @Autowired
    public void setDataDictionaryService(DataDictionaryServiceImpl dataDictionaryService) {
        CompanyFieldCustomConverter.dataDictionaryService = dataDictionaryService;
    }

    /**
     * 【行业主管部门】转换器
     */
    public static class HYZGBMTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("HYZGBM");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【涉及设备类型】转换器
     */
    public static class InvolvedEquipTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {

            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            NAME_TO_CODE_MAP.put("起重机械", "4000");
            NAME_TO_CODE_MAP.put("场内机动车辆", "5000");
            NAME_TO_CODE_MAP.put("锅炉", "1000");
            NAME_TO_CODE_MAP.put("压力容器", "2000");
            NAME_TO_CODE_MAP.put("压力管道", "8000");
            NAME_TO_CODE_MAP.put("大型游乐设施", "6000");
            NAME_TO_CODE_MAP.put("客运索道", "9000");
            NAME_TO_CODE_MAP.put("电梯", "3000");

            CODE_TO_NAME_MAP.put("4000", "起重机械");
            CODE_TO_NAME_MAP.put("5000", "场内机动车辆");
            CODE_TO_NAME_MAP.put("1000", "锅炉");
            CODE_TO_NAME_MAP.put("2000", "压力容器");
            CODE_TO_NAME_MAP.put("8000", "压力管道");
            CODE_TO_NAME_MAP.put("6000", "大型游乐设施");
            CODE_TO_NAME_MAP.put("9000", "客运索道");
            CODE_TO_NAME_MAP.put("3000", "电梯");
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            List<String> list = Arrays.asList(value.split(","));
            if (list.isEmpty()) {
                return new CellData("");
            }
            List<String> res = new ArrayList<>();
            list.forEach(item -> {
                res.add(CODE_TO_NAME_MAP.getOrDefault(item, item));
            });
            return new CellData(String.join(",", res));
        }
    }

    /**
     * 【发证机关】转换器
     */
    public static class ApprovedOrganCodeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("DJJG");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            List<String> list = Arrays.asList(value.split(","));
            if (list.isEmpty()) {
                return new CellData("");
            }
            List<String> res = new ArrayList<>();
            list.forEach(item -> {
                res.add(CODE_TO_NAME_MAP.getOrDefault(item, item));
            });
            return new CellData(String.join(",", res));
        }
    }

    /**
     * 【核准项目】转换器
     */
    public static class ItemCodeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("HZXM");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            List<String> list = Arrays.asList(value.split(","));
            if (list.isEmpty()) {
                return new CellData("");
            }
            List<String> res = new ArrayList<>();
            list.forEach(item -> {
                res.add(CODE_TO_NAME_MAP.getOrDefault(item, item));
            });
            return new CellData(String.join(",", res));
        }
    }

    /**
     * 【机构类别】转换器
     */
    public static class AgencyTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("JYJC_TYPE");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            List<String> list = Arrays.asList(value.split(","));
            if (list.isEmpty()) {
                return new CellData("");
            }
            List<String> res = new ArrayList<>();
            list.forEach(item -> {
                res.add(CODE_TO_NAME_MAP.getOrDefault(item, item));
            });
            return new CellData(String.join(",", res));
        }
    }

    /**
     * 【资质类型】转换器
     * 检验检测、其他
     */
    public static class LicenceTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {

            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            CODE_TO_NAME_MAP.put("jyjc", "检验检测");
            CODE_TO_NAME_MAP.put("other", "其他");
            NAME_TO_CODE_MAP.put("检验检测", "jyjc");
            NAME_TO_CODE_MAP.put("其他", "other");

        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            List<String> list = Arrays.asList(value.split(","));
            if (list.isEmpty()) {
                return new CellData("");
            }
            List<String> res = new ArrayList<>();
            list.forEach(item -> {
                res.add(CODE_TO_NAME_MAP.getOrDefault(item, item));
            });
            return new CellData(String.join(",", res));
        }
    }
}