package com.yeejoin.amos.boot.module.jyjc.biz.action;

import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.yeejoin.amos.boot.biz.common.enums.LicenceStateEnum;
import com.yeejoin.amos.boot.biz.common.utils.RedisUtils;
import com.yeejoin.amos.boot.module.common.api.entity.BaseEnterpriseCert;
import com.yeejoin.amos.boot.module.common.api.enums.JyAgencyClassifyEnum;
import com.yeejoin.amos.boot.module.common.biz.service.impl.BaseEnterpriseCertServiceImpl;
import com.yeejoin.amos.boot.module.jg.api.enums.CompanyTypeEnum;
import com.yeejoin.amos.boot.module.jyjc.api.dto.RuleInspectUnitIndoDto;
import com.yeejoin.amos.boot.module.jyjc.api.entity.JyjcInspectionApplicationNoAcceptLog;
import com.yeejoin.amos.boot.module.jyjc.api.enums.InspectionCompanyType;
import com.yeejoin.amos.boot.module.jyjc.api.enums.LicenceTypeEnum;
import com.yeejoin.amos.boot.module.jyjc.api.enums.OpenBizTypeEnumV2;
import com.yeejoin.amos.boot.module.jyjc.api.mapper.JyjcOpeningApplicationMapper;
import com.yeejoin.amos.boot.module.jyjc.biz.rule.InspectionEquipInfo;
import com.yeejoin.amos.boot.module.jyjc.biz.service.impl.JyjcInspectionApplicationNoAcceptLogServiceImpl;
import com.yeejoin.amos.boot.module.jyjc.biz.service.impl.RuleCommonServiceImpl;
import com.yeejoin.amos.boot.module.ymt.api.dto.TzBaseEnterpriseInfoDto;
import com.yeejoin.amos.boot.module.ymt.api.entity.TzBaseUnitLicence;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzBaseEnterpriseInfoMapper;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzBaseUnitLicenceMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Component;
import org.typroject.tyboot.core.rdbms.orm.entity.BaseEntity;

import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author LiuLin
 * @date 2024年01月02日 15:45
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class RuleActionHandler {

    private final TzBaseUnitLicenceMapper baseUnitLicenceMapper;
    private final TzBaseEnterpriseInfoMapper tzBaseEnterpriseInfoMapper;
    private final RedissonClient redissonClient;
    private final RedisUtils redisUtils;
    private final RuleCommonServiceImpl ruleCommonService;
    private final JyjcInspectionApplicationNoAcceptLogServiceImpl noAcceptLogService;
    private final BaseEnterpriseCertServiceImpl baseEnterpriseCertService;
    private final JyjcOpeningApplicationMapper openingApplicationMapper;


    /**
     * 规则回调方法-用户自选首次提交的检验机构匹配,可替换firstInspectionOrgMatchAction，由于规则修改太多，故新增一个
     *
     * @param bizObj                请求变量参数
     * @param itemCode              核对项目编码，多个用逗号分割
     * @param isMatchArea           是否进行属地过滤（检验检测机构的开通区域包含设备归属地市），true-过滤，false-不过滤
     * @param defaultInspectionCode 默认的检验机构code，多个用逗号分割，定向规则使用此字段指定检验机构
     * @param legalInspectionCodes  预留机构名单，多个用逗号分割
     * @param inspectionCompanyType 需要显示的单位类型：legal-法定、third-第3方、designated-定向单位、all-全部，默认都是legal-法定，预留需求变更点
     * @param isMatchItem           是否需要匹配核准代码：true-匹配； false-不匹配
     * @param isMustAccept          true-必须处理，false-可不予受理
     * @param orgType               jy-检验机构，jc-检测机构
     */
    public void firstInspectionOrgMatchAction2(Object bizObj, String itemCode, Boolean isMatchArea, String defaultInspectionCode, String legalInspectionCodes, String inspectionCompanyType, Boolean isMatchItem, Boolean isMustAccept, String orgType) {
        log.info("收到用户自选首次提交的检验机构匹配规则回调：请求变量参数:{},核对项目编码:{},是否进行属地过滤:{},默认的检验机构code:{}, 预留检验机构code:{}, 检验机构类型: {},是否需要匹配核准代码:{},是否必须处理：{},检验机构类型：{}", bizObj, itemCode, isMatchArea, defaultInspectionCode, legalInspectionCodes, inspectionCompanyType, isMatchItem, isMustAccept, orgType);
        InspectionEquipInfo inspectionEquipInfo = (InspectionEquipInfo) bizObj;
        // 规则不支持或者的关系消息会重复故去重处理
        RLock lock = redissonClient.getLock(this.buildLockKey("firstInspectionOrgMatchAction2", inspectionEquipInfo.getUuid()));
        try {
            boolean isLocked = lock.tryLock(0, 180, TimeUnit.SECONDS);
            if (!isLocked) {
                log.warn("规则回掉重复,消息将丢弃");
                return;
            }
            if (redisUtils.hasKey(this.buildRedisDataKey("firstInspectionOrgMatchAction2", inspectionEquipInfo.getUuid()))) {
                log.warn("规则回掉重复,消息将丢弃");
                return;
            }
            redisUtils.set(this.buildRedisDataKey("firstInspectionOrgMatchAction2", inspectionEquipInfo.getUuid()), inspectionEquipInfo.getUuid(), 3600);
            // 1.获取所有的符合资质条件的单位许可信息
            List<TzBaseUnitLicence> unitLicenceList = getBaseUnitLicenceList(itemCode, isMatchItem, orgType);
            // 2.匹配过滤机构信息,区分检验还是检测
            List<TzBaseEnterpriseInfoDto> tzBaseEnterpriseInfoList = getInspectionUnitListForFirstCommit2(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, orgType, defaultInspectionCode, legalInspectionCodes, isMustAccept, inspectionCompanyType);
            // 3.数据去重按照use_code
            Set<RuleInspectUnitIndoDto> units = this.distinctByUseCode(tzBaseEnterpriseInfoList);
            // todo 4.tzBaseEnterpriseInfoList排序
            ruleCommonService.publishMqttMessage(inspectionEquipInfo.getComponentKey(), units);
        } catch (MqttException | InterruptedException e) {
            log.error("用户自选首次提交的检验机构匹配动作执行失败: {}", e.getMessage());
        } finally {
            if (lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }

    private Set<RuleInspectUnitIndoDto> distinctByUseCode(List<TzBaseEnterpriseInfoDto> tzBaseEnterpriseInfoList) {
        return tzBaseEnterpriseInfoList.stream().map(u -> {
            RuleInspectUnitIndoDto ruleInspectUnitIndoDto = new RuleInspectUnitIndoDto();
            BeanUtil.copyProperties(u, ruleInspectUnitIndoDto);
            return ruleInspectUnitIndoDto;
        }).collect(Collectors.toSet());
    }

    private void setSelfUnitIfDetection(InspectionEquipInfo inspectionEquipInfo, List<TzBaseEnterpriseInfoDto> tzBaseEnterpriseInfoList, Boolean isMustAccept) {
        Map<String, Object> companyInfoMap = inspectionEquipInfo.getCompanyInfo();
        String companyCode = companyInfoMap.get("companyCode").toString();
        // 自检时按照登录人类型进行约束，进行对应的报检业务开通后才能接受
        OpenBizTypeEnumV2 bizTypeEnumV2 = this.getOpenBizTypeByCompanyType(companyInfoMap.get("companyType").toString());
        List<TzBaseEnterpriseInfoDto> selfDetectionUnits = tzBaseEnterpriseInfoMapper.queryInspectionUnitListByUseUnitCode(companyCode, bizTypeEnumV2.getCode());
        setIsCanNoAccept(selfDetectionUnits, isMustAccept);
        tzBaseEnterpriseInfoList.addAll(selfDetectionUnits);
    }

    private OpenBizTypeEnumV2 getOpenBizTypeByCompanyType(String companyType) {
        if (CompanyTypeEnum.CONSTRUCTION.getName().equals(companyType)) {
            return OpenBizTypeEnumV2.AGW;
        } else {
            return OpenBizTypeEnumV2.SYDW;
        }
    }

    /**
     * 规则回调方法-用户自选不予受理的检验机构匹配
     *
     * @param bizObj                请求变量参数
     * @param itemCode              核对项目编码，多个用逗号分割
     * @param isMatchArea           是否进行属地过滤（检验检测机构的开通区域包含设备归属地市），true-过滤，false-不过滤
     * @param defaultInspectionCode 默认的检验机构code，多个用逗号分割，定向规则使用此字段指定检验机构
     * @param inspectionCompanyType 需要显示的单位类型：legal-法定、third-第3方、designated-定向单位、all-全部，默认都是legal-法定，预留需求变更点
     * @param legalInspectionCodes  预留检验机构code，多个用逗号分割
     * @param isMustAccept          true-必须处理，false-可不予受理
     * @param orgType               jy-检验机构，jc-检测机构
     */
    public void noAcceptInspectionOrgMatchAction2(Object bizObj, String itemCode, Boolean isMatchArea, String defaultInspectionCode, String inspectionCompanyType, Boolean isMatchItem, String legalInspectionCodes, Boolean isMustAccept, String orgType) {
        log.info("收到不予受理的检验机构匹配规则回调：请求变量参数:{},核对项目编码:{},是否进行属地过滤:{},默认的检验机构code:{}, 法定的检验机构code:{}, 检验机构类型: {},是否需要匹配核准代码:{},是否必须处理：{}，公司类型：{}", bizObj, itemCode, isMatchArea, defaultInspectionCode, legalInspectionCodes, inspectionCompanyType, isMatchItem, isMustAccept, orgType);
        InspectionEquipInfo inspectionEquipInfo = (InspectionEquipInfo) bizObj;
        try {
            // 1.获取所有的符合资质条件的单位许可信息
            List<TzBaseUnitLicence> unitLicenceList = getBaseUnitLicenceList(itemCode, isMatchItem, orgType);
            // 2.匹配过滤机构信息,默认检验机构（目前检测没配置规则，后续检测也需要配置规则时，需要规则那回调方法新增参数，区分检验还是检测）
            List<TzBaseEnterpriseInfoDto> tzBaseEnterpriseInfoList = getInspectionUnitListForNoAccept2(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, orgType, defaultInspectionCode, legalInspectionCodes, inspectionCompanyType, isMustAccept);
            // 3.去掉已经不予受理的单位，报检时该单位不能再进行接收报检信息（列表不显示）
            this.removeNoAcceptUnit(tzBaseEnterpriseInfoList, inspectionEquipInfo);
            // 4.数据去重按照use_code
            Set<RuleInspectUnitIndoDto> units = this.distinctByUseCode(tzBaseEnterpriseInfoList);
            // todo 5.tzBaseEnterpriseInfoList排序
            ruleCommonService.publishMqttMessage(inspectionEquipInfo.getComponentKey(), units);
        } catch (MqttException e) {
            log.error("Error publishing MQTT message: {}", e.getMessage());
        }
    }

    private void removeNoAcceptUnit(List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos, InspectionEquipInfo inspectionEquipInfo) {
        // 本次报检已经不予受理的单位code
        List<String> noAcceptUnitCodes = getNoAcceptHistoryUnit(inspectionEquipInfo);
        // 不予受理的单位去掉，目的下次不能在进行选择
        matchEnterpriseInfos.removeIf(e -> noAcceptUnitCodes.contains(e.getUseCode()));
    }


    private String buildLockKey(String type, String uuid) {
        return String.format("%s-%s", type, uuid);
    }

    private String buildRedisDataKey(String type, String uuid) {
        return String.format("RECEIVED-UUID-RULE:%s-%s", type, uuid);
    }


    private List<TzBaseUnitLicence> getBaseUnitLicenceList(String itemCode, Boolean isMatchItem, String orgType) {
        List<TzBaseUnitLicence> tzBaseUnitLicences = new ArrayList<>();
        if (isMatchItem && StringUtils.isNotEmpty(itemCode)) {
            tzBaseUnitLicences = baseUnitLicenceMapper.selectList(new LambdaQueryWrapper<TzBaseUnitLicence>()
                    .select(TzBaseUnitLicence::getUnitCode, BaseEntity::getSequenceNbr, TzBaseUnitLicence::getEnterpriseCertSeq)
                    .eq(TzBaseUnitLicence::getLicenceType, LicenceTypeEnum.JY_JC.getCode())
                    .in(TzBaseUnitLicence::getItemCode, Arrays.asList(itemCode.split(",")))
                    .ge(TzBaseUnitLicence::getExpiryDate, LocalDate.now())
                    .eq(TzBaseUnitLicence::getIsDelete, false)
                    .isNotNull(TzBaseUnitLicence::getEnterpriseCertSeq)
                    .eq(TzBaseUnitLicence::getLicenceState, LicenceStateEnum.enabled.getValue()));
        }
        if (!tzBaseUnitLicences.isEmpty()) {
            // 匹配单位类型的证书
            List<String> certSeqs = baseEnterpriseCertService.list(new LambdaQueryWrapper<BaseEnterpriseCert>()
                            .in(com.yeejoin.amos.boot.biz.common.entity.BaseEntity::getSequenceNbr, tzBaseUnitLicences.stream().map(TzBaseUnitLicence::getEnterpriseCertSeq).collect(Collectors.toList()))
                            .eq(BaseEnterpriseCert::getUnitType, OpenBizTypeEnumV2.getOneByCode(orgType).getUnitType())
                            .eq(com.yeejoin.amos.boot.biz.common.entity.BaseEntity::getIsDelete, false)
                            .select(com.yeejoin.amos.boot.biz.common.entity.BaseEntity::getSequenceNbr))
                    .stream().map(e -> String.valueOf(e.getSequenceNbr())).collect(Collectors.toList());
            // 返回符合核准项目且开通的项目且开通的单位类型能匹配上的资质
            tzBaseUnitLicences = tzBaseUnitLicences.stream().filter(l -> certSeqs.contains(l.getEnterpriseCertSeq())).collect(Collectors.toList());
        }
        return tzBaseUnitLicences;
    }


    private List<TzBaseEnterpriseInfoDto> getInspectionUnitListForNoAccept2(Boolean isMatchItem, List<TzBaseUnitLicence> unitLicenceList, Boolean isMatchArea, InspectionEquipInfo inspectionEquipInfo, String openBizType, String defaultInspectionCode, String legalInspectionCodes, String inspectionCompanyType, Boolean isCanNoAccept) {
        return getInspectionUnitListForFirstCommit2(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, openBizType, defaultInspectionCode, legalInspectionCodes, isCanNoAccept, inspectionCompanyType);
    }

    private List<String> getNoAcceptHistoryUnit(InspectionEquipInfo inspectionEquipInfo) {
        LambdaQueryWrapper<JyjcInspectionApplicationNoAcceptLog> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(JyjcInspectionApplicationNoAcceptLog::getApplicationSeq, inspectionEquipInfo.getAppSeq());
        queryWrapper.select(JyjcInspectionApplicationNoAcceptLog::getInspectionUnitCode);
        List<JyjcInspectionApplicationNoAcceptLog> acceptLogs = noAcceptLogService.list(queryWrapper);
        return acceptLogs.stream().map(JyjcInspectionApplicationNoAcceptLog::getInspectionUnitCode).collect(Collectors.toList());
    }


    private List<TzBaseEnterpriseInfoDto> getInspectionUnitListForFirstCommit2(Boolean isMatchItem, List<TzBaseUnitLicence> unitLicenceList, Boolean isMatchArea, InspectionEquipInfo inspectionEquipInfo, String openBizType, String defaultInspectionCode, String legalInspectionCodes, Boolean isMustAccept, String inspectionCompanyTypeCode) {
        List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos = new ArrayList<>();
        InspectionCompanyType inspectionCompanyType = InspectionCompanyType.getByCode(inspectionCompanyTypeCode);
        switch (inspectionCompanyType) {
            case ALL:
                // 所有开通的检验检测机构
                matchEnterpriseInfos = getInspectionUnitListForAll(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, openBizType, isMustAccept);
                break;
            case LEGAL:
                // 查询省内公益机构（省内法定机构）
                matchEnterpriseInfos = getInspectionUnitListForLegal(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, openBizType, defaultInspectionCode, legalInspectionCodes, isMustAccept);
                break;
            case DESIGNATED:
                // 查询指定机构
                matchEnterpriseInfos = getDesignatedUnitList(isMatchItem, defaultInspectionCode, isMatchArea, inspectionEquipInfo, openBizType, isMustAccept, unitLicenceList);
                break;
            case ALL_OR_APP:
                // 所有开通的检验检测机构 + 发起单位
                matchEnterpriseInfos = getInspectionUnitListForAll(isMatchItem, unitLicenceList, isMatchArea, inspectionEquipInfo, openBizType, isMustAccept);
                setSelfUnitIfDetection(inspectionEquipInfo, matchEnterpriseInfos, isMustAccept);
                break;
            default:
                log.warn("未实现的检验机构名单类型：{}", inspectionCompanyType);
                return matchEnterpriseInfos;
        }
        return matchEnterpriseInfos;
    }

    private List<TzBaseEnterpriseInfoDto> getDesignatedUnitList(Boolean isMatchItem, String defaultInspectionCode, Boolean isMatchArea, InspectionEquipInfo inspectionEquipInfo, String openBizType, Boolean isMustAccept, List<TzBaseUnitLicence> unitLicenceList) {
        List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos = new ArrayList<>();
        // 是否进行属地过滤
        String areaCode = isMatchArea ? inspectionEquipInfo.getAreaCode() : "";
        if (defaultInspectionCode.split(",").length > 0) {
            Set<String> unitCodes = unitLicenceList.stream()
                    .map(TzBaseUnitLicence::getUnitCode)
                    .collect(Collectors.toSet());
            // 需要匹配核准项目时，将符合核准项目要求的单位code和指定的单位code取交集，否则直接符合指定的机构
            List<String> designatedUnitMatch = isMatchItem ? Arrays.stream(defaultInspectionCode.split(",")).filter(unitCodes::contains).collect(Collectors.toList()) : Arrays.asList(defaultInspectionCode.split(","));
            matchEnterpriseInfos = tzBaseEnterpriseInfoMapper.getInspectionUnitListByCode(designatedUnitMatch, areaCode, openBizType);
            // 返回下一步是否可不予返回，供下一节点审核时显示判断是否显示不予受理按钮，在提交时数据会写入到主表：true-必须处理，false-可不予受理
            this.setIsCanNoAccept(matchEnterpriseInfos, isMustAccept);
        }
        return matchEnterpriseInfos;
    }


    private List<TzBaseEnterpriseInfoDto> getInspectionUnitListForLegal(Boolean isMatchItem, List<TzBaseUnitLicence> unitLicenceList, Boolean isMatchArea, InspectionEquipInfo inspectionEquipInfo, String openBizType, String defaultInspectionCode, String legalInspectionCodes, Boolean isCanNoAccept) {
        // 是否进行属地过滤
        String areaCode = isMatchArea ? inspectionEquipInfo.getAreaCode() : "";
        // 所有符合资质单位的code
        Set<String> unitCodes = unitLicenceList.stream()
                .map(TzBaseUnitLicence::getUnitCode)
                .collect(Collectors.toSet());
        // 查询省内公益机构（法定机构）开通状态
        List<String> legalInspectionCodeProvince = openingApplicationMapper.selectUnitCodeListByAgencyClassify(Collections.singletonList(JyAgencyClassifyEnum.SBGY.getCode()));
        // 需要匹配核准项目时：将法定机构和符合资质要求的机构取交集，否则直接取指定的机构
        List<String> legalUnitCodes = isMatchItem ? legalInspectionCodeProvince.stream().filter(unitCodes::contains).collect(Collectors.toList()) : legalInspectionCodeProvince;
        List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos = new ArrayList<>();
        // 按照资质 + areaCode，进行设备单位的筛选
        if (!legalUnitCodes.isEmpty()) {
            matchEnterpriseInfos = tzBaseEnterpriseInfoMapper.getInspectionUnitListByCode(legalUnitCodes, areaCode, openBizType);
            // 返回下一步是否可不予返回，供下一节点审核时显示判断是否显示不予受理按钮，在提交时数据会写入
            this.setIsCanNoAccept(matchEnterpriseInfos, isCanNoAccept);
        }
        // 未匹配到法定机构
        if (matchEnterpriseInfos.isEmpty()) {
            log.info("按照资质、区域、属地，未匹配到法定检验机构");
        }
        return matchEnterpriseInfos;
    }

    private List<TzBaseEnterpriseInfoDto> getInspectionUnitListForAll(Boolean isMatchItem, List<TzBaseUnitLicence> unitLicenceList, Boolean isMatchArea, InspectionEquipInfo inspectionEquipInfo, String openBizType, Boolean isCanNoAccept) {
        // 是否进行属地过滤
        String areaCode = isMatchArea ? inspectionEquipInfo.getAreaCode() : "";
        // 所有符合单位的code
        Set<String> unitCodes = unitLicenceList.stream().map(TzBaseUnitLicence::getUnitCode).collect(Collectors.toSet());
        // 不匹配核准项目时，不按照符合资质过滤，默认开通状态
        List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos = tzBaseEnterpriseInfoMapper.getInspectionUnitListByCode(isMatchItem ? new ArrayList<>(unitCodes) : null, areaCode, openBizType);
        // 返回下一步是否可不予返回，供下一节点审核时显示判断是否显示不予受理按钮，在提交时数据会写入到主表：true-必须处理，false-可不予受理
        this.setIsCanNoAccept(matchEnterpriseInfos, isCanNoAccept);
        return matchEnterpriseInfos;
    }

    private void setIsCanNoAccept(List<TzBaseEnterpriseInfoDto> matchEnterpriseInfos, Boolean isCanNoAccept) {
        matchEnterpriseInfos.forEach(m -> m.setUseCodeAndName(m.getUseCode() + "_" + isCanNoAccept));
    }

}

