package com.yeejoin.amos.boot.biz.common.utils;

import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.net.URI;
import java.util.Map;

/**
 * @Author: xl
 * @Description: 远程调用工具类
 * @Date: 2021/10/8 13:46
 */
public class RestTemplateUtils {

    private static class SingletonRestTemplate {
        static final RestTemplate INSTANCE = new RestTemplate();
    }

    private RestTemplateUtils() {
    }

    public static RestTemplate getInstance() {
        return SingletonRestTemplate.INSTANCE;
    }

    /**
     * 带请求头的GET请求调用方式
     *
     * @param uri          请求URL
     * @param headers      请求头参数
     * @param responseType 返回对象类型
     * @param uriVariables URL中的变量，与Map中的key对应
     * @return ResponseEntity 响应对象封装类
     */
    public static <T> ResponseEntity<T> get(URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<?> requestEntity = new HttpEntity<>(headers);
        return RestTemplateUtils.getInstance().exchange(uri, HttpMethod.GET, requestEntity, responseType);
    }

    /**
     * @param restTemplate 可传入负载均衡的restTemplate
     * @param uri
     * @param headers
     * @param responseType
     * @param uriVariables
     * @param <T>
     * @return
     */
    public static <T> ResponseEntity<T> get(RestTemplate restTemplate, URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<?> requestEntity = new HttpEntity<>(headers);
        return restTemplate.exchange(uri, HttpMethod.GET, requestEntity, responseType);
    }

    /**
     * 带请求头的POST请求调用方式
     *
     * @param uri          请求URL
     * @param headers      请求头参数
     * @param requestBody  请求参数体
     * @param responseType 返回对象类型
     * @param uriVariables URL中的变量，与Map中的key对应
     * @return ResponseEntity 响应对象封装类
     */
    public static <T> ResponseEntity<T> post(URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return RestTemplateUtils.getInstance().postForEntity(uri, requestEntity, responseType);
    }

    /**
     * @param restTemplate 可传入负载均衡的restTemplate
     * @param uri
     * @param headers
     * @param requestBody
     * @param responseType
     * @param uriVariables
     * @param <T>
     * @return
     */
    public static <T> ResponseEntity<T> post(RestTemplate restTemplate, URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType, Map<String, ?> uriVariables) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return restTemplate.postForEntity(uri, requestEntity, responseType);
    }

    /**
     * 带请求头的PUT请求调用方式
     *
     * @param uri          请求URL
     * @param headers      请求头参数
     * @param requestBody  请求参数体
     * @param responseType 返回对象类型
     * @return ResponseEntity 响应对象封装类
     */
    public static <T> ResponseEntity<T> put(URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return RestTemplateUtils.getInstance().exchange(uri, HttpMethod.PUT, requestEntity, responseType);
    }

    /**
     * @param restTemplate 可传入负载均衡的restTemplate
     * @param uri
     * @param headers
     * @param requestBody
     * @param responseType
     * @param <T>
     * @return
     */
    public static <T> ResponseEntity<T> put(RestTemplate restTemplate, URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return restTemplate.exchange(uri, HttpMethod.PUT, requestEntity, responseType);
    }

    /**
     * 带请求头的DELETE请求调用方式
     *
     * @param uri          请求URL
     * @param headers      请求头参数
     * @param requestBody  请求参数体
     * @param responseType 返回对象类型
     * @return ResponseEntity 响应对象封装类
     */
    public static <T> ResponseEntity<T> delete(URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return RestTemplateUtils.getInstance().exchange(uri, HttpMethod.DELETE, requestEntity, responseType);
    }

    /**
     * @param restTemplate 可传入负载均衡的restTemplate
     * @param uri
     * @param headers
     * @param requestBody
     * @param responseType
     * @param <T>
     * @return
     */
    public static <T> ResponseEntity<T> delete(RestTemplate restTemplate, URI uri, HttpHeaders headers, Object requestBody, Class<T> responseType) {
        HttpEntity<Object> requestEntity = new HttpEntity<Object>(requestBody, headers);
        return restTemplate.exchange(uri, HttpMethod.DELETE, requestEntity, responseType);
    }
}
