package com.yeejoin.amos.boot.module.jg.biz.utils;

import com.aspose.words.Document;
import com.aspose.words.License;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Slf4j
public class WordTemplateUtils {

    public static final String BASE_PACKAGE_PATH = "/templates";
    private static WordTemplateUtils wordTemplateUtils;
    private final Configuration configuration;

    private WordTemplateUtils() {
        configuration = new Configuration(Configuration.VERSION_2_3_23);
    }

    public static synchronized WordTemplateUtils getInstance() {
        if (wordTemplateUtils == null) {
            wordTemplateUtils = new WordTemplateUtils();
        }
        return wordTemplateUtils;
    }

    /**
     * 创建doc并写入内容
     *
     * @param templatePath doc模板文件路径
     * @param dataMap      内容
     * @param template     模板
     * @return doc文件
     */
    private File createDoc(String templatePath, Map<String, ?> dataMap, Template template) throws TemplateException, IOException {
        // templatePath在后缀之前加上UUID是为了防止并发时多个线程使用同一个模板文件而导致生成的Word文档内容不一致
        int i = templatePath.lastIndexOf(".");
        templatePath = UUID.randomUUID() + templatePath.substring(i);

        if (templatePath.endsWith(".ftl")) {
            templatePath = templatePath.replace(".ftl", ".doc");
        }
        File docFile = new File(templatePath);
        try (
                // 这个地方不能使用FileWriter因为需要指定编码类型否则生成的Word文档会因为有无法识别的编码而无法打开
                Writer writer = new OutputStreamWriter(Files.newOutputStream(docFile.toPath()), StandardCharsets.UTF_8)
        ) {
            template.process(dataMap, writer);
        }
        return docFile;
    }

    public File fillAndConvertDocFile(String templatePath, String targetFileName, Map<String, ?> map, int saveFormat) throws Exception {
        // 指定模板所在包路径
        configuration.setClassForTemplateLoading(this.getClass(), BASE_PACKAGE_PATH);

        // 获取模板, 生成Word文档
        Template freemarkerTemplate = configuration.getTemplate(templatePath, "UTF-8");
        File docFile = createDoc(templatePath, map, freemarkerTemplate);

        // 转换Word文档
        File converedFile = converDocFile(docFile.getAbsolutePath(), targetFileName, saveFormat);

        // 删除临时文件
        Files.deleteIfExists(docFile.toPath());

        return converedFile;
    }

    /**
     * word转换
     *
     * @param docPath    word文件路径
     * @param targetPath 转换后文件路径
     * @param saveFormat 目标文件类型 取自 com.aspose.words.SaveFormat
     */
    private File converDocFile(String docPath, String targetPath, int saveFormat) throws Exception {
        // 验证License 若不验证则转化出的pdf文档会有水印产生
        if (!getLicense()) {
            throw new RuntimeException("验证License失败");
        }

        if (StringUtils.isEmpty(docPath)) {
            throw new FileNotFoundException("文档文件不存在");
        }

        try (
                InputStream inputStream = Files.newInputStream(Paths.get(docPath));
                OutputStream outputStream = Files.newOutputStream(Paths.get(targetPath));
        ) {
            File targetFile = new File(targetPath);
            Document doc = new Document(inputStream);
            doc.save(outputStream, saveFormat);
            return targetFile;
        }
    }


    /**
     * 获取License
     *
     * @return boolean
     */
    private boolean getLicense() {
        boolean result = false;
        try {
            String s = "<License><Data><Products><Product>Aspose.Total for Java</Product><Product>Aspose.Words for Java</Product></Products><EditionType>Enterprise</EditionType><SubscriptionExpiry>20991231</SubscriptionExpiry><LicenseExpiry>20991231</LicenseExpiry><SerialNumber>8bfe198c-7f0c-4ef8-8ff0-acc3237bf0d7</SerialNumber></Data><Signature>sNLLKGMUdF0r8O1kKilWAGdgfs2BvJb/2Xp8p5iuDVfZXmhppo+d0Ran1P9TKdjV4ABwAgKXxJ3jcQTqE/2IRfqwnPf8itN8aFZlV3TJPYeD3yWE7IT55Gz6EijUpC7aKeoohTb4w2fpox58wWoF3SNp6sK6jDfiAUGEHYJ9pjU=</Signature></License>";
            ByteArrayInputStream is = new ByteArrayInputStream(s.getBytes());
            License license = new License();
            license.setLicense(is);
            result = true;
        } catch (Exception e) {
            log.error("获取License失败", e);
        }
        return result;
    }
}
