package com.yeejoin.amos.speech;

import com.alibaba.nls.client.protocol.InputFormatEnum;
import com.alibaba.nls.client.protocol.SampleRateEnum;
import com.alibaba.nls.client.protocol.asr.SpeechTranscriberListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.util.concurrent.TimeUnit;


/**
 * 实时语音识别
 * <p>
 * 支持的输入格式：PCM（无压缩的PCM或WAV文件）、16 bit采样位数、单声道（mono）。
 * 支持的音频采样率：8000 Hz和16000 Hz。
 */
public class SpeechTranscriber {

    private static final Logger logger = LoggerFactory.getLogger(SpeechTranscriber.class);
    //语音识别文件
    private final File speechFile;
    //识别回调函数
    private final SpeechTranscriberListener listener;

    public SpeechTranscriber(File speechFile, SpeechTranscriberListener listener) {
        if (speechFile != null && speechFile.exists() && speechFile.isFile()) {
            this.speechFile = speechFile;
            this.listener = listener;
        } else {
            throw new IllegalArgumentException("待识别的文件存在异常");
        }
    }

    /**
     * 开始语音识别
     */
    public void process() {
        com.alibaba.nls.client.protocol.asr.SpeechTranscriber transcriber = null;
        try {
            //创建实例、建立连接。
            transcriber = new com.alibaba.nls.client.protocol.asr.SpeechTranscriber(AppNslClient.instance(), listener);
            transcriber.setAppKey(SpeechConfig.AppKey);
            //输入音频编码方式。
            transcriber.setFormat(InputFormatEnum.PCM);
            //输入音频采样率。
            transcriber.setSampleRate(SampleRateEnum.SAMPLE_RATE_16K);
            //是否返回中间识别结果。
            transcriber.setEnableIntermediateResult(true);
            //是否生成并返回标点符号。
            transcriber.setEnablePunctuation(true);
            //是否将返回结果规整化，比如将一百返回为100。
            transcriber.setEnableITN(false);

            //设置vad断句参数。默认值：800ms，有效值：200ms～2000ms。
            //transcriber.addCustomedParam("max_sentence_silence", 600);
            //设置是否语义断句。
            //transcriber.addCustomedParam("enable_semantic_sentence_detection",false);
            //设置是否开启顺滑。
            //transcriber.addCustomedParam("disfluency",true);
            //设置是否开启词模式。
            //transcriber.addCustomedParam("enable_words",true);
            //设置vad噪音阈值参数，参数取值为-1～+1，如-0.9、-0.8、0.2、0.9。
            //取值越趋于-1，判定为语音的概率越大，亦即有可能更多噪声被当成语音被误识别。
            //取值越趋于+1，判定为噪音的越多，亦即有可能更多语音段被当成噪音被拒绝识别。
            //该参数属高级参数，调整需慎重和重点测试。
            //transcriber.addCustomedParam("speech_noise_threshold",0.3);
            //设置训练后的定制语言模型id。
            //transcriber.addCustomedParam("customization_id","你的定制语言模型id");
            //设置训练后的定制热词id。
            //transcriber.addCustomedParam("vocabulary_id","你的定制热词id");
            //设置是否忽略单句超时。
            transcriber.addCustomedParam("enable_ignore_sentence_timeout", false);
            //vad断句开启后处理。
            //transcriber.addCustomedParam("enable_vad_unify_post",false);
            //此方法将以上参数设置序列化为JSON发送给服务端，并等待服务端确认。
            transcriber.start();
            FileInputStream fis = new FileInputStream(speechFile);
            byte[] b = new byte[3200];
            int len;
            while ((len = fis.read(b)) > 0) {
                logger.info("send data pack length: " + len);
                transcriber.send(b, len);
                //本案例用读取本地文件的形式模拟实时获取语音流并发送的，因为读取速度较快，这里需要设置sleep,如果实时获取语音则无需设置sleep
                TimeUnit.MILLISECONDS.sleep(1000);
            }
            //通知服务端语音数据发送完毕，等待服务端处理完成。
            long now = System.currentTimeMillis();
            logger.info("ASR wait for complete");
            transcriber.stop();
            logger.info("ASR latency : " + (System.currentTimeMillis() - now) + " ms");
        } catch (Exception e) {
            logger.error(e.getMessage());
        } finally {
            if (null != transcriber) {
                transcriber.close();
            }
        }
    }
}
