package com.yeejoin.amos.boot.module.jyjc.biz.listener;

import com.yeejoin.amos.boot.biz.common.utils.RedisUtils;
import com.yeejoin.amos.boot.module.jyjc.api.enums.TopicEnum;
import com.yeejoin.amos.boot.module.jyjc.api.model.InspectionEquipInfoModel;
import com.yeejoin.amos.boot.module.jyjc.biz.event.InspectionOrgRefreshEvent;
import com.yeejoin.amos.boot.module.jyjc.biz.event.publisher.EventPublisher;
import com.yeejoin.amos.boot.module.jyjc.biz.listener.message.BizMessage;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.typroject.tyboot.component.emq.EmqKeeper;
import org.typroject.tyboot.component.emq.EmqxListener;

import javax.annotation.PostConstruct;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import static com.alibaba.fastjson.JSON.parseObject;

/**
 * @author Administrator
 */
@Component
@Slf4j
public class InspectionOrgRefreshListener extends EmqxListener {

    private static final int REPEAT_TIME = 2;

    private static final BlockingQueue<BizMessage> BLOCKING_QUEUE = new LinkedBlockingQueue<>();

    private final EmqKeeper emqKeeper;

    @Value("${spring.application.name}")
    private String applicationName;

    private final RedisUtils redisUtils;

    private final EventPublisher publisher;

    public InspectionOrgRefreshListener(EmqKeeper emqKeeper, RedisUtils redisUtils, EventPublisher publisher) {
        this.emqKeeper = emqKeeper;
        this.redisUtils = redisUtils;
        this.publisher = publisher;
    }


    @Override
    public void processMessage(String topic, MqttMessage message) {
        if (log.isInfoEnabled()) {
            log.info("收到消息主题：{},消息内容：{}", topic, message.toString());
        }
        BLOCKING_QUEUE.add(new BizMessage(topic, message));
    }

    @PostConstruct
    public void init() throws Exception {
        emqKeeper.subscript(this.buildShareTopic(TopicEnum.INSPECTION_LIST_REFRESH.getTopic()), 2, this);
        Executors.newSingleThreadExecutor().submit(this::takeMsg);
    }

    private String buildShareTopic(String topic) {
        return "$share/" + applicationName + "/" + topic;
    }

    private void takeMsg() {
        while (true) {
            try {
                BizMessage bizMessage = BLOCKING_QUEUE.take();
                // 解决前端组件在属性变化时重复发送两次问题
                if (!redisUtils.hasKey(bizMessage.getTopic())) {
                    redisUtils.set(bizMessage.getTopic(), true, REPEAT_TIME);
                    processBizMessage(bizMessage);
                } else {
                    log.warn("消息在{}秒内重复，", REPEAT_TIME);
                }
            } catch (Exception e) {
                log.error("数据处理失败", e);
            }
        }
    }

    private void processBizMessage(BizMessage bizMessage) {
        log.info("收到前端消息：{}", bizMessage);
        byte[] payload = bizMessage.getMessage().getPayload();
        InspectionEquipInfoModel equipInfoModel = parseObject(new String(payload, StandardCharsets.UTF_8), InspectionEquipInfoModel.class);
        String componentKey = bizMessage.getTopic().split("/")[0];
        equipInfoModel.setComponentKey(componentKey);
        publisher.publish(new InspectionOrgRefreshEvent(this, equipInfoModel));
    }

}
