package com.yeejoin.amos.boot.module.jg.biz.edit.event.listener;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.yeejoin.amos.boot.module.jg.api.dto.ESDataChangeLogDto;
import com.yeejoin.amos.boot.module.jg.api.dto.FieldChangeMeta;
import com.yeejoin.amos.boot.module.jg.api.dto.JgResumeInfoDto;
import com.yeejoin.amos.boot.module.jg.api.entity.JgBizChangeLog;
import com.yeejoin.amos.boot.module.jg.api.enums.BusinessTypeEnum;
import com.yeejoin.amos.boot.module.jg.biz.dao.ESDataChangeLogDao;
import com.yeejoin.amos.boot.module.jg.biz.edit.event.BaseBizDataChangeEvent;
import com.yeejoin.amos.boot.module.jg.biz.service.impl.JgBizChangeLogServiceImpl;
import com.yeejoin.amos.boot.module.jg.biz.service.impl.JgResumeInfoServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.event.TransactionalEventListener;
import org.typroject.tyboot.core.foundation.context.RequestContextModel;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * @author Administrator
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class ChangeLogInsertListener {

    private static final Map<String, String> bizTypeTableMap = new HashMap<>();

    static {
        bizTypeTableMap.put("useRegister", "tzs_jg_use_registration");
        bizTypeTableMap.put("installationNotice", "tzs_jg_installation_notice");
        bizTypeTableMap.put("singleManageEquipEdit", "idx_biz_jg_use_info");
    }

    @Value("${change.log.deal.thread.number:1}")
    private int threadNumber;

    private final ESDataChangeLogDao esDataChangeLogDao;

    private final JgBizChangeLogServiceImpl bizChangeLogService;

    private final JgResumeInfoServiceImpl jgResumeInfoService;


    private final BlockingQueue<BaseBizDataChangeEvent> queue = new LinkedBlockingQueue<>();


    @TransactionalEventListener(value = BaseBizDataChangeEvent.class)
    @Async
    public void handleTransactionalEvent(BaseBizDataChangeEvent event) {
        log.info("日志处理器，收到变更业务数据消息:{}", JSONObject.toJSONString(event));
        queue.add(event);
    }

    @PostConstruct
    public void init() {
        ExecutorService executorService = Executors.newFixedThreadPool(threadNumber);
        IntStream.range(0, threadNumber).forEach(i -> {
            executorService.execute(() -> {
                while (true) {
                    try {
                        BaseBizDataChangeEvent event = queue.take();
                        processEvent(event);
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                    }
                }
            });
        });
    }

    private void processEvent(BaseBizDataChangeEvent event) {
        List<FieldChangeMeta> fieldChangeMetas = event.getData();
        RequestContextModel requestContextModel = event.getRequestContext();
        Date date = new Date();
        // 1.主表保存
        JgBizChangeLog changeLog = saveLog(event);
        // 2.子表字段保存
        List<ESDataChangeLogDto> logs = saveColumnsChange2Es(fieldChangeMetas, requestContextModel, date, changeLog);
        // 3.异步记录履历日志
        this.insertCertLog(changeLog, event);
        log.info("es 操作日志数据入库成功：{}条", logs.size());
    }

    @Async
    protected void insertCertLog(JgBizChangeLog changeLog, BaseBizDataChangeEvent event) {
        try {
            String equId = Stream.concat(
                    Optional.ofNullable(event.getBizRelationData().getProjectContraptionIds()).orElse(Collections.emptySet()).stream(),
                    Optional.ofNullable(event.getBizRelationData().getRecords()).orElse(Collections.emptySet()).stream()
            ).findFirst().orElse(null);
            String businessTypeName = BusinessTypeEnum
                    .findNameByEnumName(changeLog.getBizType())
                    .map(name -> "设备编辑_" + name)
                    .orElse("设备编辑");
            jgResumeInfoService.createWithModel(JgResumeInfoDto.builder()
                    .applyNo(event.getBizRelationData().getBizId())
                    .businessType(businessTypeName)
                    .businessId(event.getBizRelationData().getBizId())
                    .equId(equId)
                    .resumeType("1")
                    .changeLogId(String.valueOf(changeLog.getSequenceNbr()))
                    .approvalUnit(event.getBizRelationData().getUnitName())
                    .approvalUnitCode(event.getBizRelationData().getUnitCode())
                    .createUserName(event.getBizRelationData().getRecUserName())
                    .createUserId(event.getRequestContext().getExcutedUserId())
                    .status("正常")
                    .build());
        } catch (Exception e){
            log.error(e.getMessage(), e);
        }
    }

    private List<ESDataChangeLogDto> saveColumnsChange2Es(List<FieldChangeMeta> fieldChangeMetas, RequestContextModel requestContextModel, Date date, JgBizChangeLog changeLog) {
        List<ESDataChangeLogDto> logs = fieldChangeMetas.parallelStream().map(changeDataDto -> {
            ESDataChangeLogDto changeLogDto = new ESDataChangeLogDto();
            changeLogDto.setColumnKey(changeDataDto.getColumnKey());
            changeLogDto.setColumnKeyLabel(changeDataDto.getColumnLabel());
            changeLogDto.setBeforeData(changeDataDto.getColumnOldValue());
            changeLogDto.setAfterData(changeDataDto.getColumnNewValue());
            changeLogDto.setUserId(requestContextModel.getExcutedUserId());
            changeLogDto.setCreateDate(date.getTime());
            changeLogDto.setRequestDate(DateUtil.formatDateTime(date));
            changeLogDto.setChangeId(changeDataDto.getChangeId());
            changeLogDto.setColumnFamily(changeDataDto.getColumnFamily());
            changeLogDto.setBizType(changeDataDto.getBizType());
            changeLogDto.setBatchId(changeLog.getSequenceNbr() + "");
            changeLogDto.setColumnType(changeDataDto.getColumnType());
            return changeLogDto;
        }).collect(Collectors.toList());
        if (!logs.isEmpty()) {
            esDataChangeLogDao.saveAll(logs);
        }
        return logs;
    }

    private JgBizChangeLog saveLog(BaseBizDataChangeEvent event) {
        JgBizChangeLog changeLog = new JgBizChangeLog();
        BeanUtil.copyProperties(event.getBizRelationData(), changeLog);
        changeLog.setBizTable(bizTypeTableMap.get(changeLog.getBizType()));
        changeLog.setRecDate(new Date());
        changeLog.setRecUserId(event.getRequestContext().getExcutedUserId());
        changeLog.setSequenceNbr(Long.parseLong(event.getRequestContext().getTraceId()));
        bizChangeLogService.getBaseMapper().insert(changeLog);
        return changeLog;
    }
}
