package com.yeejoin.amos.boot.module.jg.biz.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.yeejoin.amos.boot.biz.common.entity.BaseEntity;
import com.yeejoin.amos.boot.module.common.api.dao.ESEquipmentCategory;
import com.yeejoin.amos.boot.module.common.api.dto.ESEquipmentCategoryDto;
import com.yeejoin.amos.boot.module.jg.api.entity.SafetyProblemTracing;
import com.yeejoin.amos.boot.module.jg.api.enums.SafetyProblemStatusEnum;
import com.yeejoin.amos.boot.module.jg.api.enums.SafetyProblemTypeEnum;
import com.yeejoin.amos.boot.module.jg.api.mapper.CommonMapper;
import com.yeejoin.amos.boot.module.ymt.api.entity.IdxBizJgOtherInfo;
import com.yeejoin.amos.boot.module.ymt.api.entity.TzBaseEnterpriseInfo;
import com.yeejoin.amos.boot.module.ymt.api.entity.TzsUserInfo;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzBaseEnterpriseInfoMapper;
import com.yeejoin.amos.boot.module.ymt.api.mapper.TzsUserInfoMapper;
import net.javacrumbs.shedlock.spring.annotation.SchedulerLock;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.typroject.tyboot.component.emq.EmqKeeper;
import org.typroject.tyboot.core.foundation.utils.ValidationUtil;

import java.util.*;
import java.util.stream.Collectors;
/**
 * 安全追溯问题生成服务实现类
 *
 * @author system_generator
 * @date 2024-05-21
 */
@Service
@EnableScheduling
public class SafetyProblemTracingGenServiceImpl{
    final static Logger logger = LoggerFactory.getLogger(SafetyProblemTracingGenServiceImpl.class);

    @Autowired
    CommonMapper commonMapper;

    @Autowired
    IdxBizJgOtherInfoServiceImpl idxBizJgOtherInfoService;

    @Autowired
    ESEquipmentCategory esEquipmentCategory;

    @Autowired
    EmqKeeper emqKeeper;

    @Autowired
    private SafetyProblemTracingServiceImpl safetyProblemTracingService;

    @Autowired
    TzsUserInfoMapper tzsUserInfoMapper;

    @Autowired
    TzBaseEnterpriseInfoMapper tzBaseEnterpriseInfoMapper;

    @Scheduled(cron = "0 0 1 * * ?")
    @SchedulerLock(name = "executeSafetyProblemCheck", lockAtMostFor = "PT5H", lockAtLeastFor = "PT10M")
    public void executeSafetyProblemCheck() {
        executeMaintenanceCheck();
        executeInspectionCheck();
        executeEnterpriseQualificationCheck();
    }

    @Scheduled(cron = "0 0 1 * * ?")
    @SchedulerLock(name = "executePersonnalCertification", lockAtMostFor = "PT5H", lockAtLeastFor = "PT10M")
    public void executePersonnalCertification() {
        executePersonnalCertificationCheck();
    }

    @Scheduled(cron = "0 0 5 ? * SAT")
    @SchedulerLock(name = "correctOverdueProblem", lockAtMostFor = "PT5H", lockAtLeastFor = "PT60M")
    public void correctOverdueProblem() {
        try {
            logger.info("开始执行 overdue problem 修复任务");
            long startTime = System.currentTimeMillis();
            repairMaintenanceAndInspectionOverdueProblem();
            long endTime = System.currentTimeMillis();
            logger.info("overdue problem 修复任务执行完成，耗时: {} ms", (endTime - startTime));
        } catch (Exception e) {
            logger.error("执行 overdue problem 修复任务时发生异常", e);
        }
    }


    public void executePersonnalCertificationCheck() {
        logger.info("开始人员资质超期检查");
        List<Map<String, Object>> certificationRecords = commonMapper.queryOutOfCertificationRecord();
        updateUserInfoAndSendMessage(certificationRecords);
        logger.info("人员资质超期检查结束");
    }

    public void executeInspectionCheck() {
        logger.info("开始检验检测超期检查");
        Set<String> outOfInspectionEquipIds = safetyProblemTracingService.getBaseMapper().selectList(new LambdaQueryWrapper<SafetyProblemTracing>()
                        .select(SafetyProblemTracing::getSourceId)
                .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.JYCQ.getProblemTypeCode()))
                .stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toSet());

        // 查询当天检验超期的设备
        List<Map<String, Object>> outOfInspectionRecords = commonMapper.queryOutOfInspectionRecord();
        // 追溯表里未处理的检验超期设备需要排除在外
        outOfInspectionRecords.removeIf(item -> outOfInspectionEquipIds.contains(String.valueOf(item.get("RECORD"))));
        updateEquipAndSendMessage(outOfInspectionRecords, SafetyProblemTypeEnum.JYCQ);
        logger.info("检验检测超期检查结束");
    }

    public void executeMaintenanceCheck() {
        logger.info("开始维保超期检查");
        Set<String> outOfMaintenanceEquipIds = safetyProblemTracingService.getBaseMapper().selectList(new LambdaQueryWrapper<SafetyProblemTracing>()
                        .select(SafetyProblemTracing::getSourceId)
                        .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                        .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.WBCQ.getProblemTypeCode()))
                .stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toSet());

        // 查询当天维保超期的设备
        List<Map<String, Object>> outOfMaintenanceRecords = commonMapper.queryOutOfMaintenanceRecord();

        // 追溯表里未处理的维保超期设备需要排除在外
        outOfMaintenanceRecords.removeIf(item -> outOfMaintenanceEquipIds.contains(String.valueOf(item.get("RECORD"))));
        updateEquipAndSendMessage(outOfMaintenanceRecords, SafetyProblemTypeEnum.WBCQ);
        logger.info("维保超期检查结束");
    }

    public void executeEnterpriseQualificationCheck() {
        logger.info("开始企业许可超期检查");
        // 查询当天许可超期的企业许可数据
        List<Map<String, Object>> outOfQualificationRecords = commonMapper.queryOutOfQualificationRecord();
        updateEnterpriseAndSendMessage(outOfQualificationRecords);
        logger.info("企业许可超期检查结束");
    }

    public Map<String, Object> wrongOverdueData() {
        Map<String, Object> result = new HashMap<>();
        // 查询当天资质超期的人员数据
        Set<String> realOutOfCertificationRecord = commonMapper.queryOutOfCertificationRecord().stream().map(m -> m.get("problemSourceId").toString()).collect(Collectors.toSet());
        // 查询当天许可超期的企业许可数据
        Set<String> realOutOfQualificationRecords = commonMapper.queryOutOfQualificationRecord().stream().map(m -> m.get("problemSourceId").toString()).collect(Collectors.toSet());
        // 查询当天维保超期的企业许可数据
        Set<String> realOutOfMaintenanceRecord = commonMapper.queryOutOfMaintenanceRecord().stream().map(m -> m.get("RECORD").toString()).collect(Collectors.toSet());
        // 查询当天检验超期的企业许可数据
        Set<String> realOutOfInspectRecord = commonMapper.queryOutOfInspectionRecord().stream().map(m -> m.get("RECORD").toString()).collect(Collectors.toSet());

        // 查询未关闭的问题
        List<SafetyProblemTracing> problemList = safetyProblemTracingService.list(new LambdaQueryWrapper<SafetyProblemTracing>()
                        .select(SafetyProblemTracing::getSequenceNbr, SafetyProblemTracing::getProblemTypeCode, SafetyProblemTracing::getSourceId, SafetyProblemTracing::getPrincipalUnitCode)
                        .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                        .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE));

        if(!ValidationUtil.isEmpty(problemList)) {
            Set<Long> set1 = problemList.stream().filter(p -> p.getProblemTypeCode().equals(SafetyProblemTypeEnum.ZZCQ.getProblemTypeCode()) && !realOutOfCertificationRecord.contains(p.getSourceId())).map(SafetyProblemTracing::getSequenceNbr).collect(Collectors.toSet());
            Set<Long> set2 = problemList.stream().filter(p -> p.getProblemTypeCode().equals(SafetyProblemTypeEnum.WBCQ.getProblemTypeCode()) && !realOutOfMaintenanceRecord.contains(p.getSourceId())).map(SafetyProblemTracing::getSequenceNbr).collect(Collectors.toSet());
            Set<Long> set3 = problemList.stream().filter(p -> p.getProblemTypeCode().equals(SafetyProblemTypeEnum.JYCQ.getProblemTypeCode()) && !realOutOfInspectRecord.contains(p.getSourceId())).map(SafetyProblemTracing::getSequenceNbr).collect(Collectors.toSet());
            Set<Long> set4 = problemList.stream().filter(p -> p.getProblemTypeCode().equals(SafetyProblemTypeEnum.XKCQ.getProblemTypeCode()) && !realOutOfQualificationRecords.contains(p.getSourceId())).map(SafetyProblemTracing::getSequenceNbr).collect(Collectors.toSet());

            result.put("zzcq", set1);
            result.put("wbqc", set2);
            result.put("jyqc", set3);
            result.put("xkqc", set4);
        }
        return result;
    }

    public void repairPersonnelQualificationOverdueProblem() {
        logger.info("开始人员资质超期数据修正");
        // 查询当天资质超期的人员数据
        List<Map<String, Object>> outOfCertificationRecord = commonMapper.queryOutOfCertificationRecord();
        if (ValidationUtil.isEmpty(outOfCertificationRecord)) {
            return;
        }
        Set<String> realOutOfQualificationRecord = outOfCertificationRecord.stream().map(m -> m.get("problemSourceId").toString()).collect(Collectors.toSet());
        // 需要修正业务表的数据
        List<String> correctData = Lists.newArrayList();
        List<SafetyProblemTracing> problemList = safetyProblemTracingService.list(new LambdaQueryWrapper<SafetyProblemTracing>()
                .select(SafetyProblemTracing::getSourceId, SafetyProblemTracing::getPrincipalUnitCode)
                .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.ZZCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
        );
        if (!ValidationUtil.isEmpty(problemList)) {
            Set<String> problemIds = problemList.stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toSet());
            problemIds.removeAll(realOutOfQualificationRecord);
            correctData.addAll(problemIds);
            if (!ValidationUtil.isEmpty(correctData)) {
                // 更新问题表
                safetyProblemTracingService.lambdaUpdate()
                        .set(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.HANDLED.getCode())
                        .set(SafetyProblemTracing::getProblemStatus, SafetyProblemStatusEnum.HANDLED.getName())
                        .in(SafetyProblemTracing::getSourceId, correctData)
                        .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.ZZCQ.getProblemTypeCode())
                        .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
                        .update();
            }
        }
        logger.info("人员资质超期数据修正结束，更新数据：{}条", correctData.size());
    }

    public void repairEnterpriseOverdueProblem() {
        logger.info("开始企业许可超期数据修正");
        // 查询当天许可超期的企业许可数据
        List<Map<String, Object>> outOfQualificationRecords = commonMapper.queryOutOfQualificationRecord();
        if (ValidationUtil.isEmpty(outOfQualificationRecords)) {
            return;
        }
        Set<String> realOutOfQualificationRecord = outOfQualificationRecords.stream().map(m -> m.get("problemSourceId").toString()).collect(Collectors.toSet());
        // 需要修正业务表的数据
        List<String> correctData = Lists.newArrayList();
        List<SafetyProblemTracing> problemList = safetyProblemTracingService.list(new LambdaQueryWrapper<SafetyProblemTracing>()
                .select(SafetyProblemTracing::getSourceId, SafetyProblemTracing::getPrincipalUnitCode)
                .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.XKCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
        );
        if (!ValidationUtil.isEmpty(problemList)) {
            Set<String> problemIds = problemList.stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toSet());
            problemIds.removeAll(realOutOfQualificationRecord);
            correctData.addAll(problemIds);
            if (!ValidationUtil.isEmpty(correctData)) {
                // 更新问题表
                safetyProblemTracingService.lambdaUpdate()
                        .set(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.HANDLED.getCode())
                        .set(SafetyProblemTracing::getProblemStatus, SafetyProblemStatusEnum.HANDLED.getName())
                        .in(SafetyProblemTracing::getSourceId, correctData)
                        .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.XKCQ.getProblemTypeCode())
                        .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
                        .update();
            }
            // 更新企业表问题状态。同一企业所有资质都正常才更新
            Set<String> problemUnitCreditCode = problemList.stream().map(SafetyProblemTracing::getPrincipalUnitCode).collect(Collectors.toSet());
            Set<String> problemUnitCodes = outOfQualificationRecords.stream().map(item -> item.get("useUnitCode").toString()).collect(Collectors.toSet());
            // 错误问题企业如果没在当天的许可超期数据中，则更新企业状态为正常
            problemUnitCreditCode.removeAll(problemUnitCodes);
            if (!ValidationUtil.isEmpty(problemUnitCreditCode)) {
                tzBaseEnterpriseInfoMapper.update(new TzBaseEnterpriseInfo(),
                        new LambdaUpdateWrapper<TzBaseEnterpriseInfo>()
                                .set(TzBaseEnterpriseInfo::getStatus, "0") // 0正常
                                .in(TzBaseEnterpriseInfo::getUseCode, problemUnitCreditCode));
            }
        }
        logger.info("企业许可超期数据修正结束，更新数据：{}条", correctData.size());
    }

    public void repairMaintenanceAndInspectionOverdueProblem() {
        logger.info("开始设备维保备案和检验检测超期数据修正");
        Set<String> realOutOfMaintenanceRecord = commonMapper.queryOutOfMaintenanceRecord().stream().map(m -> m.get("RECORD").toString()).collect(Collectors.toSet());
        Set<String> realOutOfInspectRecord = commonMapper.queryOutOfInspectionRecord().stream().map(m -> m.get("RECORD").toString()).collect(Collectors.toSet());
        // 需要修正业务表+es的数据
        List<String> correctData = Lists.newArrayList();
        // 1.处理维保超期错误数据
        List<String> outOfMaintenanceErrorData = handleOutOfMaintenanceErrorData(realOutOfMaintenanceRecord);
        // 2.处理检验超期错误数据
        List<String> outOfInspectionErrorData = handleOutOfInspectionErrorData(realOutOfInspectRecord);
        // 3.修正业务表+es数据状态
        if (ValidationUtil.isEmpty(outOfMaintenanceErrorData) && ValidationUtil.isEmpty(outOfInspectionErrorData)) {
            // 没有需要修正的数据
            return;
        }

        Set<String> maintenanceSet = new HashSet<>(ValidationUtil.isEmpty(outOfMaintenanceErrorData) ? Collections.emptyList() : outOfMaintenanceErrorData);
        Set<String> inspectionSet = new HashSet<>(ValidationUtil.isEmpty(outOfInspectionErrorData) ? Collections.emptyList() : outOfInspectionErrorData);

        if (!maintenanceSet.isEmpty() && inspectionSet.isEmpty()) {
            // 仅有维保备案错误数据
            correctData.addAll(maintenanceSet);
        } else if (maintenanceSet.isEmpty() && !inspectionSet.isEmpty()) {
            // 仅有检验超期错误数据
            correctData.addAll(inspectionSet);
        } else if (!maintenanceSet.isEmpty()) {
            // 再次查询维保备案这些数据有没有检验超期的问题
            maintenanceSet.removeAll(realOutOfInspectRecord);
            // 没有则可以愉快地更新设备状态为正常
            correctData.addAll(maintenanceSet);
            // 再次查询检验这些数据有没有维保超期问题
            inspectionSet.removeAll(realOutOfMaintenanceRecord);
            // 没有则可以愉快地更新设备状态为正常
            correctData.addAll(inspectionSet);

        }
        safetyProblemTracingService.updateOtherTableAndES(Sets.newHashSet(correctData));
        logger.info("设备维保备案和检验超期数据修正完成，更新数据：{}条", correctData.size());
    }

    private List<String> handleOutOfMaintenanceErrorData(Set<String> realOutOfMaintenanceRecord) {
        // 设备表到今日维保超期设备realOutOfMaintenanceRecord
        if (ValidationUtil.isEmpty(realOutOfMaintenanceRecord)) {
            return Lists.newArrayList();
        }
        List<SafetyProblemTracing> problemList = safetyProblemTracingService.list(new LambdaQueryWrapper<SafetyProblemTracing>()
                .select(SafetyProblemTracing::getSourceId)
                .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.WBCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
        );
        if (ValidationUtil.isEmpty(problemList)) {
            return Lists.newArrayList();
        }
        // 追溯问题表中未处理的维保超期设备
        List<String> outOfMaintenanceEquipIds = problemList.stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toList());
        // 未超期但错误生成的维保超期设备
        outOfMaintenanceEquipIds.removeIf(realOutOfMaintenanceRecord::contains);
        if (ValidationUtil.isEmpty(outOfMaintenanceEquipIds)) {
            return Lists.newArrayList();
        }
        safetyProblemTracingService.lambdaUpdate()
                .set(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.HANDLED.getCode())
                .set(SafetyProblemTracing::getProblemStatus, SafetyProblemStatusEnum.HANDLED.getName())
                .set(SafetyProblemTracing::getRecDate, new Date())
                .in(SafetyProblemTracing::getSourceId, outOfMaintenanceEquipIds)
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.WBCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
                .update();
        return outOfMaintenanceEquipIds;
    }

    private List<String> handleOutOfInspectionErrorData(Set<String> realOutOfInspectRecord) {
        // 设备表到今日检验超期设备realOutOfInspectRecord
        if (ValidationUtil.isEmpty(realOutOfInspectRecord)) {
            return Lists.newArrayList();
        }
        List<SafetyProblemTracing> problemList = safetyProblemTracingService.list(new LambdaQueryWrapper<SafetyProblemTracing>()
                .select(SafetyProblemTracing::getSourceId)
                .eq(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.UNHANDLED.getCode())
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.JYCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
        );
        if (ValidationUtil.isEmpty(problemList)) {
            return Lists.newArrayList();
        }
        // 追溯问题表中未处理的检验超期设备
        List<String> outOfInspectionEquipIds = problemList.stream().map(SafetyProblemTracing::getSourceId).collect(Collectors.toList());
        // 未超期但错误生成的检验超期设备
        outOfInspectionEquipIds.removeIf(realOutOfInspectRecord::contains);
        if (ValidationUtil.isEmpty(outOfInspectionEquipIds)) {
            return Lists.newArrayList();
        }
        safetyProblemTracingService.lambdaUpdate()
                .set(SafetyProblemTracing::getProblemStatusCode, SafetyProblemStatusEnum.HANDLED.getCode())
                .set(SafetyProblemTracing::getProblemStatus, SafetyProblemStatusEnum.HANDLED.getName())
                .set(SafetyProblemTracing::getRecDate, new Date())
                .in(SafetyProblemTracing::getSourceId, outOfInspectionEquipIds)
                .eq(SafetyProblemTracing::getProblemTypeCode, SafetyProblemTypeEnum.JYCQ.getProblemTypeCode())
                .eq(SafetyProblemTracing::getIsDelete,Boolean.FALSE)
                .update();
        return outOfInspectionEquipIds;
    }

    private void updateEquipAndSendMessage(List<Map<String, Object>> mapList, SafetyProblemTypeEnum safetyProblemTypeEnum)  {
        if (ValidationUtil.isEmpty(mapList)) {
            return;
        }
        List<String> records = mapList.stream().map(m -> m.get("RECORD").toString()).collect(Collectors.toList());
        int batchSize = 1000;
        for (int i = 0; i < records.size(); i += batchSize) {
            List<String> batch = records.subList(i, Math.min(i + batchSize, records.size()));
            // 更新设备状态为未处理（异常）
            idxBizJgOtherInfoService.lambdaUpdate()
                    .in(IdxBizJgOtherInfo::getRecord, batch)
                    .set(IdxBizJgOtherInfo::getStatus, SafetyProblemStatusEnum.UNHANDLED.getCode())
                    .update();
            List<ESEquipmentCategoryDto> esEquipmentCategoryDto = Lists.newArrayList();
            for (ESEquipmentCategoryDto equipmentCategoryDto : esEquipmentCategory.findAllById(batch)) {
                equipmentCategoryDto.setProblemStatus(SafetyProblemStatusEnum.UNHANDLED.getCode());
                esEquipmentCategoryDto.add(equipmentCategoryDto);
            }
            esEquipmentCategory.saveAll(esEquipmentCategoryDto);
            // mapList也根据batch分割
            List<Map<String, Object>> mapListBatch = mapList.subList(i, Math.min(i + batchSize, mapList.size()));
            sendSafetyProblemMessage(mapListBatch, safetyProblemTypeEnum);
        }
    }

    private void updateEnterpriseAndSendMessage(List<Map<String, Object>> mapList) {
        if (ValidationUtil.isEmpty(mapList)) {
            return;
        }
        Set<String> outOfUnitLicenseList = mapList.stream().map(m -> m.get("useUnitCode").toString()).collect(Collectors.toSet());
        if (ValidationUtil.isEmpty(outOfUnitLicenseList)) {
            return;
        }
        // 更新企业问题状态为未处理（异常）-多个许可超期归并一个企业问题
        commonMapper.updateEnterpriseSafetyStatus(outOfUnitLicenseList);
        sendSafetyProblemMessage(mapList, SafetyProblemTypeEnum.XKCQ);
    }

    private void updateUserInfoAndSendMessage(List<Map<String, Object>> certificationRecords) {
        if (ValidationUtil.isEmpty(certificationRecords)) {
            return;
        }
        Set<String> outOfUnitLicenseList = certificationRecords.stream().map(m -> m.get("userSeq").toString()).collect(Collectors.toSet());
        if (!ValidationUtil.isEmpty(outOfUnitLicenseList)) {
                LambdaUpdateWrapper<TzsUserInfo> up = new LambdaUpdateWrapper();
                up.in(BaseEntity::getSequenceNbr,outOfUnitLicenseList);
                up.set(TzsUserInfo::getQrCodeState,0);
                tzsUserInfoMapper.update(null,up);
        }
        sendSafetyProblemMessage(certificationRecords, SafetyProblemTypeEnum.ZZCQ);

    }

    /**
     * 发送安全问题
     * @param mapList
     * @param safetyProblemTypeEnum
     */
    private void sendSafetyProblemMessage(List<Map<String, Object>> mapList, SafetyProblemTypeEnum safetyProblemTypeEnum) {
        if (CollectionUtil.isNotEmpty(mapList)){
            try {
                logger.info("发送安全追溯问题主题---->{}", safetyProblemTypeEnum.getTopic());
                byte[] payload = JSON.toJSONBytes(mapList);
                logger.info("发送数据大小：{} KB", payload.length / 1024);
                emqKeeper.getMqttClient().publish(safetyProblemTypeEnum.getTopic(), JSON.toJSONBytes(mapList), 2, false);
                logger.info("发送安全追溯问题消息成功---->");
            } catch (MqttException e) {
                logger.error("发送安全追溯问题设备信息消息失败---->{}", e.getMessage());
                throw new RuntimeException(e);
            }
        }
    }

//    @Scheduled(cron = "0 0 1 * * ?")
//    @SchedulerLock(name = "executeOverDesignLifeCheck", lockAtMostFor = "PT5H", lockAtLeastFor = "PT10M")
    public void executeOverDesignLife() {
        executeOverDesignLifeCheck();
    }

    public void executeOverDesignLifeCheck() {
        logger.info("开始生成超设计使用年限问题");
        List<Map<String, Object>> allEquipRecords = Lists.newArrayList();
        // 查询当天超设计使用年限的设备-首次超期
        List<Map<String, Object>> overDesignLifeRecords = commonMapper.queryOverDesignLifeRecord(new Date());
        if (!ValidationUtil.isEmpty(overDesignLifeRecords)) {
            allEquipRecords.addAll(overDesignLifeRecords);
        }
        // 查询当天超设计使用年限的设备-办理超期登记后再次超期
        List<Map<String, Object>> overDesignLifeAgainRecords = commonMapper.queryOverDesignLifeAgainRecord(new Date());
        if (!ValidationUtil.isEmpty(overDesignLifeRecords)) {
            allEquipRecords.addAll(overDesignLifeAgainRecords);
        }
        updateEquipAndSendMessage(allEquipRecords, SafetyProblemTypeEnum.SJNXCQ);
        logger.info("生成超设计使用年限问题结束");
    }
}