import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 通用气瓶字段校验枚举示例
 * 支持多种气瓶类型，每种类型可以有不同的必填和唯一性要求
 */
public class CylinderFieldValidationExample {
    
    /**
     * 字段校验接口
     */
    public interface FieldValidation {
        boolean isRequired();
        boolean isUnique();
        String getFieldName();
    }
    
    /**
     * 通用气瓶字段枚举
     */
    public enum CylinderField implements FieldValidation {
        // 通用字段
        SEQUENCE_CODE("sequenceCode", true, true),
        CREDIT_CODE("creditCode", true, false),
        FILLING_UNIT_NAME("fillingUnitName", true, false),
        INSPECTOR_USER("inspectorUser", true, false),
        INSPECTION_DATE("inspectionDate", true, false),
        IS_VALID("isValid", false, false),
        IS_SAME("same", false, false),
        IS_REGULATIONS("isRegulations", false, false),
        IS_COMPLIANCE_WITH_GBT("isComplianceWithGBT", false, false),
        HAVE_STILL_PRESSURE("haveStillPressure", false, false),
        IS_COMPLETE("isComplete", false, false),
        HAVE_SECURITY_DOCUMENTS("haveSecurityDocuments", false, false),
        FILL_BEFORE_ITEM("fillBeforeItem", false, false),
        CHECK_RESULTS("checkResults", false, false),
        NONCONFORMANCES("nonconformances", false, false),
        SYNC_DATE("syncDate", false, false),
        SYNC_STATE("syncState", false, false);
        
        private final String fieldName;
        private final boolean required;
        private final boolean unique;
        
        CylinderField(String fieldName, boolean required, boolean unique) {
            this.fieldName = fieldName;
            this.required = required;
            this.unique = unique;
        }
        
        @Override
        public boolean isRequired() {
            return required;
        }
        
        @Override
        public boolean isUnique() {
            return unique;
        }
        
        @Override
        public String getFieldName() {
            return fieldName;
        }
        
        public static List<String> getRequiredFields() {
            return Arrays.stream(values())
                    .filter(FieldValidation::isRequired)
                    .map(FieldValidation::getFieldName)
                    .collect(Collectors.toList());
        }
        
        public static List<String> getUniqueFields() {
            return Arrays.stream(values())
                    .filter(FieldValidation::isUnique)
                    .map(FieldValidation::getFieldName)
                    .collect(Collectors.toList());
        }
    }
    
    /**
     * 气瓶类型枚举
     */
    public enum CylinderType {
        INDUSTRIAL_GAS_CYLINDER("工业气瓶"),
        LIQUEFIED_GAS_CYLINDER("液化石油气瓶"),
        OTHER_CYLINDER("其他气瓶");
        
        private final String description;
        
        CylinderType(String description) {
            this.description = description;
        }
        
        public String getDescription() {
            return description;
        }
    }
    
    /**
     * 特定气瓶类型的字段校验枚举
     * 通过组合模式支持不同类型气瓶的特殊要求
     */
    public enum CylinderTypeFieldValidation {
        // 工业气瓶特殊要求
        INDUSTRIAL_CYLINDER_UNIQUE_FIELD(CylinderType.INDUSTRIAL_GAS_CYLINDER, "manufactureDate", false, true),
        INDUSTRIAL_CYLINDER_REQUIRED_FIELD(CylinderType.INDUSTRIAL_GAS_CYLINDER, "materialType", true, false),
        
        // 液化石油气瓶特殊要求
        LIQUEFIED_GAS_CYLINDER_UNIQUE_FIELD(CylinderType.LIQUEFIED_GAS_CYLINDER, "volume", false, true),
        LIQUEFIED_GAS_CYLINDER_REQUIRED_FIELD(CylinderType.LIQUEFIED_GAS_CYLINDER, "wallThickness", true, false),
        
        // 其他气瓶特殊要求
        OTHER_CYLINDER_UNIQUE_FIELD(CylinderType.OTHER_CYLINDER, "customId", false, true),
        OTHER_CYLINDER_REQUIRED_FIELD(CylinderType.OTHER_CYLINDER, "specification", true, false);
        
        private final CylinderType cylinderType;
        private final String fieldName;
        private final boolean required;
        private final boolean unique;
        
        CylinderTypeFieldValidation(CylinderType cylinderType, String fieldName, boolean required, boolean unique) {
            this.cylinderType = cylinderType;
            this.fieldName = fieldName;
            this.required = required;
            this.unique = unique;
        }
        
        public CylinderType getCylinderType() {
            return cylinderType;
        }
        
        public String getFieldName() {
            return fieldName;
        }
        
        public boolean isRequired() {
            return required;
        }
        
        public boolean isUnique() {
            return unique;
        }
        
        /**
         * 获取指定气瓶类型的所有必填字段
         * @param type 气瓶类型
         * @return 必填字段列表
         */
        public static List<String> getRequiredFieldsForType(CylinderType type) {
            // 先获取通用必填字段
            List<String> requiredFields = CylinderField.getRequiredFields();
            
            // 再添加特定类型必填字段
            requiredFields.addAll(Arrays.stream(values())
                    .filter(validation -> validation.cylinderType == type)
                    .filter(CylinderTypeFieldValidation::isRequired)
                    .map(CylinderTypeFieldValidation::getFieldName)
                    .collect(Collectors.toList()));
            
            return requiredFields;
        }
        
        /**
         * 获取指定气瓶类型的所有唯一字段
         * @param type 气瓶类型
         * @return 唯一字段列表
         */
        public static List<String> getUniqueFieldsForType(CylinderType type) {
            // 先获取通用唯一字段
            List<String> uniqueFields = CylinderField.getUniqueFields();
            
            // 再添加特定类型唯一字段
            uniqueFields.addAll(Arrays.stream(values())
                    .filter(validation -> validation.cylinderType == type)
                    .filter(CylinderTypeFieldValidation::isUnique)
                    .map(CylinderTypeFieldValidation::getFieldName)
                    .collect(Collectors.toList()));
            
            return uniqueFields;
        }
    }
    
    public static void main(String[] args) {
        System.out.println("=== 通用气瓶字段校验 ===");
        System.out.println("通用必填字段: " + CylinderField.getRequiredFields());
        System.out.println("通用唯一字段: " + CylinderField.getUniqueFields());
        
        System.out.println("\n=== 工业气瓶字段校验 ===");
        System.out.println("必填字段: " + CylinderTypeFieldValidation.getRequiredFieldsForType(CylinderType.INDUSTRIAL_GAS_CYLINDER));
        System.out.println("唯一字段: " + CylinderTypeFieldValidation.getUniqueFieldsForType(CylinderType.INDUSTRIAL_GAS_CYLINDER));
        
        System.out.println("\n=== 液化石油气瓶字段校验 ===");
        System.out.println("必填字段: " + CylinderTypeFieldValidation.getRequiredFieldsForType(CylinderType.LIQUEFIED_GAS_CYLINDER));
        System.out.println("唯一字段: " + CylinderTypeFieldValidation.getUniqueFieldsForType(CylinderType.LIQUEFIED_GAS_CYLINDER));
        
        System.out.println("\n=== 其他气瓶字段校验 ===");
        System.out.println("必填字段: " + CylinderTypeFieldValidation.getRequiredFieldsForType(CylinderType.OTHER_CYLINDER));
        System.out.println("唯一字段: " + CylinderTypeFieldValidation.getUniqueFieldsForType(CylinderType.OTHER_CYLINDER));
    }
}