package com.yeejoin.amos.boot.module.ys.biz.service.impl;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.yeejoin.amos.boot.module.common.biz.service.impl.DynamicFormInstanceServiceImpl;
import com.yeejoin.amos.boot.module.ymt.api.enums.EquipmentClassifityEnum;
import com.yeejoin.amos.boot.module.ys.api.entity.YsEmergencyPlan;
import com.yeejoin.amos.boot.module.ys.api.entity.YsEmergencyRehearsal;
import com.yeejoin.amos.boot.module.ys.api.enums.MaintenanceExpiredEarlyWarningEnum;
import com.yeejoin.amos.boot.module.ys.api.vo.SortVo;
import com.yeejoin.amos.boot.module.ys.biz.service.CommonService;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.core.CountRequest;
import org.elasticsearch.client.core.CountResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 公共服务实现类
 *
 * @author system_generator
 * @date 2024-09-24
 */
@Service
public class CommonServiceImpl implements CommonService {

    /**
     * 气瓶设备类别
     */
    public final static String EQU_CATEGORY_CYLINDER = "2300";

    /**
     * 应急保障四项统计使用 - 饼图
     */
    public final static String PIECHART = "pieChart";
    /**
     * 应急保障四项统计使用 - 饼图模板
     */
    public final static String PIECHART_TEMPLATE = "[{\"name\": \"%s\",\"value\": %s},{\"name\": \"其他\",\"value\": %s}]";
    /**
     * 应急保障四项统计使用 - 富文本
     */
    public final static String RICHTEXT = "richText";
    /**
     * 应急保障四项统计使用 - 富文本模板
     */
    public final static String RICHTEXT_TEMPLATE = "{\"total\": %s,\"%s\": %s}";

    /**
     * 应急保障四项统计使用 - 业务类型
     */
    public final static JSONObject businessTypeJSON = new JSONObject();
    private static final Logger log = LoggerFactory.getLogger(CommonServiceImpl.class);

    @Autowired
    private RestHighLevelClient restHighLevelClient;

    @Autowired
    private DynamicFormInstanceServiceImpl dynamicFormInstanceService;

    @Autowired
    private YsEmergencyRehearsalServiceImpl ysEmergencyRehearsalService;

    @Autowired
    private YsEmergencyPlanServiceImpl ysEmergencyPlanService;

    static {
        businessTypeJSON
                .fluentPut("education", "教育培训数").fluentPut("construct", "体系建设数")
                .fluentPut("rehearsal", "应急演练数").fluentPut("prePlan", "应急预案数");
    }

    /**
     * 排序  ：页面列表排序功能支持，将 "字段,ascend" 或 "字段,descend" 转化为对应JSONObject
     *
     * @param sort "字段,ascend" 或 "字段,descend"
     * @return JSONObject
     */
    public SortVo sortFieldConversion(String sort) {
        Optional<String> optionalSort = Optional.ofNullable(sort);
        Optional<SortVo> optionalSortMap = optionalSort.filter(s -> !s.isEmpty())
                .map(s -> {
                    String[] sortParts = s.split(",");
                    if (sortParts.length == 2) {
                        String field = sortParts[0];
                        String sortSituation = sortParts[1].contains("asc") ? "ASC" : "DESC";
                        return SortVo.builder()
                                .field(convertToUnderline(field))
                                .sortType(sortSituation)
                                .build();
                    }
                    return null;
                });
        return optionalSortMap.orElse(null);
    }

    /**
     * 驼峰转下划线
     *
     * @param str
     * @return
     */
    public static String convertToUnderline(String str) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (Character.isUpperCase(c)) {
                sb.append("_").append(Character.toLowerCase(c));
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /**
     * 气瓶数量统计
     *
     * @param orgCode 区域code
     * @return
     */
    public long staticsCenterMapCountDataForCylinder(String orgCode) {
        long num = 0;
        CountRequest request = new CountRequest();
        request.indices("idx_biz_view_jg_all");
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        // 按照管辖机构区域信息模糊查询
        boolMust.must(QueryBuilders.wildcardQuery("ORG_BRANCH_CODE.keyword", QueryParser.escape(orgCode) + "*"));
        // 设备类别精确查询气瓶
        boolMust.must(QueryBuilders.termQuery("EQU_CATEGORY_CODE", EQU_CATEGORY_CYLINDER));
        // 只统计已纳管设备
        boolMust.must(QueryBuilders.termQuery("IS_INTO_MANAGEMENT", Boolean.TRUE));
        // 只统计有监管码的、状态不是草稿且不是已拒领的
        boolMust.must(QueryBuilders.existsQuery("SUPERVISORY_CODE"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "草稿"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "已拒领"));
        request.query(boolMust);
        try {
            CountResponse response = restHighLevelClient.count(request, RequestOptions.DEFAULT);
            num = response.getCount();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return num;
    }

    /**
     * 八大类设备数量统计
     *
     * @param equTypeCode
     * @param cylinderNum
     * @param orgCode
     * @return
     */
    public long staticsCenterMapCountDataForEquip(String equTypeCode, long cylinderNum, String orgCode) {
        AtomicLong result = new AtomicLong(0L);
        SearchRequest request = new SearchRequest();
        request.indices("idx_biz_view_jg_all");
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        // 按照管辖机构区域信息模糊查询
        boolMust.must(QueryBuilders.wildcardQuery("ORG_BRANCH_CODE.keyword", QueryParser.escape(orgCode) + "*"));
        // 只统计已纳管设备
        boolMust.must(QueryBuilders.termQuery("IS_INTO_MANAGEMENT", Boolean.TRUE));
        // 只统计有监管码的、状态不是草稿且不是已拒领的
        boolMust.must(QueryBuilders.existsQuery("SUPERVISORY_CODE"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "草稿"));
        boolMust.mustNot(QueryBuilders.termQuery("STATUS", "已拒领"));
        boolMust.must(QueryBuilders.termQuery("EQU_LIST_CODE", equTypeCode));
        SearchSourceBuilder builder = new SearchSourceBuilder();
        builder.query(boolMust);
        // 默认size为10 ，数据库有脏数据，所以需要多查询一些
        TermsAggregationBuilder aggregationBuilder = AggregationBuilders.terms("count_by_equ_list_code").field("EQU_LIST_CODE").size(20);
        builder.aggregation(aggregationBuilder);
        request.source(builder);
        try {
            SearchResponse response = restHighLevelClient.search(request, RequestOptions.DEFAULT);
            Terms terms = response.getAggregations().get("count_by_equ_list_code");

            for (Terms.Bucket bucket : terms.getBuckets()) {
                if (equTypeCode.equals(bucket.getKeyAsString())) {
                    // 压力容器里包括气瓶，所以需要特殊处理
                    long count = bucket.getDocCount();
                    result.set(bucket.getKeyAsString().equals(EquipmentClassifityEnum.YLRQ.getCode())
                            ? count - cylinderNum
                            : count);
                    break;
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result.get();
    }

    /**
     * 应急保障四项统计
     *
     * @param businessType  业务类型（教育培训：education、体系建设：construct、应急演练：rehearsal、应急预案：prePlan）
     * @param componentType 组件类型（饼图：pieChart、富文本：richText）
     * @return 模板数据
     */
    @Override
    public Object emergencySupportStatistics(String businessType, String componentType) {
        // 教育培训数
        long educationNum = dynamicFormInstanceService.pageList(1, 1, MaintenanceExpiredEarlyWarningEnum.JYPX.getGroupCode()).getTotal();
        // 体系建设数
        long constructNum = dynamicFormInstanceService.pageList(1, 1, MaintenanceExpiredEarlyWarningEnum.TXJS.getGroupCode()).getTotal();
        // 应急演练数
        int rehearsalNum = ysEmergencyRehearsalService.getBaseMapper().selectCount(new LambdaQueryWrapper<YsEmergencyRehearsal>().eq(YsEmergencyRehearsal::getIsDelete, Boolean.FALSE));
        // 应急预案数
        int prePlanNum = ysEmergencyPlanService.getBaseMapper().selectCount(new LambdaQueryWrapper<YsEmergencyPlan>().eq(YsEmergencyPlan::getIsDelete, Boolean.FALSE));
        // 四项总数
        BigDecimal totalNum = NumberUtil.add(educationNum, constructNum, rehearsalNum, prePlanNum);
        // 其他数量
        BigDecimal otherNum = new BigDecimal(0);
        // 请求类型数量
        BigDecimal valueNum = new BigDecimal(0);
        switch (businessType) {
            case "education":
                valueNum = BigDecimal.valueOf(educationNum);
                otherNum = NumberUtil.sub(totalNum, educationNum);
                break;
            case "construct":
                valueNum = BigDecimal.valueOf(constructNum);
                otherNum = NumberUtil.sub(totalNum, constructNum);
                break;
            case "rehearsal":
                valueNum = BigDecimal.valueOf(rehearsalNum);
                otherNum = NumberUtil.sub(totalNum, rehearsalNum);
                break;
            case "prePlan":
                valueNum = BigDecimal.valueOf(prePlanNum);
                otherNum = NumberUtil.sub(totalNum, prePlanNum);
                break;
        }
        if (PIECHART.equals(componentType)) {
            // 组装 饼图 模板数据
            return JSONArray.parseArray(String.format(PIECHART_TEMPLATE, businessTypeJSON.get(businessType).toString(), valueNum, otherNum));
        }
        if (RICHTEXT.equals(componentType)) {
            // 组装 富文本 模板数据
            return JSONObject.parseObject(String.format(RICHTEXT_TEMPLATE, totalNum, businessType, valueNum));
        }
        return null;
    }
}