package com.yeejoin.amos.boot.module.jg.biz.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.yeejoin.amos.boot.biz.common.bo.CompanyBo;
import com.yeejoin.amos.boot.biz.common.bo.ReginParams;
import com.yeejoin.amos.boot.biz.common.utils.RedisUtils;
import com.yeejoin.amos.boot.module.jg.api.bo.JgBizCountDataBO;
import com.yeejoin.amos.boot.module.jg.api.common.StringUtil;
import com.yeejoin.amos.boot.module.jg.api.dto.JgBizCountDataDto;
import com.yeejoin.amos.boot.module.jg.api.dto.ReportAnalysisSearchDTO;
import com.yeejoin.amos.boot.module.jg.api.mapper.CommonMapper;
import com.yeejoin.amos.boot.module.jg.api.service.IReportAnalysisService;
import com.yeejoin.amos.boot.module.jg.api.vo.JgBizCountBaseVo;
import com.yeejoin.amos.boot.module.jg.api.vo.JgBizEquipStateCountVo;
import com.yeejoin.amos.boot.module.ymt.api.entity.EquipmentCategory;
import com.yeejoin.amos.boot.module.ymt.api.enums.FlowStatusEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.script.Script;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.typroject.tyboot.core.foundation.utils.Bean;
import org.typroject.tyboot.core.restful.exception.instance.BadRequest;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Administrator
 */
@Service
@Slf4j
public class ReportAnalysisServiceImpl implements IReportAnalysisService {

    private CommonMapper commonMapper;

    private CommonServiceImpl commonService;

    private RestHighLevelClient restHighLevelClient;

    private Map<String, String> codeNameMap;

    private Map<String, String> companyCodeNameMap;

    private Map<String, String> orgCodeNameMap;

    private RedisUtils redisUtils;

    private String RECEIVE_COUNT_SEARCH_REDIS_KEY = "RECEIVE_COUNT_SEARCH_REDIS_KEY";

    @Value("${sx.province.org.code:50}")
    private String rootOrgCode;

    /**
     * 告知类已发起统计数据范围： 来源FlowStatusEnum： 待受理 + 已驳回 + 已完成
     */
    private List<String> noticeLaunchesStatusList = Arrays.asList("6612", "6614", "6616");

    /**
     * 维保备案已发起统计数据范围
     */
    private List<String> mainLaunchesStatusList = Arrays.asList("维保单位已撤回", "维保单位已驳回", "维保单位待受理", "监管单位已驳回", "监管单位待受理", "已完成");

    /**
     * 使用登记已发起统计数据范围
     */
    private List<String> registerLaunchesStatusList = Arrays.asList("一级待受理", "一级受理已驳回", "一级受理已撤回", "二级待受理", "二级受理已驳回", "二级受理已撤回", "三级待受理", "三级受理已驳回", "已完成");


    public ReportAnalysisServiceImpl(CommonMapper commonMapper, RestHighLevelClient restHighLevelClient, RedisUtils redisUtils, CommonServiceImpl commonService) {
        this.commonMapper = commonMapper;
        this.restHighLevelClient = restHighLevelClient;
        this.redisUtils = redisUtils;
        this.commonService = commonService;
    }

    @PostConstruct
    private void init() {
        // 缓存设备分类信息
        initEquipCategoryCache();
        // 缓存公司信息：待优化缓存 保障可以不需要重启
        initCompanyCache();
    }

    /**
     * 缓存根节点的数据的接收机构数据
     */
    public void initRootOrgReceiveTree2Redis() {
        getCompanyBos(rootOrgCode, true);
        getCompanyBos(rootOrgCode, false);
    }

    private void initEquipCategoryCache() {
        LambdaQueryWrapper<EquipmentCategory> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.select(EquipmentCategory::getCode, EquipmentCategory::getName);
        List<EquipmentCategory> equipmentCategoryList = commonMapper.selectList(queryWrapper);
        codeNameMap = equipmentCategoryList.stream().collect(Collectors.toMap(EquipmentCategory::getCode, EquipmentCategory::getName));
    }

    private void initCompanyCache() {
        List<CompanyBo> companyBos = commonMapper.queryJGCompanyCodeName(null);
        companyCodeNameMap = companyBos.stream().collect(Collectors.toMap(CompanyBo::getCompanyCode, CompanyBo::getCompanyName, (k1, k2) -> k2));
        orgCodeNameMap = companyBos.stream().collect(Collectors.toMap(CompanyBo::getOrgCode, CompanyBo::getCompanyName, (k1, k2) -> k2));
    }

    @Override
    public List<JgBizCountBaseVo> getNoticeCountList(ReportAnalysisSearchDTO reportAnalysisSearchDTO) {
        List<JgBizCountDataBO> countDataBOList = commonMapper.queryForNoticeCountData(reportAnalysisSearchDTO);
        List<JgBizCountBaseVo> result = combineCountData(countDataBOList);
        sortData(result, reportAnalysisSearchDTO);
        filledCommonCountSummaryRow(result);
        return result;
    }

    private void sortData(List<JgBizCountBaseVo> countBaseVoList, ReportAnalysisSearchDTO reportAnalysisSearchDTO) {
        JgBizCountBaseVo.sortList(countBaseVoList, reportAnalysisSearchDTO.getSortField(), reportAnalysisSearchDTO.getSortOrder());
    }

    @Override
    public List<JgBizCountBaseVo> getRegisterCountList(ReportAnalysisSearchDTO reportAnalysisSearchDTO) {
        List<JgBizCountDataBO> countDataBOList = commonMapper.queryForRegisterCountData(reportAnalysisSearchDTO);
        List<JgBizCountBaseVo> result = combineRegisterCountData(countDataBOList);
        sortData(result, reportAnalysisSearchDTO);
        filledCommonCountSummaryRow(result);
        return result;
    }

    private List<JgBizCountBaseVo> combineCountData(List<JgBizCountDataBO> countDataBOList) {
        // 去重
        Set<JgBizCountDataDto> countVoSet = distinctJgBizCountDataDtos(countDataBOList);
        // 已去重行数据为集合，填充每种流程状态下单据的数量及将code->name
        return countVoSet.stream().map(c -> {
            JgBizCountBaseVo vo = new JgBizCountBaseVo();
            vo.setBizGroup("施工告知");
            vo.setBizType(c.getBizType());
            vo.setReceiveCompanyName(companyCodeNameMap.get(c.getReceiveCompanyCode()));
            vo.setEquListName(getEquipCategoryNameByCode(c.getEquList()));
            vo.setEquCategoryName(getEquipCategoryNameByCode(c.getEquCategory()));
            vo.setEquDefineName(getEquipCategoryNameByCode(c.getEquDefine()));
            vo.setFinishedNumber(this.countDocumentNum(countDataBOList, FlowStatusEnum.TO_BE_FINISHED.getCode(), c));
            // 已发起
            vo.setLaunchesNumber(this.countDocumentNum(countDataBOList, noticeLaunchesStatusList, c));
            return vo;
        }).collect(Collectors.toList());
    }

    private String getReceiveCompanyNameReal(String receiveCompanyCode, String defaultName) {
        return commonService.getRegName(receiveCompanyCode, defaultName);
    }

    private Set<JgBizCountDataDto> distinctJgBizCountDataDtos(List<JgBizCountDataBO> countDataBOList) {
        return countDataBOList.stream().map(c -> {
            JgBizCountDataDto target = new JgBizCountDataDto();
            Bean.copyExistPropertis(c, target);
            return target;
        }).collect(Collectors.toSet());
    }

    private Long countDocumentNum(List<JgBizCountDataBO> countDataBOList, int code, JgBizCountDataDto c) {
        return countDataBOList.stream().filter(d -> Objects.equals(d.getReceiveCompanyCode(), c.getReceiveCompanyCode()) &&
                Objects.equals(d.getEquList(), c.getEquList()) && Objects.equals(d.getEquCategory(), c.getEquCategory()) &&
                Objects.equals(d.getEquCategory(), c.getEquCategory()) &&
                Objects.equals(d.getBizType(), c.getBizType()) &&
                d.getStatus().equals(String.valueOf(code))).mapToLong(JgBizCountDataBO::getNum).sum();

    }

    private Long countDocumentNum(List<JgBizCountDataBO> countDataBOList, List<String> statusList, JgBizCountDataDto c) {
        return countDataBOList.stream().filter(d -> Objects.equals(d.getReceiveCompanyCode(), c.getReceiveCompanyCode()) &&
                Objects.equals(d.getEquList(), c.getEquList()) && Objects.equals(d.getEquCategory(), c.getEquCategory()) &&
                Objects.equals(d.getEquCategory(), c.getEquCategory()) &&
                Objects.equals(d.getBizType(), c.getBizType()) &&
                statusList.contains(d.getStatus())).mapToLong(JgBizCountDataBO::getNum).sum();

    }

    private String getEquipCategoryNameByCode(String code) {
        return StringUtil.isNotEmpty(code) ? codeNameMap.get(code) : "";
    }


    @Override
    public List<JgBizCountBaseVo> getMaintenanceCountList(ReportAnalysisSearchDTO reportAnalysisSearchDTO) {
        List<JgBizCountDataBO> countDataBOList = commonMapper.queryForMaintenanceCountData(reportAnalysisSearchDTO);
        List<JgBizCountBaseVo> result = combineMaintenanceCountData(countDataBOList);
        sortData(result, reportAnalysisSearchDTO);
        filledCommonCountSummaryRow(result);
        return result;
    }


    private void filledCommonCountSummaryRow(List<JgBizCountBaseVo> result) {
        if (result.size() > 0) {
            JgBizCountBaseVo summaryRow = new JgBizCountBaseVo();
            summaryRow.setFinishedNumber(result.stream().mapToLong(JgBizCountBaseVo::getFinishedNumber).sum());
            summaryRow.setLaunchesNumber(result.stream().mapToLong(JgBizCountBaseVo::getLaunchesNumber).sum());
            result.add(summaryRow);
        }
    }


    private List<JgBizCountBaseVo> combineMaintenanceCountData(List<JgBizCountDataBO> countDataBOList) {
        // 去重
        Set<JgBizCountDataDto> countVoSet = distinctJgBizCountDataDtos(countDataBOList);
        // 已去重行数据为集合，填充每种流程状态下单据的数量及将code->name
        return countVoSet.stream().map(c -> {
            JgBizCountBaseVo vo = new JgBizCountBaseVo();
            vo.setBizGroup("维保备案");
            return buildJgBizCountBaseVo(countDataBOList, c, vo, mainLaunchesStatusList, false);
        }).collect(Collectors.toList());
    }

    private List<JgBizCountBaseVo> combineRegisterCountData(List<JgBizCountDataBO> countDataBOList) {
        // 去重
        Set<JgBizCountDataDto> countVoSet = distinctJgBizCountDataDtos(countDataBOList);
        // 已去重行数据为集合，填充每种流程状态下单据的数量及将code->name
        return countVoSet.stream().map(c -> {
            JgBizCountBaseVo vo = new JgBizCountBaseVo();
            vo.setBizGroup("使用登记");
            return buildJgBizCountBaseVo(countDataBOList, c, vo, registerLaunchesStatusList, true);
        }).collect(Collectors.toList());
    }


    private JgBizCountBaseVo buildJgBizCountBaseVo(List<JgBizCountDataBO> countDataBOList, JgBizCountDataDto c, JgBizCountBaseVo vo, List<String> registerLaunchesStatusList, Boolean isRegName) {
        vo.setBizType(c.getBizType());
        if (isRegName) {
            vo.setReceiveCompanyName(getReceiveCompanyNameReal(c.getReceiveCompanyCode(), companyCodeNameMap.get(c.getReceiveCompanyCode())));
        } else {
            vo.setReceiveCompanyName(companyCodeNameMap.get(c.getReceiveCompanyCode()));

        }
        vo.setEquListName(getEquipCategoryNameByCode(c.getEquList()));
        vo.setEquCategoryName(getEquipCategoryNameByCode(c.getEquCategory()));
        vo.setEquDefineName(getEquipCategoryNameByCode(c.getEquDefine()));
        vo.setFinishedNumber(this.countDocumentNum(countDataBOList, Collections.singletonList(FlowStatusEnum.TO_BE_FINISHED.getName()), c));
        vo.setLaunchesNumber(this.countDocumentNum(countDataBOList, registerLaunchesStatusList, c));
        return vo;
    }

    @Override
    public List<JgBizEquipStateCountVo> getEquipStatusCountList(ReportAnalysisSearchDTO reportAnalysisSearchDT) {
        if (StringUtils.isEmpty(reportAnalysisSearchDT.getReceiveOrgCode())) {
            throw new BadRequest("单位不能为空");
        }
        SearchRequest searchRequest = new SearchRequest("idx_biz_view_jg_all");
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        BoolQueryBuilder boolMust = QueryBuilders.boolQuery();
        // 前端筛选条件过滤
        this.buildRequestSearchParam(reportAnalysisSearchDT, boolMust);
        // 数据权限过滤
        this.buildPermissionLimit(reportAnalysisSearchDT, boolMust);
        // 设备范围过滤
        this.jgEquipFilterQueryBuild(boolMust);
        sourceBuilder.query(boolMust);
        sourceBuilder.size(0);
        // 由于java api查询的数据为镶套，非便扁平化数据，故将 设备状态 + 监管机构 + 设备种类 + 设备类别+ 设备品种 ，作为分组条件中间用 | 分割，注意顺序 解析时按照此字段顺序解析
        Script script = new Script(Script.DEFAULT_SCRIPT_TYPE, Script.DEFAULT_SCRIPT_LANG, "if(doc['EQU_DEFINE_CODE.keyword'].size() != 0){ return  doc['EQU_STATE'].value + '|' + doc['ORG_BRANCH_CODE.keyword'].value + '|' + doc['EQU_LIST_CODE'].value + '|' + doc['EQU_CATEGORY_CODE'].value + '|' + doc['EQU_DEFINE_CODE.keyword'].value} else { return  doc['EQU_STATE'].value + '|' + doc['ORG_BRANCH_CODE.keyword'].value + '|' + doc['EQU_LIST_CODE'].value + '|' + doc['EQU_CATEGORY_CODE'].value + '|' + 'no'}", new HashMap<>());
        TermsAggregationBuilder aggregation = AggregationBuilders.terms("result").script(script);
        sourceBuilder.aggregation(aggregation);
        searchRequest.source(sourceBuilder);
        try {
            SearchResponse searchResponse = restHighLevelClient.search(searchRequest, RequestOptions.DEFAULT);
            Terms terms = searchResponse.getAggregations().get("result");
            List<JgBizCountDataBO> jgBizCountDataBOList = new ArrayList<>();
            for (Terms.Bucket bucket : terms.getBuckets()) {
                JgBizCountDataBO bizCountData = esItemToVo(bucket);
                jgBizCountDataBOList.add(bizCountData);
            }
            List<JgBizEquipStateCountVo> result = combineEquipStatusCountData(jgBizCountDataBOList);
            equipStateSortData(result, reportAnalysisSearchDT);
            filledEquipStateSummaryRow(result);
            return result;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return new ArrayList<>();
    }

    private void equipStateSortData(List<JgBizEquipStateCountVo> result, ReportAnalysisSearchDTO reportAnalysisSearchDT) {
        JgBizEquipStateCountVo.sortList(result, reportAnalysisSearchDT.getSortField(), reportAnalysisSearchDT.getSortOrder());
    }

    private void filledEquipStateSummaryRow(List<JgBizEquipStateCountVo> result) {
        if (result.size() > 0) {
            JgBizEquipStateCountVo summaryRow = new JgBizEquipStateCountVo();
            summaryRow.setTotalNumber(result.stream().mapToLong(JgBizEquipStateCountVo::getTotalNumber).sum());
            summaryRow.setInUseNumber(result.stream().mapToLong(JgBizEquipStateCountVo::getInUseNumber).sum());
            summaryRow.setScrapCancel(result.stream().mapToLong(JgBizEquipStateCountVo::getScrapCancel).sum());
            summaryRow.setStopUseNumber(result.stream().mapToLong(JgBizEquipStateCountVo::getStopUseNumber).sum());
            result.add(summaryRow);
        }
    }

    @Override
    public List<CompanyBo> getReceiveCompanyTreeWithAuth(ReginParams selectedOrgInfo, boolean isReg) {
        return getCompanyBos(selectedOrgInfo.getCompany().getOrgCode(), isReg);
    }

    @Override
    public List<Map<String, Object>> getBizTypeList(String groupKey) {
        return JgBizCountBaseVo.BizType.getBizTypeListByGroup(groupKey);
    }

    private List<CompanyBo> getCompanyBos(String orgCode, boolean isReg) {
        List<CompanyBo> companyBoList = commonMapper.queryJGCompanyCodeName(orgCode);
        // 设置接收机构名称与使用标志上登记机构名称一样
        if (isReg) {
            this.setCompany2RealName(companyBoList);
        }
        synchronized (this) {
            String redisKey = this.buildRedisKey(orgCode, isReg);
            if (redisUtils.hasKey(redisKey)) {
                return JSONObject.parseArray(redisUtils.get(redisKey).toString(), CompanyBo.class);
            } else {
                if (redisUtils.hasKey(redisKey)) {
                    return JSONObject.parseArray(redisUtils.get(redisKey).toString(), CompanyBo.class);
                } else {
                    return setComTree2Redis(redisKey, comList2ComTree(companyBoList, orgCode));
                }
            }
        }
    }

    private void setCompany2RealName(List<CompanyBo> companyBoList) {
        companyBoList.forEach(c -> {
            c.setCompanyName(getReceiveCompanyNameReal(c.getCompanyCode(), c.getCompanyName()));
        });
    }

    private List<CompanyBo> setComTree2Redis(String redisKey, List<CompanyBo> comList2ComTree) {
        redisUtils.set(redisKey, JSONObject.toJSONString(comList2ComTree), 10 * 60 * 60);
        return comList2ComTree;
    }

    private String buildRedisKey(String currentPersonOrgCode, boolean isReg) {
        return RECEIVE_COUNT_SEARCH_REDIS_KEY + ":" + isReg + currentPersonOrgCode;
    }


    private List<CompanyBo> comList2ComTree(List<CompanyBo> companyBoList, String currentPersonOrgCode) {
        return companyBoList.stream().filter(c -> c.getOrgCode().equals(currentPersonOrgCode)).peek(c -> c.setChildren(this.getComChild(c.getSequenceNbr(), companyBoList))).collect(Collectors.toList());
    }

    private List<CompanyBo> getComChild(Long parentId, List<CompanyBo> companyBoList) {
        return companyBoList.stream().filter(c -> c.getParentId().equals(parentId)).peek(c -> c.setChildren(this.getComChild(c.getSequenceNbr(), companyBoList))).collect(Collectors.toList());
    }

    private void buildRequestSearchParam(ReportAnalysisSearchDTO reportAnalysisSearchDT, BoolQueryBuilder boolMust) {
        if (StringUtils.isNotEmpty(reportAnalysisSearchDT.getEquList())) {
            boolMust.must(QueryBuilders.termQuery("EQU_LIST_CODE", QueryParser.escape(reportAnalysisSearchDT.getEquList().toLowerCase())));
        }
        if (StringUtils.isNotEmpty(reportAnalysisSearchDT.getEquCategory())) {
            boolMust.must(QueryBuilders.termQuery("EQU_CATEGORY_CODE", QueryParser.escape(reportAnalysisSearchDT.getEquCategory().toLowerCase())));
        }
        if (StringUtils.isNotEmpty(reportAnalysisSearchDT.getEquDefine())) {
            boolMust.must(QueryBuilders.termQuery("EQU_DEFINE_CODE.keyword", QueryParser.escape(reportAnalysisSearchDT.getEquDefine().toLowerCase())));
        }
    }

    private List<JgBizEquipStateCountVo> combineEquipStatusCountData(List<JgBizCountDataBO> countDataBOList) {
        // 去重
        Set<JgBizCountDataDto> countVoSet = distinctJgBizCountDataDtos(countDataBOList);
        // 已去重行数据为集合，填充每种流程状态下单据的数量及将code->name
        return countVoSet.stream().map(c -> {
            JgBizEquipStateCountVo vo = new JgBizEquipStateCountVo();
            // 此处注意：由于es 存放的org_code 非company_code
            vo.setManagementUnitName(orgCodeNameMap.get(c.getReceiveCompanyCode()));
            filledParentCompanyName(c, vo);
            vo.setEquListName(getEquipCategoryNameByCode(c.getEquList()));
            vo.setEquCategoryName(getEquipCategoryNameByCode(c.getEquCategory()));
            vo.setEquDefineName(getEquipCategoryNameByCode(c.getEquDefine()));
            // 设备状态： 1在用 2停用 3报废 4注销
            vo.setTotalNumber(this.countDocumentNum(countDataBOList, Arrays.asList("1", "2"), c));
            vo.setInUseNumber(this.countDocumentNum(countDataBOList, Collections.singletonList("1"), c));
            vo.setStopUseNumber(this.countDocumentNum(countDataBOList, Collections.singletonList("2"), c));
            vo.setScrapCancel(this.countDocumentNum(countDataBOList, Arrays.asList("3", "4"), c));
            return vo;
        }).collect(Collectors.toList());
    }

    private void filledParentCompanyName(JgBizCountDataDto c, JgBizEquipStateCountVo vo) {
        // 生产环境特殊数据处理，之前未限制只能选所
        if (c.getReceiveCompanyCode().split("\\*").length == 2) {
            // 直接上级（区县）
            String parentOrgCode = c.getReceiveCompanyCode().substring(0, c.getReceiveCompanyCode().lastIndexOf("*"));
            vo.setCountyManagementUnitName(orgCodeNameMap.get(parentOrgCode));
        } else if (c.getReceiveCompanyCode().split("\\*").length > 2) {
            // 直接上级（区县）
            String parentOrgCode = c.getReceiveCompanyCode().substring(0, c.getReceiveCompanyCode().lastIndexOf("*"));
            vo.setCountyManagementUnitName(orgCodeNameMap.get(parentOrgCode));
            // 上上级（地市）
            String parent2OrgCode = parentOrgCode.substring(0, parentOrgCode.lastIndexOf("*"));
            vo.setCityManagementUnitName(orgCodeNameMap.get(parent2OrgCode));
        }
    }

    private JgBizCountDataBO esItemToVo(Terms.Bucket bucket) {
        // 设备状态 + 监管机构 + 设备种类 + 设备类别+ 设备品种
        String keys = bucket.getKeyAsString();
        String[] bizColumnsData = keys.split("\\|");
        JgBizCountDataBO countDataBO = new JgBizCountDataBO();
        countDataBO.setBizType("equipCount");
        countDataBO.setStatus(bizColumnsData[0]);
        countDataBO.setReceiveCompanyCode(bizColumnsData[1]);
        countDataBO.setEquList(bizColumnsData[2]);
        countDataBO.setEquCategory(bizColumnsData[3]);
        countDataBO.setEquDefine(bizColumnsData[4]);
        countDataBO.setNum(bucket.getDocCount());
        return countDataBO;
    }

    private void buildPermissionLimit(ReportAnalysisSearchDTO reportAnalysisSearchDT, BoolQueryBuilder boolMust) {
        //右模糊查询，上级看本级级下级数据
        boolMust.must(QueryBuilders.matchPhrasePrefixQuery("ORG_BRANCH_CODE", reportAnalysisSearchDT.getReceiveOrgCode()));
    }

    /**
     * 过滤设备
     *
     * @param boolMust 必输条件
     */
    private void jgEquipFilterQueryBuild(BoolQueryBuilder boolMust) {
        // 监管类型数据
        boolMust.must(QueryBuilders.termQuery("DATA_SOURCE", "jg"));
        // 设备状态存在
        boolMust.must(QueryBuilders.existsQuery("EQU_STATE"));
        // 被监管状态： 1在用 2停用 3报废 4注销
        boolMust.must(QueryBuilders.rangeQuery("EQU_STATE").gte(1));
    }
}