package com.yeejoin.amos.boot.biz.common.aop;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.yeejoin.amos.boot.biz.common.annotation.FieldMapping;
import com.yeejoin.amos.boot.biz.common.annotation.ResultFieldMapping;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

@Aspect
@Component
@Slf4j
public class FieldMappingAspect {

    @Value("${amos.fieldMapping.cache.maxSize:1000}")
    private static int cacheMaxSize;

    @Value("${amos.fieldMapping.cache.expireHours:1}")
    private static int cacheExpireMinutes;

    // 使用Guava Cache构建二级缓存
    private static final LoadingCache<Class<?>, Map<String, Field>> FIELD_CACHE = CacheBuilder.newBuilder()
            .maximumSize(cacheMaxSize) // 控制最大缓存类数量
            .expireAfterAccess(cacheExpireMinutes, TimeUnit.HOURS) // 访问1小时后过期
            .weakKeys() // 使用弱引用键防止内存泄漏
            .build(new CacheLoader<Class<?>, Map<String, Field>>() {
                // 当缓存未命中时，该方法会为指定类创建一个线程安全的空字段映射容器（ConcurrentHashMap），用于后续存储该类的反射字段信息
                @Override
                public Map<String, Field> load(Class<?> clazz) {
                    return new ConcurrentHashMap<>();
                }
            });

    // 方法缓存优化
    private static final LoadingCache<MethodKey, Method> METHOD_CACHE = CacheBuilder.newBuilder()
            .maximumSize(1000)
            .expireAfterAccess(1, TimeUnit.HOURS) // 访问1小时后过期
            .weakKeys() // 使用弱引用键防止内存泄漏
            .build(new CacheLoader<MethodKey, Method>() {
                public Method load(MethodKey key) throws Exception {
                    return key.clazz.getMethod(key.methodName, key.paramTypes);
                }
            });

    private final ApplicationContext applicationContext;

    public FieldMappingAspect(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    @Around("@annotation(fieldMapping)")
    public Object doFieldMapping(ProceedingJoinPoint joinPoint, FieldMapping fieldMapping) throws Throwable {
        Object[] args = joinPoint.getArgs();

        // 创建防御性副本
        Object[] argsCopy = Arrays.copyOf(args, args.length);
        // 遍历处理所有字段映射配置
        for (FieldMapping.FieldMap mapping : fieldMapping.value()) {
            processSingleMapping(argsCopy, mapping);
        }

        return joinPoint.proceed(argsCopy);
    }

    private void processSingleMapping(Object[] args, FieldMapping.FieldMap mapping) {
        // 使用缓存获取方法
        MethodKey methodKey;
        if (!StringUtils.isEmpty(mapping.secondParamValue())) {
            // 如果有第二个参数值，则创建包含两个参数类型的MethodKey
            methodKey = new MethodKey(
                    mapping.serviceClass(),
                    mapping.queryMethod(),
                    mapping.queryParamType(),
                    mapping.secondParamType()
            );
        } else {
            // 原有逻辑：只包含一个参数类型
            methodKey = new MethodKey(
                    mapping.serviceClass(),
                    mapping.queryMethod(),
                    mapping.queryParamType()
            );
        }
        try {
            if (args.length <= mapping.paramIndex()) return;

            Object paramObj = args[mapping.paramIndex()];
            Object sourceValue;

            // 获取源值逻辑
            if (mapping.useEntireParam()) {
                sourceValue = paramObj;
            } else {
                sourceValue = getSourceValue(paramObj, mapping.sourceField());
            }
            if (sourceValue == null) {
                if (mapping.replaceParam()) {
                    args[mapping.paramIndex()] = null; // 允许清空参数
                }
                return;
            }

            // 服务调用逻辑
            Object service = applicationContext.getBean(mapping.serviceClass());
            Method queryMethod = METHOD_CACHE.get(methodKey);
            Object convertedValue = convertType(sourceValue, mapping.queryParamType());
            Object result;
            if (!StringUtils.isEmpty(mapping.secondParamValue()) && queryMethod.getParameterCount() == 2) {
                // 如果配置了第二个参数值且方法接受两个参数，则调用两个参数的方法
                Object secondParamValue = convertType(mapping.secondParamValue(), mapping.secondParamType());
                result = queryMethod.invoke(service, convertedValue, secondParamValue);
            } else {
                // 原有逻辑：单个参数
                result = queryMethod.invoke(service, convertedValue);
            }

            // 设置目标字段
            if (mapping.replaceParam()) {
                Class<?> paramType = args[mapping.paramIndex()].getClass();
                if (!paramType.isAssignableFrom(result.getClass())) {
                    throw new IllegalArgumentException("类型不匹配: "
                            + result.getClass() + " cannot be cast to " + paramType);
                }
                args[mapping.paramIndex()] = result; // 直接替换参数
            } else {
                setTargetValue(
                        paramObj,
                        mapping.overrideSource() ? mapping.sourceField() : mapping.targetField(),
                        result
                );
            }
        } catch (Exception e) {
            log.error("Field mapping failed for {}: {}", mapping.sourceField(), e.getMessage());
            Throwable cause = e.getCause();
            if (cause instanceof NoSuchMethodException) {
                log.error("Method not found: {}.{}({})",
                        mapping.serviceClass().getSimpleName(),
                        mapping.queryMethod(),
                        mapping.queryParamType().getSimpleName());
            }
        }
    }

    private Object getSourceValue(Object paramObj, String key) {
        if (key.contains(".")) { // 新增嵌套判断
            return getNestedFieldValue(paramObj, key);
        }
        if (paramObj instanceof Map) {
            return ((Map<?, ?>) paramObj).get(key);
        }
        return getFieldValue(paramObj, key);
    }

    private Object getFieldValue(Object obj, String fieldName) {
        try {
            Field field = getCachedField(obj.getClass(), fieldName);
            return field != null ? field.get(obj) : null;
        } catch (IllegalAccessException e) {
            log.debug("Field access error: {}", e.getMessage());
            return null;
        }
    }

    private Field getCachedField(Class<?> clazz, String fieldName) {
        try {
            return FIELD_CACHE.get(clazz)
                    .computeIfAbsent(fieldName, name -> {
                        try {
                            Field field = clazz.getDeclaredField(name);
                            field.setAccessible(true);
                            return field;
                        } catch (NoSuchFieldException e) {
                            log.debug("Field not found: {} in {}", name, clazz.getName());
                            return null;
                        }
                    });
        } catch (ExecutionException e) {
            log.error("Cache operation failed", e);
            return null;
        }
    }

    private Object convertType(Object value, Class<?> targetType) {
        if (value == null) return null;
        if (targetType.isAssignableFrom(value.getClass())) {
            return value;
        }

        // 增加枚举类型支持
        if (targetType.isEnum()) {
            return Enum.valueOf((Class<Enum>) targetType, value.toString());
        }
        if (targetType == String.class) return value.toString();
        if (targetType == int.class || targetType == Integer.class) {
            return Integer.parseInt(value.toString());
        }
        if (targetType == long.class || targetType == Long.class) {
            return Long.parseLong(value.toString());
        }
        log.warn("Unsupported type conversion: {} -> {}",
                value.getClass(), targetType);
        return value;
    }

    private void setTargetValue(Object paramObj, String key, Object value) {
        if (key.contains(".")) { // 新增嵌套判断
            setNestedFieldValue(paramObj, key, value);
            return;
        }
        if (paramObj instanceof Map) {
            ((Map<String, Object>) paramObj).put(key, value);
        } else {
            Optional.ofNullable(getCachedField(paramObj.getClass(), key))
                    .ifPresent(field -> {
                        try {
                            field.set(paramObj, convertType(value, field.getType()));
                        } catch (IllegalAccessException e) {
                            log.debug("Field set error: {}", e.getMessage());
                        }
                    });
        }
    }

    private Object getNestedFieldValue(Object obj, String fieldPath) {
        String[] parts = fieldPath.split("\\.");
        Object current = obj;

        for (String part : parts) {
            if (current == null) return null;

            if (current instanceof Map) {
                current = ((Map<?, ?>) current).get(part);
            } else {
                Field field = getCachedField(current.getClass(), part);
                if (field == null) return null;
                try {
                    current = field.get(current);
                } catch (IllegalAccessException e) {
                    log.debug("Nested field access error: {}", e.getMessage());
                    return null;
                }
            }
        }
        return current;
    }

    private void setNestedFieldValue(Object obj, String fieldPath, Object value) {
        String[] parts = fieldPath.split("\\.");
        Object current = obj;

        try {
            for (int i = 0; i < parts.length - 1; i++) {
                String part = parts[i];

                if (current instanceof Map) {
                    current = ((Map<String, Object>) current).computeIfAbsent(part, k -> new ConcurrentHashMap<>());
                } else {
                    Field field = getCachedField(current.getClass(), part);
                    if (field == null) return;

                    Object next = field.get(current);
                    if (next == null) {
                        next = field.getType().newInstance();
                        field.set(current, next);
                    }
                    current = next;
                }
            }

            String lastPart = parts[parts.length - 1];
            if (current instanceof Map) {
                ((Map<String, Object>) current).put(lastPart, value);
            } else {
                Field field = getCachedField(current.getClass(), lastPart);
                if (field != null) {
                    Object converted = convertType(value, field.getType());
                    field.set(current, converted);
                }
            }
        } catch (Exception e) {
            log.error("Nested field set error", e);
        }
    }

    @Around("@annotation(resultFieldMapping)")
    public Object processResultMapping(ProceedingJoinPoint joinPoint, ResultFieldMapping resultFieldMapping) throws Throwable {
        Object result = joinPoint.proceed();
        return processResultObject(result, resultFieldMapping.value());
    }

    private Object processResultObject(Object result, ResultFieldMapping.ResultFieldMap[] resultFieldMaps) {
        if (result == null || resultFieldMaps.length == 0) return result;

        try {
            // 处理集合类型
            if (result instanceof Collection) {
                ((Collection<?>) result).forEach(item -> processItem(item, resultFieldMaps));
            }
            // 处理数组类型
            else if (result.getClass().isArray()) {
                Arrays.stream((Object[]) result).forEach(item -> processItem(item, resultFieldMaps));
            }
            // 处理单个对象
            else {
                processItem(result, resultFieldMaps);
            }
        } catch (Exception e) {
            log.error("Result mapping processing failed", e);
        }
        return result;
    }

    private void processItem(Object item, ResultFieldMapping.ResultFieldMap[] resultFieldMaps) {
        for (ResultFieldMapping.ResultFieldMap mapping : resultFieldMaps) {
            try {
                // 使用缓存获取方法
                MethodKey methodKey;
                if (!StringUtils.isEmpty(mapping.secondParamValue())) {
                    // 如果有第二个参数值，则创建包含两个参数类型的MethodKey
                    methodKey = new MethodKey(
                            mapping.serviceClass(),
                            mapping.queryMethod(),
                            mapping.queryParamType(),
                            mapping.secondQueryParamType()
                    );
                } else {
                    // 原有逻辑：只包含一个参数类型
                    methodKey = new MethodKey(
                            mapping.serviceClass(),
                            mapping.queryMethod(),
                            mapping.queryParamType()
                    );
                }

                // 获取源字段值
                Object sourceFieldValue = getNestedFieldValue(item, mapping.sourceField());
                if (sourceFieldValue == null) continue;

                // 转换类型
                Object convertedValue = convertType(sourceFieldValue, mapping.queryParamType());

                // 调用服务查询
                Object service = applicationContext.getBean(mapping.serviceClass());
                Method method = METHOD_CACHE.get(methodKey);
                Object targetFieldValue;
                if (method.getParameterCount() == 1) {
                    // 原有逻辑：单个参数
                    targetFieldValue = method.invoke(service, convertedValue);
                } else if (method.getParameterCount() == 2) {
                    // 新增逻辑：两个参数
                    // 获取第二个参数值
                    Object secondParamValue = getSecondParamValue(item, mapping);
                    Object convertedSecondValue = convertType(secondParamValue, mapping.secondQueryParamType());
                    targetFieldValue = method.invoke(service, convertedValue, convertedSecondValue);
                } else {
                    // 其他情况抛出异常或处理
                    throw new UnsupportedOperationException("Unsupported number of parameters: " + method.getParameterCount());
                }
                // 设置名称值
                setNestedFieldValue(item, mapping.targetField(), targetFieldValue);
            } catch (Exception e) {
                log.warn("Field mapping failed for {}: {}", mapping.sourceField(), e.getMessage());
            }
        }
    }

    /**
     * 获取第二个参数值
     * @param item 当前处理的对象
     * @param mapping 字段映射配置
     * @return 第二个参数值
     */
    private Object getSecondParamValue(Object item, ResultFieldMapping.ResultFieldMap mapping) throws Exception {
        // 如果配置了固定的第二个参数值，则直接返回该值
        if (!StringUtils.isEmpty(mapping.secondParamValue())) {
            return mapping.secondParamValue();
        }
        // 如果配置了第二个源字段，则从对象中获取该字段的值
        if (!StringUtils.isEmpty(mapping.secondSourceField())) {
            return getNestedFieldValue(item, mapping.secondSourceField());
        }
        // 默认返回null
        return null;
    }

    // 缓存方法内部类
    private static class MethodKey {
        final Class<?> clazz;
        final String methodName;
        final Class<?>[] paramTypes;

        MethodKey(Class<?> clazz, String methodName, Class<?>... paramTypes) {
            this.clazz = clazz;
            this.methodName = methodName;
            this.paramTypes = paramTypes;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            MethodKey methodKey = (MethodKey) o;
            return Objects.equals(clazz, methodKey.clazz) &&
                    Objects.equals(methodName, methodKey.methodName) &&
                    Arrays.equals(paramTypes, methodKey.paramTypes);
        }

        @Override
        public int hashCode() {
            int result = Objects.hash(clazz, methodName);
            result = 31 * result + Arrays.hashCode(paramTypes);
            return result;
        }
    }

}
