package com.yeejoin.amos.boot.module.common.api.converter;

import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.property.ExcelContentProperty;
import com.yeejoin.amos.boot.biz.common.entity.DataDictionary;
import com.yeejoin.amos.boot.biz.common.service.impl.DataDictionaryServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 人员信息自定义字段转换器
 * 用于处理Excel导入导出时的字段转换逻辑
 */
@Component
public class PersonFieldCustomConverter {

    private static DataDictionaryServiceImpl dataDictionaryService;

    @Autowired
    public void setDataDictionaryService(DataDictionaryServiceImpl dataDictionaryService) {
        PersonFieldCustomConverter.dataDictionaryService = dataDictionaryService;
    }

    /**
     * 【证件类型】转换器
     */
    public static class CertificateTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("ZJLX");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【性别】类型转换器
     */
    public static class GenderConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            CODE_TO_NAME_MAP.put("1", "男");
            CODE_TO_NAME_MAP.put("0", "女");
            NAME_TO_CODE_MAP.put("男", "1");
            NAME_TO_CODE_MAP.put("女", "0");
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, "未知");
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, "未知");
            return new CellData(displayValue);
        }
    }

    /**
     * 【学历】类型转换器
     */
    public static class EducationConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("QYRYXL");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }
}