package com.yeejoin.amos.boot.module.common.api.converter;

import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.metadata.CellData;
import com.alibaba.excel.metadata.GlobalConfiguration;
import com.alibaba.excel.metadata.property.ExcelContentProperty;
import com.yeejoin.amos.boot.biz.common.entity.DataDictionary;
import com.yeejoin.amos.boot.biz.common.service.impl.DataDictionaryServiceImpl;
import com.yeejoin.amos.feign.systemctl.Systemctl;
import com.yeejoin.amos.feign.systemctl.model.DictionarieValueModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 人员信息自定义字段转换器
 * 用于处理Excel导入导出时的字段转换逻辑
 */
@Component
public class EquipFieldCustomConverter {

    private static DataDictionaryServiceImpl dataDictionaryService;

    @Autowired
    public void setDataDictionaryService(DataDictionaryServiceImpl dataDictionaryService) {
        EquipFieldCustomConverter.dataDictionaryService = dataDictionaryService;
    }

    /**
     * 【设备状态】类型转换器
     */
    public static class EquStateConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("SHZT");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【检验类型】类型转换器
     */
    public static class InspectTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("JYJC");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【检验结论】类型转换器
     */
    public static class InspectConclusionConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("JYJL");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【控制方式】类型转换器
     */
    public static class ControlModeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("KZFS");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【区域防爆等级】类型转换器
     */
    public static class ExplosionProofGradeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("FBDJ");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【顶升方式】类型转换器
     */
    public static class JackingTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("DSXS");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【工作级别】类型转换器
     */
    public static class WorkLevelConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("GZJB");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【受压部件名称】类型转换器
     */
    public static class NameOfPressurePartsConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("GLBJMC");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【燃料（热源）种类】类型转换器
     */
    public static class FuelTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("GLZL");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【管道级别】类型转换器
     */
    public static class DeviceLevelConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("GLJB");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【充装介质】类型转换器
     */
    public static class ChargingMediumConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DictionarieValueModel> dictionaries = Systemctl.dictionarieClient.dictValues("FILLING_MEDIUM").getResult();
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getDictDataKey(), dictionary.getDictDataValue());
                NAME_TO_CODE_MAP.put(dictionary.getDictDataValue(), dictionary.getDictDataKey());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【无损检测方法】类型转换器
     */
    public static class GlLosslessConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("RQJCFF");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【主体结构型式】类型转换器
     */
    public static class MainStructureTypeConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("RQJG");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }

    /**
     * 【技术参数】【运载索】类型转换器
     */
    public static class CarrierLineConverter implements Converter<String> {

        private static final Map<String, String> CODE_TO_NAME_MAP = new ConcurrentHashMap<>();
        private static final Map<String, String> NAME_TO_CODE_MAP = new ConcurrentHashMap<>();

        // 使用静态初始化块加载数据字典
        static {
            refreshDictionaryCache();
        }

        /**
         * 刷新数据字典缓存
         */
        public static void refreshDictionaryCache() {
            List<DataDictionary> dictionaries = dataDictionaryService.getByType("YZS");
            CODE_TO_NAME_MAP.clear();
            NAME_TO_CODE_MAP.clear();

            dictionaries.forEach(dictionary -> {
                CODE_TO_NAME_MAP.put(dictionary.getCode(), dictionary.getName());
                NAME_TO_CODE_MAP.put(dictionary.getName(), dictionary.getCode());
            });
        }

        @Override
        public Class<?> supportJavaTypeKey() {
            return String.class;
        }

        @Override
        public CellDataTypeEnum supportExcelTypeKey() {
            return CellDataTypeEnum.STRING;
        }

        @Override
        public String convertToJavaData(CellData cellData, ExcelContentProperty contentProperty,
                                        GlobalConfiguration globalConfiguration) {
            String cellValue = cellData.getStringValue();
            return NAME_TO_CODE_MAP.getOrDefault(cellValue, cellValue);
        }

        @Override
        public CellData convertToExcelData(String value, ExcelContentProperty contentProperty,
                                           GlobalConfiguration globalConfiguration) {
            String displayValue = CODE_TO_NAME_MAP.getOrDefault(value, value);
            return new CellData(displayValue);
        }
    }
}