package com.yeejoin.equipmanage.service.impl;

import com.yeejoin.amos.boot.biz.common.entity.DataDictionary;
import com.yeejoin.equipmanage.common.entity.dto.FireResourceStatsDTO;
import com.yeejoin.equipmanage.common.enums.EmergencyEquipTypeEnum;
import com.yeejoin.equipmanage.common.utils.StringUtil;
import com.yeejoin.equipmanage.fegin.JcsFeign;
import com.yeejoin.equipmanage.mapper.FireFightingSystemMapper;
import com.yeejoin.equipmanage.service.IFireResourceSupervisionService;
import liquibase.pro.packaged.X;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.*;

@Slf4j
@Service
public class IFireResourceSupervisionServiceImpl implements IFireResourceSupervisionService {
    @Autowired
    private FireFightingSystemMapper fireFightingSystemMapper;

    @Autowired
    JcsFeign jcsFeignClient;

    /**
     * 消防系统信息
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireSystemStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectFireFightingSystemStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 监测部件
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getMonitorComponentStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectEquipmentSpecificStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 消防车辆
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireCarStats(String bizOrgCode) {
        Map<String, Object> resultMap = fireFightingSystemMapper.selectCarStats(bizOrgCode);
        return buildFireResourceStatsDTO(resultMap);
    }

    /**
     * 消防器材 按照需求统计消火栓、灭火器、消防铲、正压式空气呼吸器、消防斧、消防桶
     *
     * @param bizOrgCode 机构编码
     */
    @Override
    public FireResourceStatsDTO getFireEquipStats(String bizOrgCode) {
        List<String> list = EmergencyEquipTypeEnum.getCodeList();
        Map<String, Object> resultMap = fireFightingSystemMapper.getEquipStats(bizOrgCode, list);
        return buildFireResourceStatsDTO(resultMap);
    }

    @Override
    public List<LinkedHashMap<String, Object>> getFireEquipStatistic(String type, String bizOrgCode) {
        List<DataDictionary> dictionaryList = jcsFeignClient.dataDictionaryIdFillMenu(StringUtil.isNotEmpty(type) ? type : "ZYGL_XFQC").getResult();
        List<String> list = new ArrayList<>();
        Map<Integer, String> sortMap = new TreeMap<>(new Comparator<Integer>(){
            @Override
            public int compare(Integer o1, Integer o2) {
                return o1-o2; //按照key顺序排列，o2-o1是逆序, o1-o2是正序
            }
        });
        if (!CollectionUtils.isEmpty(dictionaryList)) {
            dictionaryList.forEach(x -> {
                list.add(x.getTreeCode());
                sortMap.put(x.getSortNum(), x.getTreeCode());
            });
        }
        List<LinkedHashMap<String, Object>> resultMap = fireFightingSystemMapper.getFireEquipStatistic(list, bizOrgCode);
        List<LinkedHashMap<String, Object>> resp = new ArrayList<>();
        if (!sortMap.isEmpty()) {
            sortMap.entrySet().forEach(x -> {
                resultMap.forEach(y -> {
                    if (x.getValue().equalsIgnoreCase(String.valueOf(y.get("code")))) {
                        resp.add(y);
                    }
                });
            });
        }
        List<LinkedHashMap<String, Object>> otherMapData = fireFightingSystemMapper.getOtherFireEquipStatistic(list, bizOrgCode);
        resp.addAll(otherMapData);
        return resp;
    }

    private FireResourceStatsDTO buildFireResourceStatsDTO(Map<String, Object> resultMap) {
        FireResourceStatsDTO fireResourceStats = new FireResourceStatsDTO();
        fireResourceStats.setTotalCounts(Long.parseLong(resultMap.get("totalCount").toString()));
        fireResourceStats.setYellowCounts(Long.parseLong(resultMap.get("yellowCodeCount").toString()) );
        fireResourceStats.setRedCounts(Long.parseLong(resultMap.get("redCodeCount").toString()));

        long expCount = fireResourceStats.getYellowCounts() + fireResourceStats.getRedCounts();
        double abnormalRatio = 0;
        if (fireResourceStats.getTotalCounts() != 0 && expCount != 0) {
            abnormalRatio = (double) expCount / fireResourceStats.getTotalCounts() * 100;
        }

        if (abnormalRatio == 0) {
            fireResourceStats.setAbnormalRatio(0);
        } else {
            DecimalFormat decimalFormat = new DecimalFormat("#.00");
            decimalFormat.setRoundingMode(RoundingMode.HALF_UP);
            String formattedRatio = decimalFormat.format(abnormalRatio);
            fireResourceStats.setAbnormalRatio(Double.parseDouble(formattedRatio));
        }

        return fireResourceStats;
    }

}
