package com.yeejoin.equipmanage.common.utils;

import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import org.springframework.util.Assert;
/**
 * Optional工具--规避空指针异常 
 * 
 *
 */
public class ParsePropertyUtil {
	
	private ParsePropertyUtil(){};
	
	/**
	 * 解析一个对象
	 * @param supplier 
	 * @return 一个Optional对象，否则返回empty()
	 */
	public static <T> Optional<T> resolve(Supplier<T> supplier){
		try{
			T result = supplier.get();
			return Optional.ofNullable(result);
		}catch(NullPointerException e){
			return Optional.empty();
		}
	}
	
	/**
	 * 设置对象属性
	 * @param con
	 * @param supplier
	 */
	public static <T> void setting(Consumer<T> con,Supplier<T> supplier){
		setting(con, supplier, null);
	}
	
	/**
	 * 设置对象属性，提供默认值
	 * @param con
	 * @param supplier
	 * @param defaultValue
	 */
	public static <T> void setting(Consumer<T> con,Supplier<T> supplier,T defaultValue){
		con.accept(resolve(supplier).orElse(defaultValue));
	}
	
	/**
	 * 设置map对象属性，提供默认值
	 * 
	 * @param map
	 * @param key
	 * @param supplier
	 * @param defaultValue
	 */
	public static <T, K> void map(Map<K, T> map, K key, Supplier<T> supplier,T defaultValue){
		Assert.notNull(map, "map对象不能为空！");
		Assert.notNull(key, "map健值不能为空！");
		
		map.put(key, resolve(supplier).orElse(defaultValue));
	}
	
	/**
	 * 设置map对象属性，无默认值
	 * 
	 * @param map
	 * @param key
	 * @param supplier
	 */
	public static <T, K> void map(Map<K, T> map, K key, Supplier<T> supplier){
		map(map, key, supplier, null);
	}
	
	/**
	 * 设置对象属性，提供默认值,当defaultValu出现空指针时抛出
	 * @param con
	 * @param supplier
	 * @param defaultValue 
	 * @throws 空指针异常 当defaultValue出现空指针时抛出
	 */
	public static <T> void settingDefault(Consumer<T> con,Supplier<T> supplier,Supplier<T> defaultValue){
		con.accept(resolve(supplier).orElseGet(defaultValue));
	}


}
