package com.yeejoin.amos.boot.module.hygf.biz.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.yeejoin.amos.boot.biz.common.utils.DateUtils;
import com.yeejoin.amos.boot.biz.common.utils.ExcelUtils;
import com.yeejoin.amos.boot.module.hygf.api.Enum.*;
import com.yeejoin.amos.boot.module.hygf.api.dto.RepaymentDto;
import com.yeejoin.amos.boot.module.hygf.api.dto.WorkflowResultDto;
import com.yeejoin.amos.boot.module.hygf.api.entity.Repayment;
import com.yeejoin.amos.boot.module.hygf.api.mapper.RepaymentMapper;
import com.yeejoin.amos.boot.module.hygf.api.service.IRepaymentService;
import com.yeejoin.amos.boot.module.hygf.biz.vo.RepaymentExcelVO;
import com.yeejoin.amos.feign.privilege.model.AgencyUserModel;
import com.yeejoin.amos.feign.privilege.model.CompanyModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;
import org.typroject.tyboot.core.rdbms.service.BaseService;
import org.typroject.tyboot.core.restful.exception.instance.BadRequest;

import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.*;

/**
 * 还款实现层
 *
 * @author hzz
 * @date 2024-09-26
 */
@Service
public class RepaymentServiceImpl extends BaseService<RepaymentDto, Repayment, RepaymentMapper> implements IRepaymentService {
    @Autowired
    CommonServiceImpl commonService;
    @Autowired
    RepaymentMapper repaymentMapper;

    public Page<RepaymentDto> queryForRepaymentPage(int current, int size) {
        PageHelper.startPage(current, size);
        List<RepaymentDto> list = repaymentMapper.queryPage();
        PageInfo<RepaymentDto> pages = new PageInfo(list);
        List<RepaymentDto> pagesList = pages.getList();
        if (CollectionUtil.isNotEmpty(pagesList)) {
            //为了前端页面展示
            for (RepaymentDto repaymentDto : pagesList) {
                repaymentDto.setMessageStateStr(RepaymentMessageStateEnum.getByCode(repaymentDto.getMessageState()).getRemark());
                repaymentDto.setRepayStateStr(RepaymentRepayStateEnum.getByCode(repaymentDto.getMessageState()).getRemark());
            }
        }
        Page<RepaymentDto> page = new Page<>();
        page.setCurrent(current);
        page.setTotal(pages.getTotal());
        page.setSize(size);
        page.setRecords(pages.getList());
        return page;
    }

    @Transactional(rollbackFor = Exception.class)
    public void batchConfirmMessageState(List<Long> sequenceNbrList) {
        if (CollectionUtil.isNotEmpty(sequenceNbrList)) {
            LambdaUpdateWrapper<Repayment> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.set(Repayment::getMessageState, RepaymentMessageStateEnum.CONFIRM.getCode())
                    .set(Repayment::getCronSendState, RepaymentCronSendStateEnum.UN_SEND.getCode())
                    .in(Repayment::getSequenceNbr, sequenceNbrList);
            this.update(updateWrapper);
            for (Long sequenceNbr : sequenceNbrList) {
                //更新工作台信息
                updateTaskModel(String.valueOf(sequenceNbr));
            }
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void batchRepay(List<Long> sequenceNbrList) {
        if (CollectionUtil.isNotEmpty(sequenceNbrList)) {
            LambdaUpdateWrapper<Repayment> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.set(Repayment::getRepayState, RepaymentRepayStateEnum.REPAY.getCode())
                    .in(Repayment::getSequenceNbr, sequenceNbrList);
            this.update(updateWrapper);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void confirmMessageState(Long sequenceNbr) {
        LambdaUpdateWrapper<Repayment> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.set(Repayment::getMessageState, RepaymentMessageStateEnum.CONFIRM.getCode())
                .set(Repayment::getCronSendState, RepaymentCronSendStateEnum.UN_SEND.getCode())
                .eq(Repayment::getSequenceNbr, sequenceNbr);
        this.update(updateWrapper);
        //更新工作台信息
        updateTaskModel(String.valueOf(sequenceNbr));
    }

    /**
     * 更新工作台信息
     * @param sequenceNbr
     */
    @Transactional(rollbackFor = Exception.class)
    private void updateTaskModel(String sequenceNbr) {
        Map<String, Object> updateTaskParam = new HashMap<>();
        updateTaskParam.put("flowStatus", FlowStatusEnum.TO_BE_FINISHED.getCode());
        updateTaskParam.put("flowStatusLabel", FlowStatusEnum.TO_BE_FINISHED.getName());
        updateTaskParam.put("taskStatus", FlowStatusEnum.TO_BE_FINISHED.getCode());
        updateTaskParam.put("taskStatusLabel", FlowStatusEnum.TO_BE_FINISHED.getName());
        updateTaskParam.put("model", sequenceNbr);
        updateTaskParam.put("relationId", sequenceNbr);
        commonService.updateTaskModel(updateTaskParam);
    }

    @Transactional(rollbackFor = Exception.class)
    public void repay(Long sequenceNbr) {
        LambdaUpdateWrapper<Repayment> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.set(Repayment::getRepayState, RepaymentRepayStateEnum.REPAY.getCode())
                .eq(Repayment::getSequenceNbr, sequenceNbr);
        this.update(updateWrapper);
    }

    /**
     * 定时发送还款信息 15天发送、5天发送、当天发送
     */
    @Scheduled(cron = "0 0 1 * * ?")
    public void cronSendRepayMessage() {
        List<Repayment> repaymentList = list();
        if (CollectionUtil.isNotEmpty(repaymentList)) {
            for (Repayment repayment : repaymentList) {
                //此处拿出还款日期进行对比15天发送、5天发送、当天发送
                LocalDate repayDate = repayment.getRepayDate();
                LocalDate now = LocalDate.now();
                long betweenDay = ChronoUnit.DAYS.between(now, repayDate);
                Boolean dateSend = betweenDay == 15 || betweenDay == 5 || betweenDay == 0;
                if (dateSend && repayment.getCronSendState() == RepaymentCronSendStateEnum.SEND.getCode()) {
                    //生成待办
                    WorkflowResultDto workflowResultDto = new WorkflowResultDto();
                    workflowResultDto.setInstanceId(String.valueOf(repayment.getSequenceNbr()));
                    commonService.buildTaskModel(commonService.buildTaskModelDto(repayment, workflowResultDto, BusinessTypeEnum.HYGF_REPAY));
                }
                if (repayment.getCronSendState() < 0) {
                    //关闭定时任务
                    LambdaUpdateWrapper<Repayment> updateWrapper = new LambdaUpdateWrapper<>();
                    updateWrapper.set(Repayment::getCronSendState, RepaymentCronSendStateEnum.UN_SEND.getCode())
                            .eq(Repayment::getSequenceNbr, repayment.getSequenceNbr());
                    this.update(updateWrapper);
                }
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void importRepaymentExcel(MultipartFile file, AgencyUserModel userInfo) {
        //获取当前用户权限的公司
        try {
            List<CompanyModel> companys = userInfo.getCompanys();
            if (CollectionUtil.isEmpty(companys)) {
                throw new BadRequest("该用户没有公司");
            }
            ExcelReader reader = ExcelUtil.getReader(file.getInputStream());
            List<List<Object>> read = reader.read(1);
            List<Repayment> repaymentList = new ArrayList<>();
            for (List<Object> objects : read) {
                if (objects.size() != 7) {
                    throw new BadRequest("模板列不是7个");
                }
                Repayment repayment = new Repayment();
                String companyName = (String) objects.get(0);
                for (CompanyModel company : companys) {
                    if (company.getCompanyName().equals(companyName)) {
                        repayment.setCompanyName(companyName);
                        repayment.setCompanyId(company.getSequenceNbr());
                        repayment.setRegionalCompaniesCode(String.valueOf(company.getCompanyOrgCode()));
                    }
                }
                if (StrUtil.isEmpty(repayment.getCompanyName())) {
                    throw new BadRequest(companyName + ",该用户没有权限");
                }

                String period = (String) objects.get(1);
                String repayDate = (String) objects.get(2);
                String rent = (String) objects.get(3);
                String interest = (String) objects.get(4);
                String principal = (String) objects.get(5);
                String loanPeriod = (String) objects.get(6);
                repayment.setMessageState(RepaymentMessageStateEnum.UN_CONFIRM.getCode());
                repayment.setCronSendState(RepaymentCronSendStateEnum.SEND.getCode());
                repayment.setRepayState(RepaymentRepayStateEnum.UN_REPAY.getCode());
                repayment.setPeriod(period);
                repayment.setRepayDate(DateUtils.dateParse(repayDate).toInstant()
                        .atZone(ZoneId.systemDefault())
                        .toLocalDate());
                repayment.setRent(Double.valueOf(rent));
                repayment.setInterest(Double.valueOf(interest));
                repayment.setPrincipal(Double.valueOf(principal));
                repayment.setLoanPeriod(loanPeriod);
                repaymentList.add(repayment);
            }
            try {
                saveBatch(repaymentList);
            } catch (Exception exception) {
                throw new BadRequest("有重复数据请检查");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new BadRequest("导入失败,失败原因[" + e.getMessage() + "]");
        }
    }

    public void exportRepaymentExcel(List<Long> sequenceNbrList, HttpServletResponse response) {
        LambdaQueryWrapper<Repayment> wrapper = new LambdaQueryWrapper<>();
        //如果是空值导出全部
        if (CollectionUtil.isNotEmpty(sequenceNbrList)) {
            wrapper.in(Repayment::getSequenceNbr, sequenceNbrList);
        }
        List<Repayment> repaymentList = list(wrapper);
        if (CollectionUtil.isNotEmpty(repaymentList)) {
            List<RepaymentExcelVO> dataList = new ArrayList<>();
            for (Repayment repayment : repaymentList) {
                RepaymentExcelVO repaymentExcelVO = BeanUtil.copyProperties(repayment, RepaymentExcelVO.class);
                dataList.add(repaymentExcelVO);
            }
            try {
                ExcelUtils.exportExcel(dataList, null, "还款管理", RepaymentExcelVO.class, "还款管理" + DateUtils.dateFormat(new Date(), "YYYY-MM-dd-HH-mm") + ".xlsx", response);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new BadRequest("导出失败,失败原因[" + e.getMessage() + "]");
            }
        }

    }
}
